/*****************************************************************************
 * Project: RooFit                                                           *
 * Package: RooFitCore                                                       *
 * @(#)root/roofitcore:$Id$
 * Authors:                                                                  *
 *   WV, Wouter Verkerke, UC Santa Barbara, verkerke@slac.stanford.edu       *
 *   DK, David Kirkby,    UC Irvine,         dkirkby@uci.edu                 *
 *                                                                           *
 * Copyright (c) 2000-2005, Regents of the University of California          *
 *                          and Stanford University. All rights reserved.    *
 *                                                                           *
 * Redistribution and use in source and binary forms,                        *
 * with or without modification, are permitted according to the terms        *
 * listed in LICENSE (http://roofit.sourceforge.net/license.txt)             *
 *****************************************************************************/

/**
\file RooMsgService.cxx
\class RooMsgService
\ingroup Roofitcore


Singleton class that organizes messages generated in RooFit.
Each message has a message level RooFit::MsgLevel (DEBUG,INFO,PROGRESS,WARNING,ERROR or FATAL),
an source object, and a RooFit::MsgTopic.
RooMsgService allows to filter and redirect messages into streams
according to message level, topic, (base) class of originating object, name of originating
object and based on attribute labels attached to individual objects.
The current default configuration creates streams for all messages at WARNING level
or higher (e.g. ERROR and FATAL) and for all INFO message on topics Generation,Plotting,
Integration and Minimization and redirects them to stdout. Users can create additional streams
for logging of e.g. DEBUG messages on particular topics or objects and/or redirect streams to
C++ streams or files.

The singleton instance is accessible through RooMsgService::instance().

### Temporarily change the message level
There is a helper, RooHelpers::LocalChangeMsgLevel, that overrides the default message levels as
long as it is alive. To suppress everything below WARNING:
~~~{.cpp}
RooHelpers::LocalChangeMessageLevel changeMsgLvl(RooFit::WARNING);
[ statements that normally generate a lot of output ]
~~~

#### Temporarily capture a message stream
RooHelpers::HijackMessageStream allows to fully capture a message stream in a std::stringstream. With this,
RooFit messages can be evaluated or suppressed.
**/


#include "RooMsgService.h"

#include <sys/types.h>
#include "RooAbsArg.h"
#include "RooCmdArg.h"
#include "RooCmdConfig.h"
#include "RooGlobalFunc.h"
#include "RooWorkspace.h"
#include "RooHelpers.h"

#include "TClass.h"
#include "TSystem.h"

#include <fstream>
#include <iomanip>

using std::ofstream, std::ostream, std::string, std::vector, std::map;
using namespace RooFit;


Int_t RooMsgService::_debugCount = 0;


////////////////////////////////////////////////////////////////////////////////
/// Constructor. Defines names of message levels
/// and mapping of topic codes to topic names
/// Install default message streams.

RooMsgService::RooMsgService()
{
  _devnull = std::make_unique<ofstream>("/dev/null") ;

  _levelNames[DEBUG]="DEBUG" ;
  _levelNames[INFO]="INFO" ;
  _levelNames[PROGRESS]="PROGRESS" ;
  _levelNames[WARNING]="WARNING" ;
  _levelNames[ERROR]="ERROR" ;
  _levelNames[FATAL]="FATAL" ;

  _topicNames[Generation]="Generation" ;
  _topicNames[Minimization]="Minimization" ;
  _topicNames[Plotting]="Plotting" ;
  _topicNames[Fitting]="Fitting" ;
  _topicNames[Integration]="Integration" ;
  _topicNames[LinkStateMgmt]="LinkStateMgmt" ;
  _topicNames[Eval]="Eval" ;
  _topicNames[Caching]="Caching" ;
  _topicNames[Optimization]="Optimization" ;
  _topicNames[ObjectHandling]="ObjectHandling" ;
  _topicNames[InputArguments]="InputArguments" ;
  _topicNames[Tracing]="Tracing" ;
  _topicNames[Contents]="Contents" ;
  _topicNames[DataHandling]="DataHandling" ;
  _topicNames[NumericIntegration]="NumericIntegration" ;
  _topicNames[FastEvaluations] = "FastEvaluations";
  _topicNames[HistFactory]="HistFactory";

  reset();
}


void RooMsgService::reset() {
  _silentMode = false ;
  _showPid = false ;
  _globMinLevel = DEBUG ;
  _lastMsgLevel = DEBUG ;

  _debugWorkspace = nullptr;
  _debugCode = 0 ;

  _files.clear();

  // Old-style streams
  _streams.clear();
  addStream(RooFit::PROGRESS, Topic(RooFit::HistFactory - 1));//All before HistFactory
  addStream(RooFit::INFO,Topic(RooFit::Eval|RooFit::Plotting|RooFit::Fitting|RooFit::Minimization|RooFit::Caching|RooFit::ObjectHandling|RooFit::NumericIntegration|RooFit::InputArguments|RooFit::DataHandling)) ;
  addStream(RooFit::INFO, Topic(RooFit::HistFactory));
}


RooMsgService::~RooMsgService() = default;


////////////////////////////////////////////////////////////////////////////////
/// Returns true if any debug level stream is active

bool RooMsgService::anyDebug()
{
  return instance()._debugCount>0 ;
}



////////////////////////////////////////////////////////////////////////////////

RooWorkspace* RooMsgService::debugWorkspace()
{
  if (!_debugWorkspace) {
    _debugWorkspace = std::make_unique<RooWorkspace>("wdebug") ;
  }
  return _debugWorkspace.get();
}



////////////////////////////////////////////////////////////////////////////////
/// Add a message logging stream for message with given RooFit::MsgLevel or higher.
/// Higher means that messages with higher priority/severity are issued.
///
/// This method accepts the following arguments to configure the stream:
/// <table>
/// <tr><th> Output Style options <th>
/// <tr><td> Prefix(bool flag=true) <td> Prefix all messages in this stream with Topic/Originator information
/// <tr><th> Filtering options <th>
/// <tr><td> Topic()                   <td> Restrict stream to messages on given topic
/// <tr><td> ObjectName(const char*)   <td> Restrict stream to messages from object with given name
/// <tr><td> ClassName(const char*)    <td> Restrict stream to messages from objects with given class name
/// <tr><td> BaseClassName(const char*)<td> Restrict stream to messages from objects with given base class name
/// <tr><td> LabelName(const chat*)    <td> Restrict stream to messages from objects setAttribute(const char*) tag with given name
/// <tr><th> Output redirection options <th>
/// <tr><td> OutputFile(const char*)  <td> Send output to file with given name. Multiple streams can write to same file.
/// <tr><td> OutputStream(ostream&)   <td> Send output to given C++ stream. Multiple message streams can write to same c++ stream
/// </table>
/// The return value is the unique ID of the defined stream.

Int_t RooMsgService::addStream(RooFit::MsgLevel level, const RooCmdArg& arg1, const RooCmdArg& arg2, const RooCmdArg& arg3,
                                    const RooCmdArg& arg4, const RooCmdArg& arg5, const RooCmdArg& arg6)
{

  // Aggregate all arguments in a list
  RooLinkedList l ;
  l.Add((TObject*)&arg1) ;  l.Add((TObject*)&arg2) ;
  l.Add((TObject*)&arg3) ;  l.Add((TObject*)&arg4) ;
  l.Add((TObject*)&arg5) ;  l.Add((TObject*)&arg6) ;

  // Define configuration for this method
  RooCmdConfig pc("RooMsgService::addReportingStream(" + std::string(GetName()) + ")") ;
  pc.defineInt("prefix","Prefix",0,true) ;
  pc.defineInt("color","Color",0,static_cast<Int_t>(kBlack)) ;
  pc.defineInt("topic","Topic",0,0xFFFFF) ;
  pc.defineString("objName","ObjectName",0,"") ;
  pc.defineString("className","ClassName",0,"") ;
  pc.defineString("baseClassName","BaseClassName",0,"") ;
  pc.defineString("tagName","LabelName",0,"") ;
  pc.defineString("outFile","OutputFile",0,"") ;
  pc.defineObject("outStream","OutputStream",0,nullptr) ;
  pc.defineMutex("OutputFile","OutputStream") ;

  // Process & check varargs
  pc.process(l) ;
  if (!pc.ok(true)) {
    return -1 ;
  }

  // Extract values from named arguments
  RooFit::MsgTopic topic =  (RooFit::MsgTopic) pc.getInt("topic") ;
  const char* objName =  pc.getString("objName") ;
  const char* className =  pc.getString("className") ;
  const char* baseClassName =  pc.getString("baseClassName") ;
  const char* tagName =  pc.getString("tagName") ;
  const char* outFile = pc.getString("outFile") ;
  bool prefix = pc.getInt("prefix") ;
  Color_t color = static_cast<Color_t>(pc.getInt("color")) ;
  auto wrapper = static_cast<RooHelpers::WrapIntoTObject<ostream>*>(pc.getObject("outStream"));
  ostream* os = nullptr;
  if (wrapper) {
    os = wrapper->_payload;
    delete wrapper;
    wrapper = nullptr;
  }

  // Create new stream object
  StreamConfig newStream ;

  // Store configuration info
  newStream.active = true ;
  newStream.minLevel = level ;
  newStream.topic = topic ;
  newStream.objectName = (objName ? objName : "" ) ;
  newStream.className = (className ? className : "" ) ;
  newStream.baseClassName = (baseClassName ? baseClassName : "" ) ;
  newStream.tagName = (tagName ? tagName : "" ) ;
  newStream.color = color ;
  newStream.prefix = prefix ;
  newStream.universal = (newStream.objectName.empty() && newStream.className.empty() && newStream.baseClassName.empty() && newStream.tagName.empty()) ;

  // Update debug stream count
  if (level==DEBUG) {
    _debugCount++ ;
  }

  // Configure output
  if (os) {

    // To given non-owned stream
    newStream.os = os ;

  } else if (!string(outFile).empty()) {

    // See if we already opened the file
    ostream* os2 = _files["outFile"].get();

    if (!os2) {

      // To given file name, create owned stream for it
      os2 = new ofstream(outFile) ;

      if (!*os2) {
   std::cout << "RooMsgService::addReportingStream ERROR: cannot open output log file " << outFile << " reverting stream to stdout" << std::endl ;
   delete os2 ;
   newStream.os = &std::cout ;
      } else {
   newStream.os = os2 ;
      }

    } else {
      newStream.os = os2 ;
      _files["outFile"] = std::unique_ptr<std::ostream>{os2};
    }


  } else {

    // To stdout
    newStream.os = &std::cout ;

  }


  // Add it to list of active streams ;
  _streams.push_back(newStream) ;

  // Return stream identifier
  return _streams.size()-1 ;
}



////////////////////////////////////////////////////////////////////////////////
/// Delete stream with given unique ID code

void RooMsgService::deleteStream(Int_t id)
{
  vector<StreamConfig>::iterator iter = _streams.begin() ;
  iter += id ;

  // Update debug stream count
  if (iter->minLevel==DEBUG) {
    _debugCount-- ;
  }

  _streams.erase(iter) ;
}



////////////////////////////////////////////////////////////////////////////////
/// (De)Activate stream with given unique ID

void RooMsgService::setStreamStatus(Int_t id, bool flag)
{
  if (id<0 || id>=static_cast<Int_t>(_streams.size())) {
    std::cout << "RooMsgService::setStreamStatus() ERROR: invalid stream ID " << id << std::endl ;
    return ;
  }

  // Update debug stream count
  if (_streams[id].minLevel==DEBUG) {
    _debugCount += flag ? 1 : -1 ;
  }

  _streams[id].active = flag ;
}



////////////////////////////////////////////////////////////////////////////////
/// Get activation status of stream with given unique ID

bool RooMsgService::getStreamStatus(Int_t id) const
{
  if (id<0 || id>= static_cast<Int_t>(_streams.size())) {
    std::cout << "RooMsgService::getStreamStatus() ERROR: invalid stream ID " << id << std::endl ;
    return false ;
  }
  return _streams[id].active ;
}



////////////////////////////////////////////////////////////////////////////////
/// Return reference to singleton instance

RooMsgService& RooMsgService::instance()
{
  static RooMsgService instance;
  return instance;
}



////////////////////////////////////////////////////////////////////////////////
/// Save current state of message service

void RooMsgService::saveState()
{
  _streamsSaved.push(_streams) ;
}



////////////////////////////////////////////////////////////////////////////////
/// Restore last saved state of message service

void RooMsgService::restoreState()
{
  _streams = _streamsSaved.top() ;
  _streamsSaved.pop() ;
}


////////////////////////////////////////////////////////////////////////////////
/// Determine if message from given object at given level on given topic is logged

bool RooMsgService::StreamConfig::match(RooFit::MsgLevel level, RooFit::MsgTopic top, const RooAbsArg* obj)
{
  if (!active) return false ;
  if (level<minLevel) return false ;
  if (!(topic&top)) return false ;

  if (universal) return true ;

  if (!obj) return false;
  if (!objectName.empty() && objectName != obj->GetName()) return false ;
  if (!className.empty() && className != obj->ClassName()) return false ;
  if (!baseClassName.empty() && !obj->IsA()->InheritsFrom(baseClassName.c_str())) return false ;
  if (!tagName.empty() && !obj->getAttribute(tagName.c_str())) return false ;

  return true ;
}


////////////////////////////////////////////////////////////////////////////////
/// Determine if message from given object at given level on given topic is logged

bool RooMsgService::StreamConfig::match(RooFit::MsgLevel level, RooFit::MsgTopic top, const TObject* obj)
{
  if (!active) return false ;
  if (level<minLevel) return false ;
  if (!(topic&top)) return false ;

  if (universal) return true ;

  if (!obj) return false;
  if (!objectName.empty() && objectName != obj->GetName()) return false ;
  if (!className.empty() && className != obj->ClassName()) return false ;
  if (!baseClassName.empty() && !obj->IsA()->InheritsFrom(baseClassName.c_str())) return false ;

  return true ;
}



////////////////////////////////////////////////////////////////////////////////
/// Log error message associated with RooAbsArg object self at given level and topic. If skipPrefix
/// is true the standard RooMsgService prefix is not added.

ostream& RooMsgService::log(const RooAbsArg* self, RooFit::MsgLevel level, RooFit::MsgTopic topic, bool skipPrefix)
{
  if (level>=ERROR) {
    _errorCount++ ;
  }

  // Return C++ ostream associated with given message configuration
  Int_t as = activeStream(self,topic,level) ;

  if (as==-1) {
    return *_devnull ;
  }

  // Flush any previous messages
  (*_streams[as].os).flush() ;

  // Insert an std::endl if we switch from progress to another level
  if (_lastMsgLevel==PROGRESS && level!=PROGRESS) {
    (*_streams[as].os) << std::endl ;
  }
  _lastMsgLevel=level ;

  if (_streams[as].prefix && !skipPrefix) {
    if (_showPid) {
      (*_streams[as].os) << "pid" << gSystem->GetPid() << " " ;
    }
    (*_streams[as].os) << "[#" << as << "] " << _levelNames[level] << ":" << _topicNames[topic]  << " -- " ;
  }
  return (*_streams[as].os) ;
}



////////////////////////////////////////////////////////////////////////////////
/// Log error message associated with TObject object self at given level and topic. If skipPrefix
/// is true the standard RooMsgService prefix is not added.

ostream& RooMsgService::log(const TObject* self, RooFit::MsgLevel level, RooFit::MsgTopic topic, bool skipPrefix)
{
  if (level>=ERROR) {
    _errorCount++ ;
  }

  // Return C++ ostream associated with given message configuration
  Int_t as = activeStream(self,topic,level) ;
  if (as==-1) {
    return *_devnull ;
  }

  // Flush any previous messages
  (*_streams[as].os).flush() ;

  if (_streams[as].prefix && !skipPrefix) {
    if (_showPid) {
      (*_streams[as].os) << "pid" << gSystem->GetPid() << " " ;
    }
    (*_streams[as].os) << "[#" << as << "] " << _levelNames[level] << ":" << _topicNames[topic]  << " -- " ;
  }
  return (*_streams[as].os) ;
}



////////////////////////////////////////////////////////////////////////////////
/// Print configuration of message service. If "v" option is given also
/// inactive streams are listed

void RooMsgService::Print(Option_t *options) const
{
  bool activeOnly = true ;
  if (TString(options).Contains("V") || TString(options).Contains("v")) {
    activeOnly = false ;
  }

  std::cout << (activeOnly?"Active Message streams":"All Message streams") << std::endl ;
  for (UInt_t i=0 ; i<_streams.size() ; i++) {

    // Skip passive streams in active only mode
    if (activeOnly && !_streams[i].active) {
      continue ;
    }


    map<int,string>::const_iterator is = _levelNames.find(_streams[i].minLevel) ;
    std::cout << "[" << i << "] MinLevel = " << is->second ;

    std::cout << " Topic = " ;
    if (_streams[i].topic != 0xFFFFF) {
      map<int,string>::const_iterator iter = _topicNames.begin() ;
      while(iter!=_topicNames.end()) {
   if (iter->first & _streams[i].topic) {
     std::cout << iter->second << " " ;
   }
   ++iter ;
      }
    } else {
      std::cout << " Any " ;
    }


    if (!_streams[i].objectName.empty()) {
      std::cout << " ObjectName = " << _streams[i].objectName ;
    }
    if (!_streams[i].className.empty()) {
      std::cout << " ClassName = " << _streams[i].className ;
    }
    if (!_streams[i].baseClassName.empty()) {
      std::cout << " BaseClassName = " << _streams[i].baseClassName ;
    }
    if (!_streams[i].tagName.empty()) {
      std::cout << " TagLabel = " << _streams[i].tagName ;
    }

    // Postfix status when printing all
    if (!activeOnly && !_streams[i].active) {
      std::cout << " (NOT ACTIVE)"  ;
    }

    std::cout << std::endl ;
  }

}
