// Copyright (C) 2016 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only WITH Qt-GPL-exception-1.0

#ifndef PARSER_H
#define PARSER_H

#include "symbols.h"
#include <QtCore/qbytearrayview.h>

#include <stack>

QT_BEGIN_NAMESPACE

class Parser
{
public:
    Symbols symbols;
    qsizetype index = 0;
    bool displayWarnings = true;
    bool displayNotes = true;
    bool activeQtMode = false;
    bool qmlMacroWarningIsFatal = false;

    struct IncludePath
    {
        inline explicit IncludePath(const QByteArray &_path)
            : path(_path), isFrameworkPath(false) {}
        QByteArray path;
        bool isFrameworkPath;
    };
    QList<IncludePath> includes;

    std::stack<QByteArray, QByteArrayList> currentFilenames;

    inline bool hasNext() const { return (index < symbols.size()); }
    inline Token next() { if (index >= symbols.size()) return NOTOKEN; return symbols.at(index++).token; }
    inline Token peek() { if (index >= symbols.size()) return NOTOKEN; return symbols.at(index).token; }
    bool test(Token);
    void next(Token);
    void next(Token, const char *msg);
    inline void prev() {--index;}
    inline Token lookup(int k = 1);
    inline const Symbol &symbol_lookup(int k = 1) { return symbols.at(index-1+k);}
    inline Token token() { return symbols.at(index-1).token;}
    inline QByteArray lexem() { return symbols.at(index-1).lexem();}
    inline QByteArray unquotedLexem() { return symbols.at(index-1).unquotedLexem();}
    inline QByteArrayView lexemView() { return symbols.at(index-1).lexemView();}
    inline QByteArrayView unquotedLexemView() { return symbols.at(index-1).unquotedLexemView();}
    inline const Symbol &symbol() { return symbols.at(index-1);}
    inline const Symbol &symbolAt(qsizetype idx) { return symbols.at(idx); }

    Q_NORETURN void error(const Symbol &symbol);
    Q_NORETURN void error(const char *msg = nullptr);
    Q_NORETURN void error(const Symbol &symbol, const char *msg);
    void warning(const char * = nullptr);
    void warning(const Symbol &sym, QByteArrayView msg);
    void note(const char * = nullptr);
    void defaultErrorMsg(const Symbol &sym);
    void printMsg(QByteArrayView formatStringSuffix, QByteArrayView msg, const Symbol &sym);

};

inline bool Parser::test(Token token)
{
    if (index < symbols.size() && symbols.at(index).token == token) {
        ++index;
        return true;
    }
    return false;
}

inline Token Parser::lookup(int k)
{
    const qsizetype l = index - 1 + k;
    return l < symbols.size() ? symbols.at(l).token : NOTOKEN;
}

inline void Parser::next(Token token)
{
    if (!test(token))
        error();
}

inline void Parser::next(Token token, const char *msg)
{
    if (!test(token))
        error(msg);
}

QT_END_NAMESPACE

#endif
