{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- An event controller to receive Drag-and-Drop operations.
-- 
-- The most basic way to use a @GtkDropTarget@ to receive drops on a
-- widget is to create it via 'GI.Gtk.Objects.DropTarget.dropTargetNew', passing in the
-- @GType@ of the data you want to receive and connect to the
-- [DropTarget::drop]("GI.Gtk.Objects.DropTarget#g:signal:drop") signal to receive the data:
-- 
-- 
-- === /c code/
-- >static gboolean
-- >on_drop (GtkDropTarget *target,
-- >         const GValue  *value,
-- >         double         x,
-- >         double         y,
-- >         gpointer       data)
-- >{
-- >  MyWidget *self = data;
-- >
-- >  // Call the appropriate setter depending on the type of data
-- >  // that we received
-- >  if (G_VALUE_HOLDS (value, G_TYPE_FILE))
-- >    my_widget_set_file (self, g_value_get_object (value));
-- >  else if (G_VALUE_HOLDS (value, GDK_TYPE_PIXBUF))
-- >    my_widget_set_pixbuf (self, g_value_get_object (value));
-- >  else
-- >    return FALSE;
-- >
-- >  return TRUE;
-- >}
-- >
-- >static void
-- >my_widget_init (MyWidget *self)
-- >{
-- >  GtkDropTarget *target =
-- >    gtk_drop_target_new (G_TYPE_INVALID, GDK_ACTION_COPY);
-- >
-- >  // This widget accepts two types of drop types: GFile objects
-- >  // and GdkPixbuf objects
-- >  gtk_drop_target_set_gtypes (target, (GType [2]) {
-- >    G_TYPE_FILE,
-- >    GDK_TYPE_PIXBUF,
-- >  }, 2);
-- >
-- >  g_signal_connect (target, "drop", G_CALLBACK (on_drop), self);
-- >  gtk_widget_add_controller (GTK_WIDGET (self), GTK_EVENT_CONTROLLER (target));
-- >}
-- 
-- 
-- @GtkDropTarget@ supports more options, such as:
-- 
--  * rejecting potential drops via the [DropTarget::accept]("GI.Gtk.Objects.DropTarget#g:signal:accept") signal
--    and the 'GI.Gtk.Objects.DropTarget.dropTargetReject' function to let other drop
--    targets handle the drop
--  * tracking an ongoing drag operation before the drop via the
--    [DropTarget::enter]("GI.Gtk.Objects.DropTarget#g:signal:enter"), [DropTarget::motion]("GI.Gtk.Objects.DropTarget#g:signal:motion") and
--    [DropTarget::leave]("GI.Gtk.Objects.DropTarget#g:signal:leave") signals
--  * configuring how to receive data by setting the
--    [DropTarget:preload]("GI.Gtk.Objects.DropTarget#g:attr:preload") property and listening for its
--    availability via the [DropTarget:value]("GI.Gtk.Objects.DropTarget#g:attr:value") property
-- 
-- However, @GtkDropTarget@ is ultimately modeled in a synchronous way
-- and only supports data transferred via @GType@. If you want full control
-- over an ongoing drop, the t'GI.Gtk.Objects.DropTargetAsync.DropTargetAsync' object gives you
-- this ability.
-- 
-- While a pointer is dragged over the drop target\'s widget and the drop
-- has not been rejected, that widget will receive the
-- 'GI.Gtk.Flags.StateFlagsDropActive' state, which can be used to style the widget.
-- 
-- If you are not interested in receiving the drop, but just want to update
-- UI state during a Drag-and-Drop operation (e.g. switching tabs), you can
-- use t'GI.Gtk.Objects.DropControllerMotion.DropControllerMotion'.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.DropTarget
    ( 

-- * Exported types
    DropTarget(..)                          ,
    IsDropTarget                            ,
    toDropTarget                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [reject]("GI.Gtk.Objects.DropTarget#g:method:reject"), [reset]("GI.Gtk.Objects.EventController#g:method:reset"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getActions]("GI.Gtk.Objects.DropTarget#g:method:getActions"), [getCurrentDrop]("GI.Gtk.Objects.DropTarget#g:method:getCurrentDrop"), [getCurrentEvent]("GI.Gtk.Objects.EventController#g:method:getCurrentEvent"), [getCurrentEventDevice]("GI.Gtk.Objects.EventController#g:method:getCurrentEventDevice"), [getCurrentEventState]("GI.Gtk.Objects.EventController#g:method:getCurrentEventState"), [getCurrentEventTime]("GI.Gtk.Objects.EventController#g:method:getCurrentEventTime"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDrop]("GI.Gtk.Objects.DropTarget#g:method:getDrop"), [getFormats]("GI.Gtk.Objects.DropTarget#g:method:getFormats"), [getGtypes]("GI.Gtk.Objects.DropTarget#g:method:getGtypes"), [getName]("GI.Gtk.Objects.EventController#g:method:getName"), [getPreload]("GI.Gtk.Objects.DropTarget#g:method:getPreload"), [getPropagationLimit]("GI.Gtk.Objects.EventController#g:method:getPropagationLimit"), [getPropagationPhase]("GI.Gtk.Objects.EventController#g:method:getPropagationPhase"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getValue]("GI.Gtk.Objects.DropTarget#g:method:getValue"), [getWidget]("GI.Gtk.Objects.EventController#g:method:getWidget").
-- 
-- ==== Setters
-- [setActions]("GI.Gtk.Objects.DropTarget#g:method:setActions"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setGtypes]("GI.Gtk.Objects.DropTarget#g:method:setGtypes"), [setName]("GI.Gtk.Objects.EventController#g:method:setName"), [setPreload]("GI.Gtk.Objects.DropTarget#g:method:setPreload"), [setPropagationLimit]("GI.Gtk.Objects.EventController#g:method:setPropagationLimit"), [setPropagationPhase]("GI.Gtk.Objects.EventController#g:method:setPropagationPhase"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setStaticName]("GI.Gtk.Objects.EventController#g:method:setStaticName").

#if defined(ENABLE_OVERLOADING)
    ResolveDropTargetMethod                 ,
#endif

-- ** getActions #method:getActions#

#if defined(ENABLE_OVERLOADING)
    DropTargetGetActionsMethodInfo          ,
#endif
    dropTargetGetActions                    ,


-- ** getCurrentDrop #method:getCurrentDrop#

#if defined(ENABLE_OVERLOADING)
    DropTargetGetCurrentDropMethodInfo      ,
#endif
    dropTargetGetCurrentDrop                ,


-- ** getDrop #method:getDrop#

#if defined(ENABLE_OVERLOADING)
    DropTargetGetDropMethodInfo             ,
#endif
    dropTargetGetDrop                       ,


-- ** getFormats #method:getFormats#

#if defined(ENABLE_OVERLOADING)
    DropTargetGetFormatsMethodInfo          ,
#endif
    dropTargetGetFormats                    ,


-- ** getGtypes #method:getGtypes#

#if defined(ENABLE_OVERLOADING)
    DropTargetGetGtypesMethodInfo           ,
#endif
    dropTargetGetGtypes                     ,


-- ** getPreload #method:getPreload#

#if defined(ENABLE_OVERLOADING)
    DropTargetGetPreloadMethodInfo          ,
#endif
    dropTargetGetPreload                    ,


-- ** getValue #method:getValue#

#if defined(ENABLE_OVERLOADING)
    DropTargetGetValueMethodInfo            ,
#endif
    dropTargetGetValue                      ,


-- ** new #method:new#

    dropTargetNew                           ,


-- ** reject #method:reject#

#if defined(ENABLE_OVERLOADING)
    DropTargetRejectMethodInfo              ,
#endif
    dropTargetReject                        ,


-- ** setActions #method:setActions#

#if defined(ENABLE_OVERLOADING)
    DropTargetSetActionsMethodInfo          ,
#endif
    dropTargetSetActions                    ,


-- ** setGtypes #method:setGtypes#

#if defined(ENABLE_OVERLOADING)
    DropTargetSetGtypesMethodInfo           ,
#endif
    dropTargetSetGtypes                     ,


-- ** setPreload #method:setPreload#

#if defined(ENABLE_OVERLOADING)
    DropTargetSetPreloadMethodInfo          ,
#endif
    dropTargetSetPreload                    ,




 -- * Properties


-- ** actions #attr:actions#
-- | The @GdkDragActions@ that this drop target supports.

#if defined(ENABLE_OVERLOADING)
    DropTargetActionsPropertyInfo           ,
#endif
    constructDropTargetActions              ,
#if defined(ENABLE_OVERLOADING)
    dropTargetActions                       ,
#endif
    getDropTargetActions                    ,
    setDropTargetActions                    ,


-- ** currentDrop #attr:currentDrop#
-- | The @GdkDrop@ that is currently being performed.
-- 
-- /Since: 4.4/

#if defined(ENABLE_OVERLOADING)
    DropTargetCurrentDropPropertyInfo       ,
#endif
#if defined(ENABLE_OVERLOADING)
    dropTargetCurrentDrop                   ,
#endif
    getDropTargetCurrentDrop                ,


-- ** drop #attr:drop#
-- | The @GdkDrop@ that is currently being performed.

#if defined(ENABLE_OVERLOADING)
    DropTargetDropPropertyInfo              ,
#endif
#if defined(ENABLE_OVERLOADING)
    dropTargetDrop                          ,
#endif
    getDropTargetDrop                       ,


-- ** formats #attr:formats#
-- | The @GdkContentFormats@ that determine the supported data formats.

#if defined(ENABLE_OVERLOADING)
    DropTargetFormatsPropertyInfo           ,
#endif
    constructDropTargetFormats              ,
#if defined(ENABLE_OVERLOADING)
    dropTargetFormats                       ,
#endif
    getDropTargetFormats                    ,


-- ** preload #attr:preload#
-- | Whether the drop data should be preloaded when the pointer is only
-- hovering over the widget but has not been released.
-- 
-- Setting this property allows finer grained reaction to an ongoing
-- drop at the cost of loading more data.
-- 
-- The default value for this property is 'P.False' to avoid downloading
-- huge amounts of data by accident.
-- 
-- For example, if somebody drags a full document of gigabytes of text
-- from a text editor across a widget with a preloading drop target,
-- this data will be downloaded, even if the data is ultimately dropped
-- elsewhere.
-- 
-- For a lot of data formats, the amount of data is very small (like
-- @/GDK_TYPE_RGBA/@), so enabling this property does not hurt at all.
-- And for local-only Drag-and-Drop operations, no data transfer is done,
-- so enabling it there is free.

#if defined(ENABLE_OVERLOADING)
    DropTargetPreloadPropertyInfo           ,
#endif
    constructDropTargetPreload              ,
#if defined(ENABLE_OVERLOADING)
    dropTargetPreload                       ,
#endif
    getDropTargetPreload                    ,
    setDropTargetPreload                    ,


-- ** value #attr:value#
-- | The value for this drop operation.
-- 
-- This is 'P.Nothing' if the data has not been loaded yet or no drop
-- operation is going on.
-- 
-- Data may be available before the [DropTarget::drop]("GI.Gtk.Objects.DropTarget#g:signal:drop")
-- signal gets emitted - for example when the [DropTarget:preload]("GI.Gtk.Objects.DropTarget#g:attr:preload")
-- property is set. You can use the [notify](#g:signal:notify) signal to be notified
-- of available data.

#if defined(ENABLE_OVERLOADING)
    DropTargetValuePropertyInfo             ,
#endif
#if defined(ENABLE_OVERLOADING)
    dropTargetValue                         ,
#endif
    getDropTargetValue                      ,




 -- * Signals


-- ** accept #signal:accept#

    DropTargetAcceptCallback                ,
#if defined(ENABLE_OVERLOADING)
    DropTargetAcceptSignalInfo              ,
#endif
    afterDropTargetAccept                   ,
    onDropTargetAccept                      ,


-- ** drop #signal:drop#

    DropTargetDropCallback                  ,
#if defined(ENABLE_OVERLOADING)
    DropTargetDropSignalInfo                ,
#endif
    afterDropTargetDrop                     ,
    onDropTargetDrop                        ,


-- ** enter #signal:enter#

    DropTargetEnterCallback                 ,
#if defined(ENABLE_OVERLOADING)
    DropTargetEnterSignalInfo               ,
#endif
    afterDropTargetEnter                    ,
    onDropTargetEnter                       ,


-- ** leave #signal:leave#

    DropTargetLeaveCallback                 ,
#if defined(ENABLE_OVERLOADING)
    DropTargetLeaveSignalInfo               ,
#endif
    afterDropTargetLeave                    ,
    onDropTargetLeave                       ,


-- ** motion #signal:motion#

    DropTargetMotionCallback                ,
#if defined(ENABLE_OVERLOADING)
    DropTargetMotionSignalInfo              ,
#endif
    afterDropTargetMotion                   ,
    onDropTargetMotion                      ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Drop as Gdk.Drop
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.ContentFormats as Gdk.ContentFormats
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.ComponentTransfer as Gsk.ComponentTransfer
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Objects.Drop as Gdk.Drop
import qualified GI.Gdk.Structs.ContentFormats as Gdk.ContentFormats
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController

#endif

-- | Memory-managed wrapper type.
newtype DropTarget = DropTarget (SP.ManagedPtr DropTarget)
    deriving (Eq)

instance SP.ManagedPtrNewtype DropTarget where
    toManagedPtr (DropTarget p) = p

foreign import ccall "gtk_drop_target_get_type"
    c_gtk_drop_target_get_type :: IO B.Types.GType

instance B.Types.TypedObject DropTarget where
    glibType = c_gtk_drop_target_get_type

instance B.Types.GObject DropTarget

-- | Type class for types which can be safely cast to t'DropTarget', for instance with `toDropTarget`.
class (SP.GObject o, O.IsDescendantOf DropTarget o) => IsDropTarget o
instance (SP.GObject o, O.IsDescendantOf DropTarget o) => IsDropTarget o

instance O.HasParentTypes DropTarget
type instance O.ParentTypes DropTarget = '[Gtk.EventController.EventController, GObject.Object.Object]

-- | Cast to t'DropTarget', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toDropTarget :: (MIO.MonadIO m, IsDropTarget o) => o -> m DropTarget
toDropTarget = MIO.liftIO . B.ManagedPtr.unsafeCastTo DropTarget

-- | Convert t'DropTarget' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe DropTarget) where
    gvalueGType_ = c_gtk_drop_target_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr DropTarget)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr DropTarget)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject DropTarget ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveDropTargetMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveDropTargetMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveDropTargetMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveDropTargetMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveDropTargetMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveDropTargetMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveDropTargetMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveDropTargetMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveDropTargetMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveDropTargetMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveDropTargetMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveDropTargetMethod "reject" o = DropTargetRejectMethodInfo
    ResolveDropTargetMethod "reset" o = Gtk.EventController.EventControllerResetMethodInfo
    ResolveDropTargetMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveDropTargetMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveDropTargetMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveDropTargetMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveDropTargetMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveDropTargetMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveDropTargetMethod "getActions" o = DropTargetGetActionsMethodInfo
    ResolveDropTargetMethod "getCurrentDrop" o = DropTargetGetCurrentDropMethodInfo
    ResolveDropTargetMethod "getCurrentEvent" o = Gtk.EventController.EventControllerGetCurrentEventMethodInfo
    ResolveDropTargetMethod "getCurrentEventDevice" o = Gtk.EventController.EventControllerGetCurrentEventDeviceMethodInfo
    ResolveDropTargetMethod "getCurrentEventState" o = Gtk.EventController.EventControllerGetCurrentEventStateMethodInfo
    ResolveDropTargetMethod "getCurrentEventTime" o = Gtk.EventController.EventControllerGetCurrentEventTimeMethodInfo
    ResolveDropTargetMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveDropTargetMethod "getDrop" o = DropTargetGetDropMethodInfo
    ResolveDropTargetMethod "getFormats" o = DropTargetGetFormatsMethodInfo
    ResolveDropTargetMethod "getGtypes" o = DropTargetGetGtypesMethodInfo
    ResolveDropTargetMethod "getName" o = Gtk.EventController.EventControllerGetNameMethodInfo
    ResolveDropTargetMethod "getPreload" o = DropTargetGetPreloadMethodInfo
    ResolveDropTargetMethod "getPropagationLimit" o = Gtk.EventController.EventControllerGetPropagationLimitMethodInfo
    ResolveDropTargetMethod "getPropagationPhase" o = Gtk.EventController.EventControllerGetPropagationPhaseMethodInfo
    ResolveDropTargetMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveDropTargetMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveDropTargetMethod "getValue" o = DropTargetGetValueMethodInfo
    ResolveDropTargetMethod "getWidget" o = Gtk.EventController.EventControllerGetWidgetMethodInfo
    ResolveDropTargetMethod "setActions" o = DropTargetSetActionsMethodInfo
    ResolveDropTargetMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveDropTargetMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveDropTargetMethod "setGtypes" o = DropTargetSetGtypesMethodInfo
    ResolveDropTargetMethod "setName" o = Gtk.EventController.EventControllerSetNameMethodInfo
    ResolveDropTargetMethod "setPreload" o = DropTargetSetPreloadMethodInfo
    ResolveDropTargetMethod "setPropagationLimit" o = Gtk.EventController.EventControllerSetPropagationLimitMethodInfo
    ResolveDropTargetMethod "setPropagationPhase" o = Gtk.EventController.EventControllerSetPropagationPhaseMethodInfo
    ResolveDropTargetMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveDropTargetMethod "setStaticName" o = Gtk.EventController.EventControllerSetStaticNameMethodInfo
    ResolveDropTargetMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveDropTargetMethod t DropTarget, O.OverloadedMethod info DropTarget p) => OL.IsLabel t (DropTarget -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveDropTargetMethod t DropTarget, O.OverloadedMethod info DropTarget p, R.HasField t DropTarget p) => R.HasField t DropTarget p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveDropTargetMethod t DropTarget, O.OverloadedMethodInfo info DropTarget) => OL.IsLabel t (O.MethodProxy info DropTarget) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal DropTarget::accept
-- | Emitted on the drop site when a drop operation is about to begin.
-- 
-- If the drop is not accepted, 'P.False' will be returned and the drop target
-- will ignore the drop. If 'P.True' is returned, the drop is accepted for now
-- but may be rejected later via a call to 'GI.Gtk.Objects.DropTarget.dropTargetReject'
-- or ultimately by returning 'P.False' from a [DropTarget::drop]("GI.Gtk.Objects.DropTarget#g:signal:drop")
-- handler.
-- 
-- The default handler for this signal decides whether to accept the drop
-- based on the formats provided by the /@drop@/.
-- 
-- If the decision whether the drop will be accepted or rejected depends
-- on the data, this function should return 'P.True', the
-- [DropTarget:preload]("GI.Gtk.Objects.DropTarget#g:attr:preload") property should be set and the value
-- should be inspected via the [notify](#g:signal:notify):value signal, calling
-- 'GI.Gtk.Objects.DropTarget.dropTargetReject' if required.
type DropTargetAcceptCallback =
    Gdk.Drop.Drop
    -- ^ /@drop@/: the @GdkDrop@
    -> IO Bool
    -- ^ __Returns:__ 'P.True' if /@drop@/ is accepted

type C_DropTargetAcceptCallback =
    Ptr DropTarget ->                       -- object
    Ptr Gdk.Drop.Drop ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_DropTargetAcceptCallback`.
foreign import ccall "wrapper"
    mk_DropTargetAcceptCallback :: C_DropTargetAcceptCallback -> IO (FunPtr C_DropTargetAcceptCallback)

wrap_DropTargetAcceptCallback :: 
    GObject a => (a -> DropTargetAcceptCallback) ->
    C_DropTargetAcceptCallback
wrap_DropTargetAcceptCallback gi'cb gi'selfPtr drop _ = do
    drop' <- (newObject Gdk.Drop.Drop) drop
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  drop'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [accept](#signal:accept) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dropTarget #accept callback
-- @
-- 
-- 
onDropTargetAccept :: (IsDropTarget a, MonadIO m) => a -> ((?self :: a) => DropTargetAcceptCallback) -> m SignalHandlerId
onDropTargetAccept obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DropTargetAcceptCallback wrapped
    wrapped'' <- mk_DropTargetAcceptCallback wrapped'
    connectSignalFunPtr obj "accept" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [accept](#signal:accept) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dropTarget #accept callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDropTargetAccept :: (IsDropTarget a, MonadIO m) => a -> ((?self :: a) => DropTargetAcceptCallback) -> m SignalHandlerId
afterDropTargetAccept obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DropTargetAcceptCallback wrapped
    wrapped'' <- mk_DropTargetAcceptCallback wrapped'
    connectSignalFunPtr obj "accept" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DropTargetAcceptSignalInfo
instance SignalInfo DropTargetAcceptSignalInfo where
    type HaskellCallbackType DropTargetAcceptSignalInfo = DropTargetAcceptCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DropTargetAcceptCallback cb
        cb'' <- mk_DropTargetAcceptCallback cb'
        connectSignalFunPtr obj "accept" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget::accept"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:signal:accept"})

#endif

-- signal DropTarget::drop
-- | Emitted on the drop site when the user drops the data onto the widget.
-- 
-- The signal handler must determine whether the pointer position is in
-- a drop zone or not. If it is not in a drop zone, it returns 'P.False'
-- and no further processing is necessary.
-- 
-- Otherwise, the handler returns 'P.True'. In this case, this handler will
-- accept the drop. The handler is responsible for using the given /@value@/
-- and performing the drop operation.
type DropTargetDropCallback =
    GValue
    -- ^ /@value@/: the @GValue@ being dropped
    -> Double
    -- ^ /@x@/: the x coordinate of the current pointer position
    -> Double
    -- ^ /@y@/: the y coordinate of the current pointer position
    -> IO Bool
    -- ^ __Returns:__ whether the drop was accepted at the given pointer position

type C_DropTargetDropCallback =
    Ptr DropTarget ->                       -- object
    Ptr GValue ->
    CDouble ->
    CDouble ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_DropTargetDropCallback`.
foreign import ccall "wrapper"
    mk_DropTargetDropCallback :: C_DropTargetDropCallback -> IO (FunPtr C_DropTargetDropCallback)

wrap_DropTargetDropCallback :: 
    GObject a => (a -> DropTargetDropCallback) ->
    C_DropTargetDropCallback
wrap_DropTargetDropCallback gi'cb gi'selfPtr value x y _ = do
    value' <- B.GValue.newGValueFromPtr value
    let x' = realToFrac x
    let y' = realToFrac y
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  value' x' y'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [drop](#signal:drop) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dropTarget #drop callback
-- @
-- 
-- 
onDropTargetDrop :: (IsDropTarget a, MonadIO m) => a -> ((?self :: a) => DropTargetDropCallback) -> m SignalHandlerId
onDropTargetDrop obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DropTargetDropCallback wrapped
    wrapped'' <- mk_DropTargetDropCallback wrapped'
    connectSignalFunPtr obj "drop" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [drop](#signal:drop) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dropTarget #drop callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDropTargetDrop :: (IsDropTarget a, MonadIO m) => a -> ((?self :: a) => DropTargetDropCallback) -> m SignalHandlerId
afterDropTargetDrop obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DropTargetDropCallback wrapped
    wrapped'' <- mk_DropTargetDropCallback wrapped'
    connectSignalFunPtr obj "drop" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DropTargetDropSignalInfo
instance SignalInfo DropTargetDropSignalInfo where
    type HaskellCallbackType DropTargetDropSignalInfo = DropTargetDropCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DropTargetDropCallback cb
        cb'' <- mk_DropTargetDropCallback cb'
        connectSignalFunPtr obj "drop" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget::drop"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:signal:drop"})

#endif

-- signal DropTarget::enter
-- | Emitted on the drop site when the pointer enters the widget.
-- 
-- It can be used to set up custom highlighting.
type DropTargetEnterCallback =
    Double
    -- ^ /@x@/: the x coordinate of the current pointer position
    -> Double
    -- ^ /@y@/: the y coordinate of the current pointer position
    -> IO [Gdk.Flags.DragAction]
    -- ^ __Returns:__ Preferred action for this drag operation or @GDK_ACTION_NONE@ if
    --   dropping is not supported at the current /@x@/,/@y@/ location.

type C_DropTargetEnterCallback =
    Ptr DropTarget ->                       -- object
    CDouble ->
    CDouble ->
    Ptr () ->                               -- user_data
    IO CUInt

-- | Generate a function pointer callable from C code, from a `C_DropTargetEnterCallback`.
foreign import ccall "wrapper"
    mk_DropTargetEnterCallback :: C_DropTargetEnterCallback -> IO (FunPtr C_DropTargetEnterCallback)

wrap_DropTargetEnterCallback :: 
    GObject a => (a -> DropTargetEnterCallback) ->
    C_DropTargetEnterCallback
wrap_DropTargetEnterCallback gi'cb gi'selfPtr x y _ = do
    let x' = realToFrac x
    let y' = realToFrac y
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  x' y'
    let result' = gflagsToWord result
    return result'


-- | Connect a signal handler for the [enter](#signal:enter) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dropTarget #enter callback
-- @
-- 
-- 
onDropTargetEnter :: (IsDropTarget a, MonadIO m) => a -> ((?self :: a) => DropTargetEnterCallback) -> m SignalHandlerId
onDropTargetEnter obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DropTargetEnterCallback wrapped
    wrapped'' <- mk_DropTargetEnterCallback wrapped'
    connectSignalFunPtr obj "enter" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [enter](#signal:enter) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dropTarget #enter callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDropTargetEnter :: (IsDropTarget a, MonadIO m) => a -> ((?self :: a) => DropTargetEnterCallback) -> m SignalHandlerId
afterDropTargetEnter obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DropTargetEnterCallback wrapped
    wrapped'' <- mk_DropTargetEnterCallback wrapped'
    connectSignalFunPtr obj "enter" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DropTargetEnterSignalInfo
instance SignalInfo DropTargetEnterSignalInfo where
    type HaskellCallbackType DropTargetEnterSignalInfo = DropTargetEnterCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DropTargetEnterCallback cb
        cb'' <- mk_DropTargetEnterCallback cb'
        connectSignalFunPtr obj "enter" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget::enter"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:signal:enter"})

#endif

-- signal DropTarget::leave
-- | Emitted on the drop site when the pointer leaves the widget.
-- 
-- Its main purpose it to undo things done in
-- [DropTarget::enter]("GI.Gtk.Objects.DropTarget#g:signal:enter").
type DropTargetLeaveCallback =
    IO ()

type C_DropTargetLeaveCallback =
    Ptr DropTarget ->                       -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_DropTargetLeaveCallback`.
foreign import ccall "wrapper"
    mk_DropTargetLeaveCallback :: C_DropTargetLeaveCallback -> IO (FunPtr C_DropTargetLeaveCallback)

wrap_DropTargetLeaveCallback :: 
    GObject a => (a -> DropTargetLeaveCallback) ->
    C_DropTargetLeaveCallback
wrap_DropTargetLeaveCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [leave](#signal:leave) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dropTarget #leave callback
-- @
-- 
-- 
onDropTargetLeave :: (IsDropTarget a, MonadIO m) => a -> ((?self :: a) => DropTargetLeaveCallback) -> m SignalHandlerId
onDropTargetLeave obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DropTargetLeaveCallback wrapped
    wrapped'' <- mk_DropTargetLeaveCallback wrapped'
    connectSignalFunPtr obj "leave" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [leave](#signal:leave) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dropTarget #leave callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDropTargetLeave :: (IsDropTarget a, MonadIO m) => a -> ((?self :: a) => DropTargetLeaveCallback) -> m SignalHandlerId
afterDropTargetLeave obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DropTargetLeaveCallback wrapped
    wrapped'' <- mk_DropTargetLeaveCallback wrapped'
    connectSignalFunPtr obj "leave" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DropTargetLeaveSignalInfo
instance SignalInfo DropTargetLeaveSignalInfo where
    type HaskellCallbackType DropTargetLeaveSignalInfo = DropTargetLeaveCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DropTargetLeaveCallback cb
        cb'' <- mk_DropTargetLeaveCallback cb'
        connectSignalFunPtr obj "leave" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget::leave"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:signal:leave"})

#endif

-- signal DropTarget::motion
-- | Emitted while the pointer is moving over the drop target.
type DropTargetMotionCallback =
    Double
    -- ^ /@x@/: the x coordinate of the current pointer position
    -> Double
    -- ^ /@y@/: the y coordinate of the current pointer position
    -> IO [Gdk.Flags.DragAction]
    -- ^ __Returns:__ Preferred action for this drag operation or @GDK_ACTION_NONE@ if
    --   dropping is not supported at the current /@x@/,/@y@/ location.

type C_DropTargetMotionCallback =
    Ptr DropTarget ->                       -- object
    CDouble ->
    CDouble ->
    Ptr () ->                               -- user_data
    IO CUInt

-- | Generate a function pointer callable from C code, from a `C_DropTargetMotionCallback`.
foreign import ccall "wrapper"
    mk_DropTargetMotionCallback :: C_DropTargetMotionCallback -> IO (FunPtr C_DropTargetMotionCallback)

wrap_DropTargetMotionCallback :: 
    GObject a => (a -> DropTargetMotionCallback) ->
    C_DropTargetMotionCallback
wrap_DropTargetMotionCallback gi'cb gi'selfPtr x y _ = do
    let x' = realToFrac x
    let y' = realToFrac y
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  x' y'
    let result' = gflagsToWord result
    return result'


-- | Connect a signal handler for the [motion](#signal:motion) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dropTarget #motion callback
-- @
-- 
-- 
onDropTargetMotion :: (IsDropTarget a, MonadIO m) => a -> ((?self :: a) => DropTargetMotionCallback) -> m SignalHandlerId
onDropTargetMotion obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DropTargetMotionCallback wrapped
    wrapped'' <- mk_DropTargetMotionCallback wrapped'
    connectSignalFunPtr obj "motion" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [motion](#signal:motion) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dropTarget #motion callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDropTargetMotion :: (IsDropTarget a, MonadIO m) => a -> ((?self :: a) => DropTargetMotionCallback) -> m SignalHandlerId
afterDropTargetMotion obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DropTargetMotionCallback wrapped
    wrapped'' <- mk_DropTargetMotionCallback wrapped'
    connectSignalFunPtr obj "motion" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DropTargetMotionSignalInfo
instance SignalInfo DropTargetMotionSignalInfo where
    type HaskellCallbackType DropTargetMotionSignalInfo = DropTargetMotionCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DropTargetMotionCallback cb
        cb'' <- mk_DropTargetMotionCallback cb'
        connectSignalFunPtr obj "motion" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget::motion"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:signal:motion"})

#endif

-- VVV Prop "actions"
   -- Type: TInterface (Name {namespace = "Gdk", name = "DragAction"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@actions@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dropTarget #actions
-- @
getDropTargetActions :: (MonadIO m, IsDropTarget o) => o -> m [Gdk.Flags.DragAction]
getDropTargetActions obj = MIO.liftIO $ B.Properties.getObjectPropertyFlags obj "actions"

-- | Set the value of the “@actions@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' dropTarget [ #actions 'Data.GI.Base.Attributes.:=' value ]
-- @
setDropTargetActions :: (MonadIO m, IsDropTarget o) => o -> [Gdk.Flags.DragAction] -> m ()
setDropTargetActions obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyFlags obj "actions" val

-- | Construct a t'GValueConstruct' with valid value for the “@actions@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDropTargetActions :: (IsDropTarget o, MIO.MonadIO m) => [Gdk.Flags.DragAction] -> m (GValueConstruct o)
constructDropTargetActions val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFlags "actions" val

#if defined(ENABLE_OVERLOADING)
data DropTargetActionsPropertyInfo
instance AttrInfo DropTargetActionsPropertyInfo where
    type AttrAllowedOps DropTargetActionsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint DropTargetActionsPropertyInfo = IsDropTarget
    type AttrSetTypeConstraint DropTargetActionsPropertyInfo = (~) [Gdk.Flags.DragAction]
    type AttrTransferTypeConstraint DropTargetActionsPropertyInfo = (~) [Gdk.Flags.DragAction]
    type AttrTransferType DropTargetActionsPropertyInfo = [Gdk.Flags.DragAction]
    type AttrGetType DropTargetActionsPropertyInfo = [Gdk.Flags.DragAction]
    type AttrLabel DropTargetActionsPropertyInfo = "actions"
    type AttrOrigin DropTargetActionsPropertyInfo = DropTarget
    attrGet = getDropTargetActions
    attrSet = setDropTargetActions
    attrPut = setDropTargetActions
    attrTransfer _ v = do
        return v
    attrConstruct = constructDropTargetActions
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.actions"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:attr:actions"
        })
#endif

-- VVV Prop "current-drop"
   -- Type: TInterface (Name {namespace = "Gdk", name = "Drop"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@current-drop@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dropTarget #currentDrop
-- @
getDropTargetCurrentDrop :: (MonadIO m, IsDropTarget o) => o -> m (Maybe Gdk.Drop.Drop)
getDropTargetCurrentDrop obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "current-drop" Gdk.Drop.Drop

#if defined(ENABLE_OVERLOADING)
data DropTargetCurrentDropPropertyInfo
instance AttrInfo DropTargetCurrentDropPropertyInfo where
    type AttrAllowedOps DropTargetCurrentDropPropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DropTargetCurrentDropPropertyInfo = IsDropTarget
    type AttrSetTypeConstraint DropTargetCurrentDropPropertyInfo = (~) ()
    type AttrTransferTypeConstraint DropTargetCurrentDropPropertyInfo = (~) ()
    type AttrTransferType DropTargetCurrentDropPropertyInfo = ()
    type AttrGetType DropTargetCurrentDropPropertyInfo = (Maybe Gdk.Drop.Drop)
    type AttrLabel DropTargetCurrentDropPropertyInfo = "current-drop"
    type AttrOrigin DropTargetCurrentDropPropertyInfo = DropTarget
    attrGet = getDropTargetCurrentDrop
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.currentDrop"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:attr:currentDrop"
        })
#endif

-- VVV Prop "drop"
   -- Type: TInterface (Name {namespace = "Gdk", name = "Drop"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@drop@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dropTarget #drop
-- @
getDropTargetDrop :: (MonadIO m, IsDropTarget o) => o -> m (Maybe Gdk.Drop.Drop)
getDropTargetDrop obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "drop" Gdk.Drop.Drop

#if defined(ENABLE_OVERLOADING)
data DropTargetDropPropertyInfo
instance AttrInfo DropTargetDropPropertyInfo where
    type AttrAllowedOps DropTargetDropPropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DropTargetDropPropertyInfo = IsDropTarget
    type AttrSetTypeConstraint DropTargetDropPropertyInfo = (~) ()
    type AttrTransferTypeConstraint DropTargetDropPropertyInfo = (~) ()
    type AttrTransferType DropTargetDropPropertyInfo = ()
    type AttrGetType DropTargetDropPropertyInfo = (Maybe Gdk.Drop.Drop)
    type AttrLabel DropTargetDropPropertyInfo = "drop"
    type AttrOrigin DropTargetDropPropertyInfo = DropTarget
    attrGet = getDropTargetDrop
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.drop"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:attr:drop"
        })
#endif

-- VVV Prop "formats"
   -- Type: TInterface (Name {namespace = "Gdk", name = "ContentFormats"})
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@formats@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dropTarget #formats
-- @
getDropTargetFormats :: (MonadIO m, IsDropTarget o) => o -> m (Maybe Gdk.ContentFormats.ContentFormats)
getDropTargetFormats obj = MIO.liftIO $ B.Properties.getObjectPropertyBoxed obj "formats" Gdk.ContentFormats.ContentFormats

-- | Construct a t'GValueConstruct' with valid value for the “@formats@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDropTargetFormats :: (IsDropTarget o, MIO.MonadIO m) => Gdk.ContentFormats.ContentFormats -> m (GValueConstruct o)
constructDropTargetFormats val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBoxed "formats" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data DropTargetFormatsPropertyInfo
instance AttrInfo DropTargetFormatsPropertyInfo where
    type AttrAllowedOps DropTargetFormatsPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DropTargetFormatsPropertyInfo = IsDropTarget
    type AttrSetTypeConstraint DropTargetFormatsPropertyInfo = (~) Gdk.ContentFormats.ContentFormats
    type AttrTransferTypeConstraint DropTargetFormatsPropertyInfo = (~) Gdk.ContentFormats.ContentFormats
    type AttrTransferType DropTargetFormatsPropertyInfo = Gdk.ContentFormats.ContentFormats
    type AttrGetType DropTargetFormatsPropertyInfo = (Maybe Gdk.ContentFormats.ContentFormats)
    type AttrLabel DropTargetFormatsPropertyInfo = "formats"
    type AttrOrigin DropTargetFormatsPropertyInfo = DropTarget
    attrGet = getDropTargetFormats
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructDropTargetFormats
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.formats"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:attr:formats"
        })
#endif

-- VVV Prop "preload"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@preload@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dropTarget #preload
-- @
getDropTargetPreload :: (MonadIO m, IsDropTarget o) => o -> m Bool
getDropTargetPreload obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "preload"

-- | Set the value of the “@preload@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' dropTarget [ #preload 'Data.GI.Base.Attributes.:=' value ]
-- @
setDropTargetPreload :: (MonadIO m, IsDropTarget o) => o -> Bool -> m ()
setDropTargetPreload obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "preload" val

-- | Construct a t'GValueConstruct' with valid value for the “@preload@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDropTargetPreload :: (IsDropTarget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructDropTargetPreload val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "preload" val

#if defined(ENABLE_OVERLOADING)
data DropTargetPreloadPropertyInfo
instance AttrInfo DropTargetPreloadPropertyInfo where
    type AttrAllowedOps DropTargetPreloadPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint DropTargetPreloadPropertyInfo = IsDropTarget
    type AttrSetTypeConstraint DropTargetPreloadPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint DropTargetPreloadPropertyInfo = (~) Bool
    type AttrTransferType DropTargetPreloadPropertyInfo = Bool
    type AttrGetType DropTargetPreloadPropertyInfo = Bool
    type AttrLabel DropTargetPreloadPropertyInfo = "preload"
    type AttrOrigin DropTargetPreloadPropertyInfo = DropTarget
    attrGet = getDropTargetPreload
    attrSet = setDropTargetPreload
    attrPut = setDropTargetPreload
    attrTransfer _ v = do
        return v
    attrConstruct = constructDropTargetPreload
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.preload"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:attr:preload"
        })
#endif

-- VVV Prop "value"
   -- Type: TGValue
   -- Flags: [PropertyReadable]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@value@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dropTarget #value
-- @
getDropTargetValue :: (MonadIO m, IsDropTarget o) => o -> m (Maybe GValue)
getDropTargetValue obj = MIO.liftIO $ B.Properties.getObjectPropertyGValue obj "value"

#if defined(ENABLE_OVERLOADING)
data DropTargetValuePropertyInfo
instance AttrInfo DropTargetValuePropertyInfo where
    type AttrAllowedOps DropTargetValuePropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DropTargetValuePropertyInfo = IsDropTarget
    type AttrSetTypeConstraint DropTargetValuePropertyInfo = (~) ()
    type AttrTransferTypeConstraint DropTargetValuePropertyInfo = (~) ()
    type AttrTransferType DropTargetValuePropertyInfo = ()
    type AttrGetType DropTargetValuePropertyInfo = (Maybe GValue)
    type AttrLabel DropTargetValuePropertyInfo = "value"
    type AttrOrigin DropTargetValuePropertyInfo = DropTarget
    attrGet = getDropTargetValue
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.value"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#g:attr:value"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList DropTarget
type instance O.AttributeList DropTarget = DropTargetAttributeList
type DropTargetAttributeList = ('[ '("actions", DropTargetActionsPropertyInfo), '("currentDrop", DropTargetCurrentDropPropertyInfo), '("drop", DropTargetDropPropertyInfo), '("formats", DropTargetFormatsPropertyInfo), '("name", Gtk.EventController.EventControllerNamePropertyInfo), '("preload", DropTargetPreloadPropertyInfo), '("propagationLimit", Gtk.EventController.EventControllerPropagationLimitPropertyInfo), '("propagationPhase", Gtk.EventController.EventControllerPropagationPhasePropertyInfo), '("value", DropTargetValuePropertyInfo), '("widget", Gtk.EventController.EventControllerWidgetPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
dropTargetActions :: AttrLabelProxy "actions"
dropTargetActions = AttrLabelProxy

dropTargetCurrentDrop :: AttrLabelProxy "currentDrop"
dropTargetCurrentDrop = AttrLabelProxy

dropTargetDrop :: AttrLabelProxy "drop"
dropTargetDrop = AttrLabelProxy

dropTargetFormats :: AttrLabelProxy "formats"
dropTargetFormats = AttrLabelProxy

dropTargetPreload :: AttrLabelProxy "preload"
dropTargetPreload = AttrLabelProxy

dropTargetValue :: AttrLabelProxy "value"
dropTargetValue = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList DropTarget = DropTargetSignalList
type DropTargetSignalList = ('[ '("accept", DropTargetAcceptSignalInfo), '("drop", DropTargetDropSignalInfo), '("enter", DropTargetEnterSignalInfo), '("leave", DropTargetLeaveSignalInfo), '("motion", DropTargetMotionSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method DropTarget::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The supported type or %G_TYPE_INVALID"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "actions"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "DragAction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the supported actions"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "DropTarget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_new" gtk_drop_target_new :: 
    CGType ->                               -- type : TBasicType TGType
    CUInt ->                                -- actions : TInterface (Name {namespace = "Gdk", name = "DragAction"})
    IO (Ptr DropTarget)

-- | Creates a new @GtkDropTarget@ object.
-- 
-- If the drop target should support more than 1 type, pass
-- @/G_TYPE_INVALID/@ for /@type@/ and then call
-- 'GI.Gtk.Objects.DropTarget.dropTargetSetGtypes'.
dropTargetNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GType
    -- ^ /@type@/: The supported type or @/G_TYPE_INVALID/@
    -> [Gdk.Flags.DragAction]
    -- ^ /@actions@/: the supported actions
    -> m DropTarget
    -- ^ __Returns:__ the new @GtkDropTarget@
dropTargetNew type_ actions = liftIO $ do
    let type_' = gtypeToCGType type_
    let actions' = gflagsToWord actions
    result <- gtk_drop_target_new type_' actions'
    checkUnexpectedReturnNULL "dropTargetNew" result
    result' <- (wrapObject DropTarget) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method DropTarget::get_actions
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "DragAction" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_get_actions" gtk_drop_target_get_actions :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    IO CUInt

-- | Gets the actions that this drop target supports.
dropTargetGetActions ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> m [Gdk.Flags.DragAction]
    -- ^ __Returns:__ the actions that this drop target supports
dropTargetGetActions self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_drop_target_get_actions self'
    let result' = wordToGFlags result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data DropTargetGetActionsMethodInfo
instance (signature ~ (m [Gdk.Flags.DragAction]), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetGetActionsMethodInfo a signature where
    overloadedMethod = dropTargetGetActions

instance O.OverloadedMethodInfo DropTargetGetActionsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetGetActions",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetGetActions"
        })


#endif

-- method DropTarget::get_current_drop
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Drop" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_get_current_drop" gtk_drop_target_get_current_drop :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    IO (Ptr Gdk.Drop.Drop)

-- | Gets the currently handled drop operation.
-- 
-- If no drop operation is going on, 'P.Nothing' is returned.
-- 
-- /Since: 4.4/
dropTargetGetCurrentDrop ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> m (Maybe Gdk.Drop.Drop)
    -- ^ __Returns:__ The current drop
dropTargetGetCurrentDrop self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_drop_target_get_current_drop self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gdk.Drop.Drop) result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DropTargetGetCurrentDropMethodInfo
instance (signature ~ (m (Maybe Gdk.Drop.Drop)), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetGetCurrentDropMethodInfo a signature where
    overloadedMethod = dropTargetGetCurrentDrop

instance O.OverloadedMethodInfo DropTargetGetCurrentDropMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetGetCurrentDrop",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetGetCurrentDrop"
        })


#endif

-- method DropTarget::get_drop
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Drop" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_get_drop" gtk_drop_target_get_drop :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    IO (Ptr Gdk.Drop.Drop)

{-# DEPRECATED dropTargetGetDrop ["(Since version 4.4)","Use 'GI.Gtk.Objects.DropTarget.dropTargetGetCurrentDrop' instead"] #-}
-- | Gets the currently handled drop operation.
-- 
-- If no drop operation is going on, 'P.Nothing' is returned.
dropTargetGetDrop ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> m (Maybe Gdk.Drop.Drop)
    -- ^ __Returns:__ The current drop
dropTargetGetDrop self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_drop_target_get_drop self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gdk.Drop.Drop) result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DropTargetGetDropMethodInfo
instance (signature ~ (m (Maybe Gdk.Drop.Drop)), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetGetDropMethodInfo a signature where
    overloadedMethod = dropTargetGetDrop

instance O.OverloadedMethodInfo DropTargetGetDropMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetGetDrop",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetGetDrop"
        })


#endif

-- method DropTarget::get_formats
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gdk" , name = "ContentFormats" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_get_formats" gtk_drop_target_get_formats :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    IO (Ptr Gdk.ContentFormats.ContentFormats)

-- | Gets the data formats that this drop target accepts.
-- 
-- If the result is 'P.Nothing', all formats are expected to be supported.
dropTargetGetFormats ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> m (Maybe Gdk.ContentFormats.ContentFormats)
    -- ^ __Returns:__ the supported data formats
dropTargetGetFormats self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_drop_target_get_formats self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Gdk.ContentFormats.ContentFormats) result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DropTargetGetFormatsMethodInfo
instance (signature ~ (m (Maybe Gdk.ContentFormats.ContentFormats)), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetGetFormatsMethodInfo a signature where
    overloadedMethod = dropTargetGetFormats

instance O.OverloadedMethodInfo DropTargetGetFormatsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetGetFormats",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetGetFormats"
        })


#endif

-- method DropTarget::get_gtypes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_types"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the number of `GType`s contained in the\n  return value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_types"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just "the number of `GType`s contained in the\n  return value"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 1 (TBasicType TGType))
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_get_gtypes" gtk_drop_target_get_gtypes :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    Ptr FCT.CSize ->                        -- n_types : TBasicType TSize
    IO (Ptr CGType)

-- | Gets the list of supported @GType@s that can be dropped on the target.
-- 
-- If no types have been set, @NULL@ will be returned.
dropTargetGetGtypes ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> m (Maybe [GType])
    -- ^ __Returns:__ 
    --   the @G_TYPE_INVALID@-terminated array of types included in
    --   formats
dropTargetGetGtypes self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    nTypes <- allocMem :: IO (Ptr FCT.CSize)
    result <- gtk_drop_target_get_gtypes self' nTypes
    nTypes' <- peek nTypes
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (unpackMapStorableArrayWithLength GType nTypes') result'
        return result''
    touchManagedPtr self
    freeMem nTypes
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DropTargetGetGtypesMethodInfo
instance (signature ~ (m (Maybe [GType])), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetGetGtypesMethodInfo a signature where
    overloadedMethod = dropTargetGetGtypes

instance O.OverloadedMethodInfo DropTargetGetGtypesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetGetGtypes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetGetGtypes"
        })


#endif

-- method DropTarget::get_preload
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_get_preload" gtk_drop_target_get_preload :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    IO CInt

-- | Gets whether data should be preloaded on hover.
dropTargetGetPreload ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if drop data should be preloaded
dropTargetGetPreload self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_drop_target_get_preload self'
    let result' = (/= 0) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data DropTargetGetPreloadMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetGetPreloadMethodInfo a signature where
    overloadedMethod = dropTargetGetPreload

instance O.OverloadedMethodInfo DropTargetGetPreloadMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetGetPreload",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetGetPreload"
        })


#endif

-- method DropTarget::get_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TGValue
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_get_value" gtk_drop_target_get_value :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    IO (Ptr GValue)

-- | Gets the current drop data, as a @GValue@.
dropTargetGetValue ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> m (Maybe GValue)
    -- ^ __Returns:__ The current drop data
dropTargetGetValue self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_drop_target_get_value self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GValue.newGValueFromPtr result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DropTargetGetValueMethodInfo
instance (signature ~ (m (Maybe GValue)), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetGetValueMethodInfo a signature where
    overloadedMethod = dropTargetGetValue

instance O.OverloadedMethodInfo DropTargetGetValueMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetGetValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetGetValue"
        })


#endif

-- method DropTarget::reject
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_reject" gtk_drop_target_reject :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    IO ()

-- | Rejects the ongoing drop operation.
-- 
-- If no drop operation is ongoing, i.e when [DropTarget:currentDrop]("GI.Gtk.Objects.DropTarget#g:attr:currentDrop")
-- is 'P.Nothing', this function does nothing.
-- 
-- This function should be used when delaying the decision
-- on whether to accept a drag or not until after reading
-- the data.
dropTargetReject ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> m ()
dropTargetReject self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    gtk_drop_target_reject self'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data DropTargetRejectMethodInfo
instance (signature ~ (m ()), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetRejectMethodInfo a signature where
    overloadedMethod = dropTargetReject

instance O.OverloadedMethodInfo DropTargetRejectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetReject",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetReject"
        })


#endif

-- method DropTarget::set_actions
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "actions"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "DragAction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the supported actions"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_set_actions" gtk_drop_target_set_actions :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    CUInt ->                                -- actions : TInterface (Name {namespace = "Gdk", name = "DragAction"})
    IO ()

-- | Sets the actions that this drop target supports.
dropTargetSetActions ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> [Gdk.Flags.DragAction]
    -- ^ /@actions@/: the supported actions
    -> m ()
dropTargetSetActions self actions = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let actions' = gflagsToWord actions
    gtk_drop_target_set_actions self' actions'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data DropTargetSetActionsMethodInfo
instance (signature ~ ([Gdk.Flags.DragAction] -> m ()), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetSetActionsMethodInfo a signature where
    overloadedMethod = dropTargetSetActions

instance O.OverloadedMethodInfo DropTargetSetActionsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetSetActions",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetSetActions"
        })


#endif

-- method DropTarget::set_gtypes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "types"
--           , argType = TCArray False (-1) 2 (TBasicType TGType)
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "all supported `GType`s\n  that can be dropped on the target"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_types"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "number of @types" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_types"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "number of @types" , sinceVersion = Nothing }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_set_gtypes" gtk_drop_target_set_gtypes :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    Ptr CGType ->                           -- types : TCArray False (-1) 2 (TBasicType TGType)
    FCT.CSize ->                            -- n_types : TBasicType TSize
    IO ()

-- | Sets the supported @GType@s for this drop target.
dropTargetSetGtypes ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> Maybe ([GType])
    -- ^ /@types@/: all supported @GType@s
    --   that can be dropped on the target
    -> m ()
dropTargetSetGtypes self types = liftIO $ do
    let nTypes = case types of
            Nothing -> 0
            Just jTypes -> fromIntegral $ P.length jTypes
    self' <- unsafeManagedPtrCastPtr self
    maybeTypes <- case types of
        Nothing -> return FP.nullPtr
        Just jTypes -> do
            jTypes' <- (packMapStorableArray gtypeToCGType) jTypes
            return jTypes'
    gtk_drop_target_set_gtypes self' maybeTypes nTypes
    touchManagedPtr self
    freeMem maybeTypes
    return ()

#if defined(ENABLE_OVERLOADING)
data DropTargetSetGtypesMethodInfo
instance (signature ~ (Maybe ([GType]) -> m ()), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetSetGtypesMethodInfo a signature where
    overloadedMethod = dropTargetSetGtypes

instance O.OverloadedMethodInfo DropTargetSetGtypesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetSetGtypes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetSetGtypes"
        })


#endif

-- method DropTarget::set_preload
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DropTarget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDropTarget`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "preload"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE to preload drop data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drop_target_set_preload" gtk_drop_target_set_preload :: 
    Ptr DropTarget ->                       -- self : TInterface (Name {namespace = "Gtk", name = "DropTarget"})
    CInt ->                                 -- preload : TBasicType TBoolean
    IO ()

-- | Sets whether data should be preloaded on hover.
dropTargetSetPreload ::
    (B.CallStack.HasCallStack, MonadIO m, IsDropTarget a) =>
    a
    -- ^ /@self@/: a @GtkDropTarget@
    -> Bool
    -- ^ /@preload@/: 'P.True' to preload drop data
    -> m ()
dropTargetSetPreload self preload = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let preload' = (P.fromIntegral . P.fromEnum) preload
    gtk_drop_target_set_preload self' preload'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data DropTargetSetPreloadMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsDropTarget a) => O.OverloadedMethod DropTargetSetPreloadMethodInfo a signature where
    overloadedMethod = dropTargetSetPreload

instance O.OverloadedMethodInfo DropTargetSetPreloadMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.DropTarget.dropTargetSetPreload",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-DropTarget.html#v:dropTargetSetPreload"
        })


#endif


