// Copyright (C) 2018 The Android Open Source Project
// Copyright (C) 2018 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Autogenerated module VkDecoder
//
// (impl) generated by codegen/vulkan/vulkan-docs/scripts/genvk.py -registry
// codegen/vulkan/vulkan-docs/xml/vk.xml -registryGfxstream
// codegen/vulkan/vulkan-docs/xml/vk_gfxstream.xml cereal -o host/vulkan/cereal
//
// Please do not modify directly;
// re-run gfxstream-protocols/scripts/generate-vulkan-sources.sh,
// or directly from Python by defining:
// VULKAN_REGISTRY_XML_DIR : Directory containing vk.xml
// VULKAN_REGISTRY_SCRIPTS_DIR : Directory containing genvk.py
// CEREAL_OUTPUT_DIR: Where to put the generated sources.
//
// python3 $VULKAN_REGISTRY_SCRIPTS_DIR/genvk.py -registry $VULKAN_REGISTRY_XML_DIR/vk.xml cereal -o
// $CEREAL_OUTPUT_DIR
//

#include "VkDecoder.h"

#include <functional>
#include <optional>
#include <unordered_map>

#include "VkDecoderGlobalState.h"
#include "VkDecoderSnapshot.h"
#include "VulkanDispatch.h"
#include "VulkanStream.h"
#include "aemu/base/BumpPool.h"
#include "aemu/base/Metrics.h"
#include "aemu/base/Tracing.h"
#include "aemu/base/system/System.h"
#include "common/goldfish_vk_marshaling.h"
#include "common/goldfish_vk_private_defs.h"
#include "common/goldfish_vk_reserved_marshaling.h"
#include "common/goldfish_vk_transform.h"
#include "host-common/GfxstreamFatalError.h"
#include "host-common/feature_control.h"
#include "host-common/logging.h"
#include "host/FrameBuffer.h"
#include "render-utils/IOStream.h"
#define MAX_PACKET_LENGTH (400 * 1024 * 1024)  // 400MB

namespace gfxstream {
namespace vk {

using android::base::MetricEventBadPacketLength;
using android::base::MetricEventDuplicateSequenceNum;

class VkDecoder::Impl {
   public:
    Impl()
        : m_logCalls(android::base::getEnvironmentVariable("ANDROID_EMU_VK_LOG_CALLS") == "1"),
          m_vk(vkDispatch()),
          m_state(VkDecoderGlobalState::get()),
          m_boxedHandleUnwrapMapping(m_state),
          m_boxedHandleCreateMapping(m_state),
          m_boxedHandleDestroyMapping(m_state),
          m_boxedHandleUnwrapAndDeleteMapping(m_state),
          m_boxedHandleUnwrapAndDeletePreserveBoxedMapping(m_state),
          m_prevSeqno(std::nullopt) {}
    VulkanStream* stream() { return &m_vkStream; }
    VulkanMemReadingStream* readStream() { return &m_vkMemReadingStream; }

    void setForSnapshotLoad(bool forSnapshotLoad) { m_forSnapshotLoad = forSnapshotLoad; }

    size_t decode(void* buf, size_t bufsize, IOStream* stream,
                  const ProcessResources* processResources, const VkDecoderContext&);

   private:
    bool m_logCalls;
    bool m_forSnapshotLoad = false;
    VulkanDispatch* m_vk;
    VkDecoderGlobalState* m_state;
    VulkanStream m_vkStream{nullptr};
    VulkanMemReadingStream m_vkMemReadingStream{nullptr};
    BoxedHandleUnwrapMapping m_boxedHandleUnwrapMapping;
    BoxedHandleCreateMapping m_boxedHandleCreateMapping;
    BoxedHandleDestroyMapping m_boxedHandleDestroyMapping;
    BoxedHandleUnwrapAndDeleteMapping m_boxedHandleUnwrapAndDeleteMapping;
    android::base::BumpPool m_pool;
    BoxedHandleUnwrapAndDeletePreserveBoxedMapping m_boxedHandleUnwrapAndDeletePreserveBoxedMapping;
    std::optional<uint32_t> m_prevSeqno;
};

VkDecoder::VkDecoder() : mImpl(new VkDecoder::Impl()) {}

VkDecoder::~VkDecoder() = default;

void VkDecoder::setForSnapshotLoad(bool forSnapshotLoad) {
    mImpl->setForSnapshotLoad(forSnapshotLoad);
}

size_t VkDecoder::decode(void* buf, size_t bufsize, IOStream* stream,
                         const ProcessResources* processResources,
                         const VkDecoderContext& context) {
    return mImpl->decode(buf, bufsize, stream, processResources, context);
}

// VkDecoder::Impl::decode to follow

size_t VkDecoder::Impl::decode(void* buf, size_t len, IOStream* ioStream,
                               const ProcessResources* processResources,
                               const VkDecoderContext& context) {
    const char* processName = context.processName;
    auto& gfx_logger = *context.gfxApiLogger;
    auto* healthMonitor = context.healthMonitor;
    auto& metricsLogger = *context.metricsLogger;
    if (len < 8) return 0;
    bool queueSubmitWithCommandsEnabled =
        feature_is_enabled(kFeature_VulkanQueueSubmitWithCommands);
    unsigned char* ptr = (unsigned char*)buf;
    const unsigned char* const end = (const unsigned char*)buf + len;
    if (m_forSnapshotLoad) {
        ptr += m_state->setCreatedHandlesForSnapshotLoad(ptr);
    }
    while (end - ptr >= 8) {
        uint32_t opcode = *(uint32_t*)ptr;
        uint32_t packetLen = *(uint32_t*)(ptr + 4);

        // packetLen should be at least 8 (op code and packet length) and should not be excessively
        // large
        if (packetLen < 8 || packetLen > MAX_PACKET_LENGTH) {
            WARN("Bad packet length %d detected, decode may fail", packetLen);
            metricsLogger.logMetricEvent(MetricEventBadPacketLength{.len = packetLen});
        }

        if (end - ptr < packetLen) return ptr - (unsigned char*)buf;
        gfx_logger.record(ptr, std::min(size_t(packetLen + 8), size_t(end - ptr)));
        stream()->setStream(ioStream);
        VulkanStream* vkStream = stream();
        VulkanMemReadingStream* vkReadStream = readStream();
        vkReadStream->setBuf((uint8_t*)(ptr + 8));
        uint8_t* readStreamPtr = vkReadStream->getBuf();
        uint8_t** readStreamPtrPtr = &readStreamPtr;
        uint8_t* snapshotTraceBegin = vkReadStream->beginTrace();
        vkReadStream->setHandleMapping(&m_boxedHandleUnwrapMapping);

        std::unique_ptr<EventHangMetadata::HangAnnotations> executionData =
            std::make_unique<EventHangMetadata::HangAnnotations>();
        if (healthMonitor) {
            executionData->insert(
                {{"packet_length", std::to_string(packetLen)}, {"opcode", std::to_string(opcode)}});
            if (processName) {
                executionData->insert({{"renderthread_guest_process", std::string(processName)}});
            }
            if (m_prevSeqno) {
                executionData->insert({{"previous_seqno", std::to_string(m_prevSeqno.value())}});
            }
        }

        std::atomic<uint32_t>* seqnoPtr =
            processResources ? processResources->getSequenceNumberPtr() : nullptr;

        if (queueSubmitWithCommandsEnabled &&
            ((opcode >= OP_vkFirst && opcode < OP_vkLast) ||
             (opcode >= OP_vkFirst_old && opcode < OP_vkLast_old))) {
            uint32_t seqno;
            memcpy(&seqno, *readStreamPtrPtr, sizeof(uint32_t));
            *readStreamPtrPtr += sizeof(uint32_t);
            if (healthMonitor) executionData->insert({{"seqno", std::to_string(seqno)}});
            if (m_prevSeqno && seqno == m_prevSeqno.value()) {
                WARN(
                    "Seqno %d is the same as previously processed on thread %d. It might be a "
                    "duplicate command.",
                    seqno, getCurrentThreadId());
                metricsLogger.logMetricEvent(MetricEventDuplicateSequenceNum{.opcode = opcode});
            }
            if (seqnoPtr && !m_forSnapshotLoad) {
                {
                    auto seqnoWatchdog =
                        WATCHDOG_BUILDER(healthMonitor, "RenderThread seqno loop")
                            .setHangType(EventHangMetadata::HangType::kRenderThread)
                            .setAnnotations(std::make_unique<EventHangMetadata::HangAnnotations>(
                                *executionData))
                            /* Data gathered if this hangs*/
                            .setOnHangCallback([=]() {
                                auto annotations =
                                    std::make_unique<EventHangMetadata::HangAnnotations>();
                                annotations->insert(
                                    {{"seqnoPtr",
                                      std::to_string(seqnoPtr->load(std::memory_order_seq_cst))}});
                                return annotations;
                            })
                            .build();
                    while ((seqno - seqnoPtr->load(std::memory_order_seq_cst) != 1)) {
#if (defined(_MSC_VER) && (defined(_M_IX86) || defined(_M_X64)))
                        _mm_pause();
#elif (defined(__GNUC__) && (defined(__i386__) || defined(__x86_64__)))
                        __asm__ __volatile__("pause;");
#endif
                    }
                    m_prevSeqno = seqno;
                }
            }
        }

        gfx_logger.recordCommandExecution();

        auto executionWatchdog =
            WATCHDOG_BUILDER(healthMonitor, "RenderThread VkDecoder command execution")
                .setHangType(EventHangMetadata::HangType::kRenderThread)
                .setAnnotations(std::move(executionData))
                .build();

        auto vk = m_vk;
        switch (opcode) {
#ifdef VK_VERSION_1_0
            case OP_vkCreateInstance: {
                android::base::beginTrace("vkCreateInstance decode");
                const VkInstanceCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkInstance* pInstance;
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkInstanceCreateInfo));
                reservedunmarshal_VkInstanceCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                       (VkInstanceCreateInfo*)(pCreateInfo),
                                                       readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pInstance;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pInstance, sizeof(VkInstance));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkInstance*)pInstance = (VkInstance)(VkInstance)((VkInstance)(*&cgen_var_1));
                if (pCreateInfo) {
                    transform_tohost_VkInstanceCreateInfo(m_state,
                                                          (VkInstanceCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCreateInstance 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pInstance);
                }
                VkResult vkCreateInstance_VkResult_return = (VkResult)0;
                vkCreateInstance_VkResult_return =
                    m_state->on_vkCreateInstance(&m_pool, pCreateInfo, pAllocator, pInstance);
                if ((vkCreateInstance_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateInstance_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_2;
                static_assert(8 == sizeof(VkInstance),
                              "handle map overwrite requires VkInstance to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkInstance((VkInstance*)pInstance, 1);
                vkStream->write((VkInstance*)pInstance, 8 * 1);
                vkStream->write(&vkCreateInstance_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateInstance(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, vkCreateInstance_VkResult_return,
                                                          pCreateInfo, pAllocator, pInstance);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyInstance: {
                android::base::beginTrace("vkDestroyInstance decode");
                VkInstance instance;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for instance;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkInstance*)&instance = (VkInstance)(VkInstance)((VkInstance)(*&cgen_var_0));
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyInstance 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)instance, (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyInstance(&m_pool, instance, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyInstance(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, instance, pAllocator);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkEnumeratePhysicalDevices: {
                android::base::beginTrace("vkEnumeratePhysicalDevices decode");
                VkInstance instance;
                uint32_t* pPhysicalDeviceCount;
                VkPhysicalDevice* pPhysicalDevices;
                // Begin global wrapped dispatchable handle unboxing for instance;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkInstance*)&instance = (VkInstance)(VkInstance)((VkInstance)(*&cgen_var_0));
                // Begin manual dispatchable handle unboxing for pPhysicalDeviceCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pPhysicalDeviceCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPhysicalDeviceCount);
                *readStreamPtrPtr += 8;
                if (pPhysicalDeviceCount) {
                    vkReadStream->alloc((void**)&pPhysicalDeviceCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pPhysicalDeviceCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pPhysicalDevices;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkPhysicalDevice**)&pPhysicalDevices, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPhysicalDevices);
                *readStreamPtrPtr += 8;
                if (pPhysicalDevices) {
                    vkReadStream->alloc((void**)&pPhysicalDevices,
                                        (*(pPhysicalDeviceCount)) * sizeof(VkPhysicalDevice));
                    if ((*(pPhysicalDeviceCount))) {
                        uint8_t* cgen_var_2_0_ptr = (uint8_t*)(*readStreamPtrPtr);
                        *readStreamPtrPtr += 8 * (*(pPhysicalDeviceCount));
                        if (pPhysicalDeviceCount) {
                            for (uint32_t k = 0; k < (*(pPhysicalDeviceCount)); ++k) {
                                uint64_t tmpval;
                                memcpy(&tmpval, cgen_var_2_0_ptr + k * 8, sizeof(uint64_t));
                                *(((VkPhysicalDevice*)pPhysicalDevices) + k) =
                                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)tmpval);
                            }
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkEnumeratePhysicalDevices 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)instance,
                            (unsigned long long)pPhysicalDeviceCount,
                            (unsigned long long)pPhysicalDevices);
                }
                VkResult vkEnumeratePhysicalDevices_VkResult_return = (VkResult)0;
                vkEnumeratePhysicalDevices_VkResult_return = m_state->on_vkEnumeratePhysicalDevices(
                    &m_pool, instance, pPhysicalDeviceCount, pPhysicalDevices);
                if ((vkEnumeratePhysicalDevices_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkEnumeratePhysicalDevices_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pPhysicalDeviceCount;
                vkStream->putBe64(cgen_var_3);
                if (pPhysicalDeviceCount) {
                    vkStream->write((uint32_t*)pPhysicalDeviceCount, sizeof(uint32_t));
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pPhysicalDevices;
                vkStream->putBe64(cgen_var_4);
                if (pPhysicalDevices) {
                    if ((*(pPhysicalDeviceCount))) {
                        uint64_t* cgen_var_4_0;
                        vkStream->alloc((void**)&cgen_var_4_0, (*(pPhysicalDeviceCount)) * 8);
                        static_assert(
                            8 == sizeof(VkPhysicalDevice),
                            "handle map overwrite requires VkPhysicalDevice to be 8 bytes long");
                        vkStream->handleMapping()->mapHandles_VkPhysicalDevice(
                            (VkPhysicalDevice*)pPhysicalDevices, (*(pPhysicalDeviceCount)));
                        vkStream->write((VkPhysicalDevice*)pPhysicalDevices,
                                        8 * (*(pPhysicalDeviceCount)));
                    }
                }
                vkStream->write(&vkEnumeratePhysicalDevices_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkEnumeratePhysicalDevices(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkEnumeratePhysicalDevices_VkResult_return, instance, pPhysicalDeviceCount,
                        pPhysicalDevices);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceFeatures: {
                android::base::beginTrace("vkGetPhysicalDeviceFeatures decode");
                VkPhysicalDevice physicalDevice;
                VkPhysicalDeviceFeatures* pFeatures;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                // Begin manual dispatchable handle unboxing for pFeatures;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pFeatures, sizeof(VkPhysicalDeviceFeatures));
                reservedunmarshal_VkPhysicalDeviceFeatures(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                           (VkPhysicalDeviceFeatures*)(pFeatures),
                                                           readStreamPtrPtr);
                if (pFeatures) {
                    transform_tohost_VkPhysicalDeviceFeatures(
                        m_state, (VkPhysicalDeviceFeatures*)(pFeatures));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetPhysicalDeviceFeatures 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pFeatures);
                }
                m_state->on_vkGetPhysicalDeviceFeatures(&m_pool, physicalDevice, pFeatures);
                vkStream->unsetHandleMapping();
                if (pFeatures) {
                    transform_fromhost_VkPhysicalDeviceFeatures(
                        m_state, (VkPhysicalDeviceFeatures*)(pFeatures));
                }
                marshal_VkPhysicalDeviceFeatures(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                 (VkPhysicalDeviceFeatures*)(pFeatures));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceFeatures(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice, pFeatures);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceFormatProperties: {
                android::base::beginTrace("vkGetPhysicalDeviceFormatProperties decode");
                VkPhysicalDevice physicalDevice;
                VkFormat format;
                VkFormatProperties* pFormatProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                memcpy((VkFormat*)&format, *readStreamPtrPtr, sizeof(VkFormat));
                *readStreamPtrPtr += sizeof(VkFormat);
                // Begin manual dispatchable handle unboxing for pFormatProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pFormatProperties, sizeof(VkFormatProperties));
                reservedunmarshal_VkFormatProperties(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkFormatProperties*)(pFormatProperties),
                                                     readStreamPtrPtr);
                if (pFormatProperties) {
                    transform_tohost_VkFormatProperties(m_state,
                                                        (VkFormatProperties*)(pFormatProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceFormatProperties 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)format, (unsigned long long)pFormatProperties);
                }
                m_state->on_vkGetPhysicalDeviceFormatProperties(&m_pool, physicalDevice, format,
                                                                pFormatProperties);
                vkStream->unsetHandleMapping();
                if (pFormatProperties) {
                    transform_fromhost_VkFormatProperties(m_state,
                                                          (VkFormatProperties*)(pFormatProperties));
                }
                marshal_VkFormatProperties(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                           (VkFormatProperties*)(pFormatProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceFormatProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice, format,
                        pFormatProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceImageFormatProperties: {
                android::base::beginTrace("vkGetPhysicalDeviceImageFormatProperties decode");
                VkPhysicalDevice physicalDevice;
                VkFormat format;
                VkImageType type;
                VkImageTiling tiling;
                VkImageUsageFlags usage;
                VkImageCreateFlags flags;
                VkImageFormatProperties* pImageFormatProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                memcpy((VkFormat*)&format, *readStreamPtrPtr, sizeof(VkFormat));
                *readStreamPtrPtr += sizeof(VkFormat);
                memcpy((VkImageType*)&type, *readStreamPtrPtr, sizeof(VkImageType));
                *readStreamPtrPtr += sizeof(VkImageType);
                memcpy((VkImageTiling*)&tiling, *readStreamPtrPtr, sizeof(VkImageTiling));
                *readStreamPtrPtr += sizeof(VkImageTiling);
                memcpy((VkImageUsageFlags*)&usage, *readStreamPtrPtr, sizeof(VkImageUsageFlags));
                *readStreamPtrPtr += sizeof(VkImageUsageFlags);
                memcpy((VkImageCreateFlags*)&flags, *readStreamPtrPtr, sizeof(VkImageCreateFlags));
                *readStreamPtrPtr += sizeof(VkImageCreateFlags);
                // Begin manual dispatchable handle unboxing for pImageFormatProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pImageFormatProperties,
                                    sizeof(VkImageFormatProperties));
                reservedunmarshal_VkImageFormatProperties(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImageFormatProperties*)(pImageFormatProperties), readStreamPtrPtr);
                if (pImageFormatProperties) {
                    transform_tohost_VkImageFormatProperties(
                        m_state, (VkImageFormatProperties*)(pImageFormatProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceImageFormatProperties 0x%llx "
                            "0x%llx 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)format, (unsigned long long)type,
                            (unsigned long long)tiling, (unsigned long long)usage,
                            (unsigned long long)flags, (unsigned long long)pImageFormatProperties);
                }
                VkResult vkGetPhysicalDeviceImageFormatProperties_VkResult_return = (VkResult)0;
                vkGetPhysicalDeviceImageFormatProperties_VkResult_return =
                    m_state->on_vkGetPhysicalDeviceImageFormatProperties(
                        &m_pool, physicalDevice, format, type, tiling, usage, flags,
                        pImageFormatProperties);
                if ((vkGetPhysicalDeviceImageFormatProperties_VkResult_return) ==
                    VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(
                    vkGetPhysicalDeviceImageFormatProperties_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                if (pImageFormatProperties) {
                    transform_fromhost_VkImageFormatProperties(
                        m_state, (VkImageFormatProperties*)(pImageFormatProperties));
                }
                marshal_VkImageFormatProperties(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                (VkImageFormatProperties*)(pImageFormatProperties));
                vkStream->write(&vkGetPhysicalDeviceImageFormatProperties_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceImageFormatProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetPhysicalDeviceImageFormatProperties_VkResult_return, physicalDevice,
                        format, type, tiling, usage, flags, pImageFormatProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceProperties: {
                android::base::beginTrace("vkGetPhysicalDeviceProperties decode");
                VkPhysicalDevice physicalDevice;
                VkPhysicalDeviceProperties* pProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pProperties, sizeof(VkPhysicalDeviceProperties));
                reservedunmarshal_VkPhysicalDeviceProperties(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceProperties*)(pProperties), readStreamPtrPtr);
                if (pProperties) {
                    transform_tohost_VkPhysicalDeviceProperties(
                        m_state, (VkPhysicalDeviceProperties*)(pProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceProperties 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pProperties);
                }
                m_state->on_vkGetPhysicalDeviceProperties(&m_pool, physicalDevice, pProperties);
                vkStream->unsetHandleMapping();
                if (pProperties) {
                    transform_fromhost_VkPhysicalDeviceProperties(
                        m_state, (VkPhysicalDeviceProperties*)(pProperties));
                }
                marshal_VkPhysicalDeviceProperties(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkPhysicalDeviceProperties*)(pProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceProperties(snapshotTraceBegin,
                                                                       snapshotTraceBytes, &m_pool,
                                                                       physicalDevice, pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceQueueFamilyProperties: {
                android::base::beginTrace("vkGetPhysicalDeviceQueueFamilyProperties decode");
                VkPhysicalDevice physicalDevice;
                uint32_t* pQueueFamilyPropertyCount;
                VkQueueFamilyProperties* pQueueFamilyProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                // Begin manual dispatchable handle unboxing for pQueueFamilyPropertyCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pQueueFamilyPropertyCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pQueueFamilyPropertyCount);
                *readStreamPtrPtr += 8;
                if (pQueueFamilyPropertyCount) {
                    vkReadStream->alloc((void**)&pQueueFamilyPropertyCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pQueueFamilyPropertyCount, *readStreamPtrPtr,
                           sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pQueueFamilyProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkQueueFamilyProperties**)&pQueueFamilyProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pQueueFamilyProperties);
                *readStreamPtrPtr += 8;
                if (pQueueFamilyProperties) {
                    vkReadStream->alloc(
                        (void**)&pQueueFamilyProperties,
                        (*(pQueueFamilyPropertyCount)) * sizeof(VkQueueFamilyProperties));
                    for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                        reservedunmarshal_VkQueueFamilyProperties(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkQueueFamilyProperties*)(pQueueFamilyProperties + i),
                            readStreamPtrPtr);
                    }
                }
                if (pQueueFamilyPropertyCount) {
                    if (pQueueFamilyProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                            transform_tohost_VkQueueFamilyProperties(
                                m_state, (VkQueueFamilyProperties*)(pQueueFamilyProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceQueueFamilyProperties 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pQueueFamilyPropertyCount,
                            (unsigned long long)pQueueFamilyProperties);
                }
                vk->vkGetPhysicalDeviceQueueFamilyProperties(
                    unboxed_physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pQueueFamilyPropertyCount;
                vkStream->putBe64(cgen_var_3);
                if (pQueueFamilyPropertyCount) {
                    vkStream->write((uint32_t*)pQueueFamilyPropertyCount, sizeof(uint32_t));
                }
                if (pQueueFamilyPropertyCount) {
                    if (pQueueFamilyProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                            transform_fromhost_VkQueueFamilyProperties(
                                m_state, (VkQueueFamilyProperties*)(pQueueFamilyProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pQueueFamilyProperties;
                vkStream->putBe64(cgen_var_4);
                if (pQueueFamilyProperties) {
                    if (pQueueFamilyPropertyCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                            marshal_VkQueueFamilyProperties(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkQueueFamilyProperties*)(pQueueFamilyProperties + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceQueueFamilyProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pQueueFamilyPropertyCount, pQueueFamilyProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceMemoryProperties: {
                android::base::beginTrace("vkGetPhysicalDeviceMemoryProperties decode");
                VkPhysicalDevice physicalDevice;
                VkPhysicalDeviceMemoryProperties* pMemoryProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                // Begin manual dispatchable handle unboxing for pMemoryProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryProperties,
                                    sizeof(VkPhysicalDeviceMemoryProperties));
                reservedunmarshal_VkPhysicalDeviceMemoryProperties(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceMemoryProperties*)(pMemoryProperties), readStreamPtrPtr);
                if (pMemoryProperties) {
                    transform_tohost_VkPhysicalDeviceMemoryProperties(
                        m_state, (VkPhysicalDeviceMemoryProperties*)(pMemoryProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceMemoryProperties 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pMemoryProperties);
                }
                m_state->on_vkGetPhysicalDeviceMemoryProperties(&m_pool, physicalDevice,
                                                                pMemoryProperties);
                vkStream->unsetHandleMapping();
                if (pMemoryProperties) {
                    transform_fromhost_VkPhysicalDeviceMemoryProperties(
                        m_state, (VkPhysicalDeviceMemoryProperties*)(pMemoryProperties));
                }
                marshal_VkPhysicalDeviceMemoryProperties(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceMemoryProperties*)(pMemoryProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceMemoryProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pMemoryProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetInstanceProcAddr: {
                android::base::beginTrace("vkGetInstanceProcAddr decode");
                VkInstance instance;
                const char* pName;
                // Begin non wrapped dispatchable handle unboxing for instance;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkInstance*)&instance = (VkInstance)(VkInstance)((VkInstance)(*&cgen_var_0));
                auto unboxed_instance = unbox_VkInstance(instance);
                auto vk = dispatch_VkInstance(instance);
                // End manual dispatchable handle unboxing for instance;
                vkReadStream->loadStringInPlaceWithStreamPtr((char**)&pName, readStreamPtrPtr);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetInstanceProcAddr 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)instance, (unsigned long long)pName);
                }
                PFN_vkVoidFunction vkGetInstanceProcAddr_PFN_vkVoidFunction_return =
                    (PFN_vkVoidFunction)0;
                vkGetInstanceProcAddr_PFN_vkVoidFunction_return =
                    vk->vkGetInstanceProcAddr(unboxed_instance, pName);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkGetInstanceProcAddr_PFN_vkVoidFunction_return,
                                sizeof(PFN_vkVoidFunction));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetInstanceProcAddr(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetInstanceProcAddr_PFN_vkVoidFunction_return, instance, pName);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceProcAddr: {
                android::base::beginTrace("vkGetDeviceProcAddr decode");
                VkDevice device;
                const char* pName;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->loadStringInPlaceWithStreamPtr((char**)&pName, readStreamPtrPtr);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetDeviceProcAddr 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pName);
                }
                PFN_vkVoidFunction vkGetDeviceProcAddr_PFN_vkVoidFunction_return =
                    (PFN_vkVoidFunction)0;
                vkGetDeviceProcAddr_PFN_vkVoidFunction_return =
                    vk->vkGetDeviceProcAddr(unboxed_device, pName);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkGetDeviceProcAddr_PFN_vkVoidFunction_return,
                                sizeof(PFN_vkVoidFunction));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceProcAddr(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetDeviceProcAddr_PFN_vkVoidFunction_return, device, pName);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateDevice: {
                android::base::beginTrace("vkCreateDevice decode");
                VkPhysicalDevice physicalDevice;
                const VkDeviceCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkDevice* pDevice;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkDeviceCreateInfo));
                reservedunmarshal_VkDeviceCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkDeviceCreateInfo*)(pCreateInfo),
                                                     readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pDevice;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pDevice, sizeof(VkDevice));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkDevice*)pDevice = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkDeviceCreateInfo(m_state,
                                                        (VkDeviceCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCreateDevice 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pCreateInfo, (unsigned long long)pAllocator,
                            (unsigned long long)pDevice);
                }
                VkResult vkCreateDevice_VkResult_return = (VkResult)0;
                vkCreateDevice_VkResult_return = m_state->on_vkCreateDevice(
                    &m_pool, physicalDevice, pCreateInfo, pAllocator, pDevice);
                if ((vkCreateDevice_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateDevice_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkDevice),
                              "handle map overwrite requires VkDevice to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkDevice((VkDevice*)pDevice, 1);
                vkStream->write((VkDevice*)pDevice, 8 * 1);
                vkStream->write(&vkCreateDevice_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateDevice(snapshotTraceBegin, snapshotTraceBytes,
                                                        &m_pool, vkCreateDevice_VkResult_return,
                                                        physicalDevice, pCreateInfo, pAllocator,
                                                        pDevice);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyDevice: {
                android::base::beginTrace("vkDestroyDevice decode");
                VkDevice device;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyDevice 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)device, (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyDevice(&m_pool, device, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyDevice(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, device, pAllocator);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkEnumerateInstanceExtensionProperties: {
                android::base::beginTrace("vkEnumerateInstanceExtensionProperties decode");
                const char* pLayerName;
                uint32_t* pPropertyCount;
                VkExtensionProperties* pProperties;
                if (vkReadStream->getFeatureBits() &
                    VULKAN_STREAM_FEATURE_NULL_OPTIONAL_STRINGS_BIT) {
                    // WARNING PTR CHECK
                    memcpy((char**)&pLayerName, (*readStreamPtrPtr), 8);
                    android::base::Stream::fromBe64((uint8_t*)&pLayerName);
                    *readStreamPtrPtr += 8;
                    if (pLayerName) {
                        vkReadStream->loadStringInPlaceWithStreamPtr((char**)&pLayerName,
                                                                     readStreamPtrPtr);
                    }
                } else {
                    vkReadStream->loadStringInPlaceWithStreamPtr((char**)&pLayerName,
                                                                 readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pPropertyCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pPropertyCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPropertyCount);
                *readStreamPtrPtr += 8;
                if (pPropertyCount) {
                    vkReadStream->alloc((void**)&pPropertyCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pPropertyCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkExtensionProperties**)&pProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pProperties);
                *readStreamPtrPtr += 8;
                if (pProperties) {
                    vkReadStream->alloc((void**)&pProperties,
                                        (*(pPropertyCount)) * sizeof(VkExtensionProperties));
                    for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                        reservedunmarshal_VkExtensionProperties(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkExtensionProperties*)(pProperties + i), readStreamPtrPtr);
                    }
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_tohost_VkExtensionProperties(
                                m_state, (VkExtensionProperties*)(pProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkEnumerateInstanceExtensionProperties 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)pLayerName,
                            (unsigned long long)pPropertyCount, (unsigned long long)pProperties);
                }
                VkResult vkEnumerateInstanceExtensionProperties_VkResult_return = (VkResult)0;
                vkEnumerateInstanceExtensionProperties_VkResult_return =
                    vk->vkEnumerateInstanceExtensionProperties(pLayerName, pPropertyCount,
                                                               pProperties);
                if ((vkEnumerateInstanceExtensionProperties_VkResult_return) ==
                    VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkEnumerateInstanceExtensionProperties_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_2 = (uint64_t)(uintptr_t)pPropertyCount;
                vkStream->putBe64(cgen_var_2);
                if (pPropertyCount) {
                    vkStream->write((uint32_t*)pPropertyCount, sizeof(uint32_t));
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_fromhost_VkExtensionProperties(
                                m_state, (VkExtensionProperties*)(pProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pProperties;
                vkStream->putBe64(cgen_var_3);
                if (pProperties) {
                    if (pPropertyCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            marshal_VkExtensionProperties(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkExtensionProperties*)(pProperties + i));
                        }
                    }
                }
                vkStream->write(&vkEnumerateInstanceExtensionProperties_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkEnumerateInstanceExtensionProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkEnumerateInstanceExtensionProperties_VkResult_return, pLayerName,
                        pPropertyCount, pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkEnumerateDeviceExtensionProperties: {
                android::base::beginTrace("vkEnumerateDeviceExtensionProperties decode");
                VkPhysicalDevice physicalDevice;
                const char* pLayerName;
                uint32_t* pPropertyCount;
                VkExtensionProperties* pProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                if (vkReadStream->getFeatureBits() &
                    VULKAN_STREAM_FEATURE_NULL_OPTIONAL_STRINGS_BIT) {
                    // WARNING PTR CHECK
                    memcpy((char**)&pLayerName, (*readStreamPtrPtr), 8);
                    android::base::Stream::fromBe64((uint8_t*)&pLayerName);
                    *readStreamPtrPtr += 8;
                    if (pLayerName) {
                        vkReadStream->loadStringInPlaceWithStreamPtr((char**)&pLayerName,
                                                                     readStreamPtrPtr);
                    }
                } else {
                    vkReadStream->loadStringInPlaceWithStreamPtr((char**)&pLayerName,
                                                                 readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pPropertyCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pPropertyCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPropertyCount);
                *readStreamPtrPtr += 8;
                if (pPropertyCount) {
                    vkReadStream->alloc((void**)&pPropertyCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pPropertyCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkExtensionProperties**)&pProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pProperties);
                *readStreamPtrPtr += 8;
                if (pProperties) {
                    vkReadStream->alloc((void**)&pProperties,
                                        (*(pPropertyCount)) * sizeof(VkExtensionProperties));
                    for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                        reservedunmarshal_VkExtensionProperties(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkExtensionProperties*)(pProperties + i), readStreamPtrPtr);
                    }
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_tohost_VkExtensionProperties(
                                m_state, (VkExtensionProperties*)(pProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkEnumerateDeviceExtensionProperties 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pLayerName, (unsigned long long)pPropertyCount,
                            (unsigned long long)pProperties);
                }
                VkResult vkEnumerateDeviceExtensionProperties_VkResult_return = (VkResult)0;
                vkEnumerateDeviceExtensionProperties_VkResult_return =
                    m_state->on_vkEnumerateDeviceExtensionProperties(
                        &m_pool, physicalDevice, pLayerName, pPropertyCount, pProperties);
                if ((vkEnumerateDeviceExtensionProperties_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkEnumerateDeviceExtensionProperties_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pPropertyCount;
                vkStream->putBe64(cgen_var_3);
                if (pPropertyCount) {
                    vkStream->write((uint32_t*)pPropertyCount, sizeof(uint32_t));
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_fromhost_VkExtensionProperties(
                                m_state, (VkExtensionProperties*)(pProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pProperties;
                vkStream->putBe64(cgen_var_4);
                if (pProperties) {
                    if (pPropertyCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            marshal_VkExtensionProperties(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkExtensionProperties*)(pProperties + i));
                        }
                    }
                }
                vkStream->write(&vkEnumerateDeviceExtensionProperties_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkEnumerateDeviceExtensionProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkEnumerateDeviceExtensionProperties_VkResult_return, physicalDevice,
                        pLayerName, pPropertyCount, pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkEnumerateInstanceLayerProperties: {
                android::base::beginTrace("vkEnumerateInstanceLayerProperties decode");
                uint32_t* pPropertyCount;
                VkLayerProperties* pProperties;
                // Begin manual dispatchable handle unboxing for pPropertyCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pPropertyCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPropertyCount);
                *readStreamPtrPtr += 8;
                if (pPropertyCount) {
                    vkReadStream->alloc((void**)&pPropertyCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pPropertyCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkLayerProperties**)&pProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pProperties);
                *readStreamPtrPtr += 8;
                if (pProperties) {
                    vkReadStream->alloc((void**)&pProperties,
                                        (*(pPropertyCount)) * sizeof(VkLayerProperties));
                    for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                        reservedunmarshal_VkLayerProperties(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkLayerProperties*)(pProperties + i), readStreamPtrPtr);
                    }
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_tohost_VkLayerProperties(
                                m_state, (VkLayerProperties*)(pProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkEnumerateInstanceLayerProperties 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)pPropertyCount,
                            (unsigned long long)pProperties);
                }
                VkResult vkEnumerateInstanceLayerProperties_VkResult_return = (VkResult)0;
                vkEnumerateInstanceLayerProperties_VkResult_return =
                    vk->vkEnumerateInstanceLayerProperties(pPropertyCount, pProperties);
                if ((vkEnumerateInstanceLayerProperties_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkEnumerateInstanceLayerProperties_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_2 = (uint64_t)(uintptr_t)pPropertyCount;
                vkStream->putBe64(cgen_var_2);
                if (pPropertyCount) {
                    vkStream->write((uint32_t*)pPropertyCount, sizeof(uint32_t));
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_fromhost_VkLayerProperties(
                                m_state, (VkLayerProperties*)(pProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pProperties;
                vkStream->putBe64(cgen_var_3);
                if (pProperties) {
                    if (pPropertyCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            marshal_VkLayerProperties(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkLayerProperties*)(pProperties + i));
                        }
                    }
                }
                vkStream->write(&vkEnumerateInstanceLayerProperties_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkEnumerateInstanceLayerProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkEnumerateInstanceLayerProperties_VkResult_return, pPropertyCount,
                        pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkEnumerateDeviceLayerProperties: {
                android::base::beginTrace("vkEnumerateDeviceLayerProperties decode");
                VkPhysicalDevice physicalDevice;
                uint32_t* pPropertyCount;
                VkLayerProperties* pProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                // Begin manual dispatchable handle unboxing for pPropertyCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pPropertyCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPropertyCount);
                *readStreamPtrPtr += 8;
                if (pPropertyCount) {
                    vkReadStream->alloc((void**)&pPropertyCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pPropertyCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkLayerProperties**)&pProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pProperties);
                *readStreamPtrPtr += 8;
                if (pProperties) {
                    vkReadStream->alloc((void**)&pProperties,
                                        (*(pPropertyCount)) * sizeof(VkLayerProperties));
                    for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                        reservedunmarshal_VkLayerProperties(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkLayerProperties*)(pProperties + i), readStreamPtrPtr);
                    }
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_tohost_VkLayerProperties(
                                m_state, (VkLayerProperties*)(pProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkEnumerateDeviceLayerProperties 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)physicalDevice,
                        (unsigned long long)pPropertyCount, (unsigned long long)pProperties);
                }
                VkResult vkEnumerateDeviceLayerProperties_VkResult_return = (VkResult)0;
                vkEnumerateDeviceLayerProperties_VkResult_return =
                    vk->vkEnumerateDeviceLayerProperties(unboxed_physicalDevice, pPropertyCount,
                                                         pProperties);
                if ((vkEnumerateDeviceLayerProperties_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkEnumerateDeviceLayerProperties_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pPropertyCount;
                vkStream->putBe64(cgen_var_3);
                if (pPropertyCount) {
                    vkStream->write((uint32_t*)pPropertyCount, sizeof(uint32_t));
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_fromhost_VkLayerProperties(
                                m_state, (VkLayerProperties*)(pProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pProperties;
                vkStream->putBe64(cgen_var_4);
                if (pProperties) {
                    if (pPropertyCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            marshal_VkLayerProperties(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkLayerProperties*)(pProperties + i));
                        }
                    }
                }
                vkStream->write(&vkEnumerateDeviceLayerProperties_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkEnumerateDeviceLayerProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkEnumerateDeviceLayerProperties_VkResult_return, physicalDevice,
                        pPropertyCount, pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceQueue: {
                android::base::beginTrace("vkGetDeviceQueue decode");
                VkDevice device;
                uint32_t queueFamilyIndex;
                uint32_t queueIndex;
                VkQueue* pQueue;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                memcpy((uint32_t*)&queueFamilyIndex, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&queueIndex, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                // Begin manual dispatchable handle unboxing for pQueue;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pQueue, sizeof(VkQueue));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkQueue*)pQueue = (VkQueue)(VkQueue)((VkQueue)(*&cgen_var_1));
                if (m_logCalls) {
                    fprintf(
                        stderr, "stream %p: call vkGetDeviceQueue 0x%llx 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)queueFamilyIndex,
                        (unsigned long long)queueIndex, (unsigned long long)pQueue);
                }
                m_state->on_vkGetDeviceQueue(&m_pool, device, queueFamilyIndex, queueIndex, pQueue);
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_2;
                static_assert(8 == sizeof(VkQueue),
                              "handle map overwrite requires VkQueue to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkQueue((VkQueue*)pQueue, 1);
                vkStream->write((VkQueue*)pQueue, 8 * 1);
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceQueue(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, device, queueFamilyIndex,
                                                          queueIndex, pQueue);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkQueueSubmit: {
                android::base::beginTrace("vkQueueSubmit decode");
                VkQueue queue;
                uint32_t submitCount;
                const VkSubmitInfo* pSubmits;
                VkFence fence;
                // Begin global wrapped dispatchable handle unboxing for queue;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueue*)&queue = (VkQueue)(VkQueue)((VkQueue)(*&cgen_var_0));
                memcpy((uint32_t*)&submitCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pSubmits,
                                    ((submitCount)) * sizeof(const VkSubmitInfo));
                for (uint32_t i = 0; i < (uint32_t)((submitCount)); ++i) {
                    reservedunmarshal_VkSubmitInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkSubmitInfo*)(pSubmits + i), readStreamPtrPtr);
                }
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkFence*)&fence = (VkFence)unbox_VkFence((VkFence)(*&cgen_var_1));
                if (pSubmits) {
                    for (uint32_t i = 0; i < (uint32_t)((submitCount)); ++i) {
                        transform_tohost_VkSubmitInfo(m_state, (VkSubmitInfo*)(pSubmits + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkQueueSubmit 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)queue, (unsigned long long)submitCount,
                            (unsigned long long)pSubmits, (unsigned long long)fence);
                }
                VkResult vkQueueSubmit_VkResult_return = (VkResult)0;
                vkQueueSubmit_VkResult_return =
                    m_state->on_vkQueueSubmit(&m_pool, queue, submitCount, pSubmits, fence);
                if ((vkQueueSubmit_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkQueueSubmit_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkQueueSubmit_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkQueueSubmit(snapshotTraceBegin, snapshotTraceBytes,
                                                       &m_pool, vkQueueSubmit_VkResult_return,
                                                       queue, submitCount, pSubmits, fence);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkQueueWaitIdle: {
                android::base::beginTrace("vkQueueWaitIdle decode");
                VkQueue queue;
                // Begin global wrapped dispatchable handle unboxing for queue;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueue*)&queue = (VkQueue)(VkQueue)((VkQueue)(*&cgen_var_0));
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkQueueWaitIdle 0x%llx \n", ioStream,
                            (unsigned long long)queue);
                }
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                VkResult vkQueueWaitIdle_VkResult_return = (VkResult)0;
                vkQueueWaitIdle_VkResult_return = m_state->on_vkQueueWaitIdle(&m_pool, queue);
                if ((vkQueueWaitIdle_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkQueueWaitIdle_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkQueueWaitIdle_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkQueueWaitIdle(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, vkQueueWaitIdle_VkResult_return,
                                                         queue);
                }
                vkReadStream->clearPool();
                android::base::endTrace();
                break;
            }
            case OP_vkDeviceWaitIdle: {
                android::base::beginTrace("vkDeviceWaitIdle decode");
                VkDevice device;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDeviceWaitIdle 0x%llx \n", ioStream,
                            (unsigned long long)device);
                }
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                VkResult vkDeviceWaitIdle_VkResult_return = (VkResult)0;
                vkDeviceWaitIdle_VkResult_return = vk->vkDeviceWaitIdle(unboxed_device);
                if ((vkDeviceWaitIdle_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkDeviceWaitIdle_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkDeviceWaitIdle_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDeviceWaitIdle(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, vkDeviceWaitIdle_VkResult_return,
                                                          device);
                }
                vkReadStream->clearPool();
                android::base::endTrace();
                break;
            }
            case OP_vkAllocateMemory: {
                android::base::beginTrace("vkAllocateMemory decode");
                VkDevice device;
                const VkMemoryAllocateInfo* pAllocateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkDeviceMemory* pMemory;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pAllocateInfo, sizeof(const VkMemoryAllocateInfo));
                reservedunmarshal_VkMemoryAllocateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                       (VkMemoryAllocateInfo*)(pAllocateInfo),
                                                       readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pMemory;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemory, sizeof(VkDeviceMemory));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkDeviceMemory*)pMemory =
                    (VkDeviceMemory)(VkDeviceMemory)((VkDeviceMemory)(*&cgen_var_2));
                if (pAllocateInfo) {
                    transform_tohost_VkMemoryAllocateInfo(m_state,
                                                          (VkMemoryAllocateInfo*)(pAllocateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkAllocateMemory 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pAllocateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pMemory);
                }
                VkResult vkAllocateMemory_VkResult_return = (VkResult)0;
                vkAllocateMemory_VkResult_return = m_state->on_vkAllocateMemory(
                    &m_pool, device, pAllocateInfo, pAllocator, pMemory);
                if ((vkAllocateMemory_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(
                    vkAllocateMemory_VkResult_return, opcode, context,
                    std::make_optional<uint64_t>(pAllocateInfo->allocationSize));
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pMemory;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkDeviceMemory),
                              "handle map overwrite requires VkDeviceMemory to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkDeviceMemory((VkDeviceMemory*)pMemory, 1);
                vkStream->write((VkDeviceMemory*)pMemory, 8 * 1);
                // Begin manual non dispatchable handle create for pMemory;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkAllocateMemory_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkAllocateMemory(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, vkAllocateMemory_VkResult_return,
                                                          device, pAllocateInfo, pAllocator,
                                                          pMemory);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkFreeMemory: {
                android::base::beginTrace("vkFreeMemory decode");
                VkDevice device;
                VkDeviceMemory memory;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for memory;
                VkDeviceMemory boxed_memory_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDeviceMemory*)&memory =
                    (VkDeviceMemory)(VkDeviceMemory)((VkDeviceMemory)(*&cgen_var_1));
                boxed_memory_preserve = memory;
                memory = unbox_VkDeviceMemory(memory);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkFreeMemory 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)memory,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkFreeMemory(&m_pool, device, memory, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkFreeMemory(snapshotTraceBegin, snapshotTraceBytes,
                                                      &m_pool, device, boxed_memory_preserve,
                                                      pAllocator);
                }
                delete_VkDeviceMemory(boxed_memory_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkMapMemory: {
                android::base::beginTrace("vkMapMemory decode");
                VkDevice device;
                VkDeviceMemory memory;
                VkDeviceSize offset;
                VkDeviceSize size;
                VkMemoryMapFlags flags;
                void** ppData;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDeviceMemory*)&memory =
                    (VkDeviceMemory)unbox_VkDeviceMemory((VkDeviceMemory)(*&cgen_var_1));
                memcpy((VkDeviceSize*)&offset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((VkDeviceSize*)&size, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((VkMemoryMapFlags*)&flags, *readStreamPtrPtr, sizeof(VkMemoryMapFlags));
                *readStreamPtrPtr += sizeof(VkMemoryMapFlags);
                // Begin manual dispatchable handle unboxing for ppData;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((void***)&ppData, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&ppData);
                *readStreamPtrPtr += 8;
                if (ppData) {
                    vkReadStream->alloc((void**)&ppData, sizeof(void*));
                    memcpy((void**)ppData, *readStreamPtrPtr, sizeof(void*));
                    *readStreamPtrPtr += sizeof(void*);
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkMapMemory 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)memory,
                        (unsigned long long)offset, (unsigned long long)size,
                        (unsigned long long)flags, (unsigned long long)ppData);
                }
                VkResult vkMapMemory_VkResult_return = (VkResult)0;
                vkMapMemory_VkResult_return =
                    m_state->on_vkMapMemory(&m_pool, device, memory, offset, size, flags, ppData);
                if ((vkMapMemory_VkResult_return) == VK_ERROR_DEVICE_LOST) m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkMapMemory_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)ppData;
                vkStream->putBe64(cgen_var_3);
                if (ppData) {
                    vkStream->write((void**)ppData, sizeof(void*));
                }
                vkStream->write(&vkMapMemory_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkMapMemory(snapshotTraceBegin, snapshotTraceBytes,
                                                     &m_pool, vkMapMemory_VkResult_return, device,
                                                     memory, offset, size, flags, ppData);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkUnmapMemory: {
                android::base::beginTrace("vkUnmapMemory decode");
                VkDevice device;
                VkDeviceMemory memory;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDeviceMemory*)&memory =
                    (VkDeviceMemory)unbox_VkDeviceMemory((VkDeviceMemory)(*&cgen_var_1));
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkUnmapMemory 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)device, (unsigned long long)memory);
                }
                m_state->on_vkUnmapMemory(&m_pool, device, memory);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkUnmapMemory(snapshotTraceBegin, snapshotTraceBytes,
                                                       &m_pool, device, memory);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkFlushMappedMemoryRanges: {
                android::base::beginTrace("vkFlushMappedMemoryRanges decode");
                VkDevice device;
                uint32_t memoryRangeCount;
                const VkMappedMemoryRange* pMemoryRanges;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                memcpy((uint32_t*)&memoryRangeCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pMemoryRanges,
                                    ((memoryRangeCount)) * sizeof(const VkMappedMemoryRange));
                for (uint32_t i = 0; i < (uint32_t)((memoryRangeCount)); ++i) {
                    reservedunmarshal_VkMappedMemoryRange(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                          (VkMappedMemoryRange*)(pMemoryRanges + i),
                                                          readStreamPtrPtr);
                }
                if (pMemoryRanges) {
                    for (uint32_t i = 0; i < (uint32_t)((memoryRangeCount)); ++i) {
                        transform_tohost_VkMappedMemoryRange(
                            m_state, (VkMappedMemoryRange*)(pMemoryRanges + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(
                        stderr, "stream %p: call vkFlushMappedMemoryRanges 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)memoryRangeCount,
                        (unsigned long long)pMemoryRanges);
                }
                if (!m_state->usingDirectMapping()) {
                    // This is to deal with a deficiency in the encoder,;
                    // where usingDirectMapping fails to set the proper packet size,;
                    // meaning we can read off the end of the packet.;
                    uint64_t sizeLeft = end - *readStreamPtrPtr;
                    for (uint32_t i = 0; i < memoryRangeCount; ++i) {
                        if (sizeLeft < sizeof(uint64_t)) {
                            if (m_prevSeqno) {
                                m_prevSeqno = m_prevSeqno.value() - 1;
                            }
                            return ptr - (unsigned char*)buf;
                            ;
                        }
                        auto range = pMemoryRanges[i];
                        auto memory = pMemoryRanges[i].memory;
                        auto size = pMemoryRanges[i].size;
                        auto offset = pMemoryRanges[i].offset;
                        uint64_t readStream = 0;
                        memcpy(&readStream, *readStreamPtrPtr, sizeof(uint64_t));
                        *readStreamPtrPtr += sizeof(uint64_t);
                        sizeLeft -= sizeof(uint64_t);
                        auto hostPtr = m_state->getMappedHostPointer(memory);
                        if (!hostPtr && readStream > 0)
                            GFXSTREAM_ABORT(::emugl::FatalError(::emugl::ABORT_REASON_OTHER));
                        if (!hostPtr) continue;
                        if (sizeLeft < readStream) {
                            if (m_prevSeqno) {
                                m_prevSeqno = m_prevSeqno.value() - 1;
                            }
                            return ptr - (unsigned char*)buf;
                            ;
                        }
                        sizeLeft -= readStream;
                        uint8_t* targetRange = hostPtr + offset;
                        memcpy(targetRange, *readStreamPtrPtr, readStream);
                        *readStreamPtrPtr += readStream;
                        packetLen += 8 + readStream;
                    }
                }
                VkResult vkFlushMappedMemoryRanges_VkResult_return = (VkResult)0;
                vkFlushMappedMemoryRanges_VkResult_return =
                    vk->vkFlushMappedMemoryRanges(unboxed_device, memoryRangeCount, pMemoryRanges);
                if ((vkFlushMappedMemoryRanges_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkFlushMappedMemoryRanges_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkFlushMappedMemoryRanges_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkFlushMappedMemoryRanges(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkFlushMappedMemoryRanges_VkResult_return, device, memoryRangeCount,
                        pMemoryRanges);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkInvalidateMappedMemoryRanges: {
                android::base::beginTrace("vkInvalidateMappedMemoryRanges decode");
                VkDevice device;
                uint32_t memoryRangeCount;
                const VkMappedMemoryRange* pMemoryRanges;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                memcpy((uint32_t*)&memoryRangeCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pMemoryRanges,
                                    ((memoryRangeCount)) * sizeof(const VkMappedMemoryRange));
                for (uint32_t i = 0; i < (uint32_t)((memoryRangeCount)); ++i) {
                    reservedunmarshal_VkMappedMemoryRange(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                          (VkMappedMemoryRange*)(pMemoryRanges + i),
                                                          readStreamPtrPtr);
                }
                if (pMemoryRanges) {
                    for (uint32_t i = 0; i < (uint32_t)((memoryRangeCount)); ++i) {
                        transform_tohost_VkMappedMemoryRange(
                            m_state, (VkMappedMemoryRange*)(pMemoryRanges + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkInvalidateMappedMemoryRanges 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)memoryRangeCount,
                        (unsigned long long)pMemoryRanges);
                }
                VkResult vkInvalidateMappedMemoryRanges_VkResult_return = (VkResult)0;
                vkInvalidateMappedMemoryRanges_VkResult_return = vk->vkInvalidateMappedMemoryRanges(
                    unboxed_device, memoryRangeCount, pMemoryRanges);
                if ((vkInvalidateMappedMemoryRanges_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkInvalidateMappedMemoryRanges_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkInvalidateMappedMemoryRanges_VkResult_return, sizeof(VkResult));
                if (!m_state->usingDirectMapping()) {
                    for (uint32_t i = 0; i < memoryRangeCount; ++i) {
                        auto range = pMemoryRanges[i];
                        auto memory = range.memory;
                        auto size = range.size;
                        auto offset = range.offset;
                        auto hostPtr = m_state->getMappedHostPointer(memory);
                        auto actualSize =
                            size == VK_WHOLE_SIZE ? m_state->getDeviceMemorySize(memory) : size;
                        uint64_t writeStream = 0;
                        if (!hostPtr) {
                            vkStream->write(&writeStream, sizeof(uint64_t));
                            continue;
                        };
                        uint8_t* targetRange = hostPtr + offset;
                        writeStream = actualSize;
                        vkStream->write(&writeStream, sizeof(uint64_t));
                        vkStream->write(targetRange, actualSize);
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkInvalidateMappedMemoryRanges(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkInvalidateMappedMemoryRanges_VkResult_return, device, memoryRangeCount,
                        pMemoryRanges);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceMemoryCommitment: {
                android::base::beginTrace("vkGetDeviceMemoryCommitment decode");
                VkDevice device;
                VkDeviceMemory memory;
                VkDeviceSize* pCommittedMemoryInBytes;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDeviceMemory*)&memory =
                    (VkDeviceMemory)unbox_VkDeviceMemory((VkDeviceMemory)(*&cgen_var_1));
                // Begin manual dispatchable handle unboxing for pCommittedMemoryInBytes;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pCommittedMemoryInBytes, sizeof(VkDeviceSize));
                memcpy((VkDeviceSize*)pCommittedMemoryInBytes, *readStreamPtrPtr,
                       sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetDeviceMemoryCommitment 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)memory,
                            (unsigned long long)pCommittedMemoryInBytes);
                }
                vk->vkGetDeviceMemoryCommitment(unboxed_device, memory, pCommittedMemoryInBytes);
                vkStream->unsetHandleMapping();
                vkStream->write((VkDeviceSize*)pCommittedMemoryInBytes, sizeof(VkDeviceSize));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceMemoryCommitment(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, memory,
                        pCommittedMemoryInBytes);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkBindBufferMemory: {
                android::base::beginTrace("vkBindBufferMemory decode");
                VkDevice device;
                VkBuffer buffer;
                VkDeviceMemory memory;
                VkDeviceSize memoryOffset;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&buffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDeviceMemory*)&memory =
                    (VkDeviceMemory)unbox_VkDeviceMemory((VkDeviceMemory)(*&cgen_var_2));
                memcpy((VkDeviceSize*)&memoryOffset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkBindBufferMemory 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)buffer,
                            (unsigned long long)memory, (unsigned long long)memoryOffset);
                }
                VkResult vkBindBufferMemory_VkResult_return = (VkResult)0;
                vkBindBufferMemory_VkResult_return =
                    m_state->on_vkBindBufferMemory(&m_pool, device, buffer, memory, memoryOffset);
                if ((vkBindBufferMemory_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkBindBufferMemory_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkBindBufferMemory_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkBindBufferMemory(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkBindBufferMemory_VkResult_return, device, buffer, memory, memoryOffset);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkBindImageMemory: {
                android::base::beginTrace("vkBindImageMemory decode");
                VkDevice device;
                VkImage image;
                VkDeviceMemory memory;
                VkDeviceSize memoryOffset;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&image = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_1));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDeviceMemory*)&memory =
                    (VkDeviceMemory)unbox_VkDeviceMemory((VkDeviceMemory)(*&cgen_var_2));
                memcpy((VkDeviceSize*)&memoryOffset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkBindImageMemory 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)image,
                            (unsigned long long)memory, (unsigned long long)memoryOffset);
                }
                VkResult vkBindImageMemory_VkResult_return = (VkResult)0;
                vkBindImageMemory_VkResult_return =
                    m_state->on_vkBindImageMemory(&m_pool, device, image, memory, memoryOffset);
                if ((vkBindImageMemory_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkBindImageMemory_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkBindImageMemory_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkBindImageMemory(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkBindImageMemory_VkResult_return, device, image, memory, memoryOffset);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetBufferMemoryRequirements: {
                android::base::beginTrace("vkGetBufferMemoryRequirements decode");
                VkDevice device;
                VkBuffer buffer;
                VkMemoryRequirements* pMemoryRequirements;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&buffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                // Begin manual dispatchable handle unboxing for pMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryRequirements, sizeof(VkMemoryRequirements));
                reservedunmarshal_VkMemoryRequirements(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                       (VkMemoryRequirements*)(pMemoryRequirements),
                                                       readStreamPtrPtr);
                if (pMemoryRequirements) {
                    transform_tohost_VkMemoryRequirements(
                        m_state, (VkMemoryRequirements*)(pMemoryRequirements));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetBufferMemoryRequirements 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)buffer,
                            (unsigned long long)pMemoryRequirements);
                }
                m_state->on_vkGetBufferMemoryRequirements(&m_pool, device, buffer,
                                                          pMemoryRequirements);
                vkStream->unsetHandleMapping();
                if (pMemoryRequirements) {
                    transform_fromhost_VkMemoryRequirements(
                        m_state, (VkMemoryRequirements*)(pMemoryRequirements));
                }
                marshal_VkMemoryRequirements(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                             (VkMemoryRequirements*)(pMemoryRequirements));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetBufferMemoryRequirements(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, buffer,
                        pMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetImageMemoryRequirements: {
                android::base::beginTrace("vkGetImageMemoryRequirements decode");
                VkDevice device;
                VkImage image;
                VkMemoryRequirements* pMemoryRequirements;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&image = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_1));
                // Begin manual dispatchable handle unboxing for pMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryRequirements, sizeof(VkMemoryRequirements));
                reservedunmarshal_VkMemoryRequirements(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                       (VkMemoryRequirements*)(pMemoryRequirements),
                                                       readStreamPtrPtr);
                if (pMemoryRequirements) {
                    transform_tohost_VkMemoryRequirements(
                        m_state, (VkMemoryRequirements*)(pMemoryRequirements));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetImageMemoryRequirements 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)image,
                            (unsigned long long)pMemoryRequirements);
                }
                m_state->on_vkGetImageMemoryRequirements(&m_pool, device, image,
                                                         pMemoryRequirements);
                vkStream->unsetHandleMapping();
                if (pMemoryRequirements) {
                    transform_fromhost_VkMemoryRequirements(
                        m_state, (VkMemoryRequirements*)(pMemoryRequirements));
                }
                marshal_VkMemoryRequirements(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                             (VkMemoryRequirements*)(pMemoryRequirements));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetImageMemoryRequirements(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, image,
                        pMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetImageSparseMemoryRequirements: {
                android::base::beginTrace("vkGetImageSparseMemoryRequirements decode");
                VkDevice device;
                VkImage image;
                uint32_t* pSparseMemoryRequirementCount;
                VkSparseImageMemoryRequirements* pSparseMemoryRequirements;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&image = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_1));
                // Begin manual dispatchable handle unboxing for pSparseMemoryRequirementCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pSparseMemoryRequirementCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSparseMemoryRequirementCount);
                *readStreamPtrPtr += 8;
                if (pSparseMemoryRequirementCount) {
                    vkReadStream->alloc((void**)&pSparseMemoryRequirementCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pSparseMemoryRequirementCount, *readStreamPtrPtr,
                           sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pSparseMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkSparseImageMemoryRequirements**)&pSparseMemoryRequirements,
                       (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSparseMemoryRequirements);
                *readStreamPtrPtr += 8;
                if (pSparseMemoryRequirements) {
                    vkReadStream->alloc((void**)&pSparseMemoryRequirements,
                                        (*(pSparseMemoryRequirementCount)) *
                                            sizeof(VkSparseImageMemoryRequirements));
                    for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount)); ++i) {
                        reservedunmarshal_VkSparseImageMemoryRequirements(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkSparseImageMemoryRequirements*)(pSparseMemoryRequirements + i),
                            readStreamPtrPtr);
                    }
                }
                if (pSparseMemoryRequirementCount) {
                    if (pSparseMemoryRequirements) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            transform_tohost_VkSparseImageMemoryRequirements(
                                m_state,
                                (VkSparseImageMemoryRequirements*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetImageSparseMemoryRequirements 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)image,
                            (unsigned long long)pSparseMemoryRequirementCount,
                            (unsigned long long)pSparseMemoryRequirements);
                }
                vk->vkGetImageSparseMemoryRequirements(unboxed_device, image,
                                                       pSparseMemoryRequirementCount,
                                                       pSparseMemoryRequirements);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pSparseMemoryRequirementCount;
                vkStream->putBe64(cgen_var_4);
                if (pSparseMemoryRequirementCount) {
                    vkStream->write((uint32_t*)pSparseMemoryRequirementCount, sizeof(uint32_t));
                }
                if (pSparseMemoryRequirementCount) {
                    if (pSparseMemoryRequirements) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            transform_fromhost_VkSparseImageMemoryRequirements(
                                m_state,
                                (VkSparseImageMemoryRequirements*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_5 = (uint64_t)(uintptr_t)pSparseMemoryRequirements;
                vkStream->putBe64(cgen_var_5);
                if (pSparseMemoryRequirements) {
                    if (pSparseMemoryRequirementCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            marshal_VkSparseImageMemoryRequirements(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkSparseImageMemoryRequirements*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetImageSparseMemoryRequirements(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, image,
                        pSparseMemoryRequirementCount, pSparseMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceSparseImageFormatProperties: {
                android::base::beginTrace("vkGetPhysicalDeviceSparseImageFormatProperties decode");
                VkPhysicalDevice physicalDevice;
                VkFormat format;
                VkImageType type;
                VkSampleCountFlagBits samples;
                VkImageUsageFlags usage;
                VkImageTiling tiling;
                uint32_t* pPropertyCount;
                VkSparseImageFormatProperties* pProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                memcpy((VkFormat*)&format, *readStreamPtrPtr, sizeof(VkFormat));
                *readStreamPtrPtr += sizeof(VkFormat);
                memcpy((VkImageType*)&type, *readStreamPtrPtr, sizeof(VkImageType));
                *readStreamPtrPtr += sizeof(VkImageType);
                memcpy((VkSampleCountFlagBits*)&samples, *readStreamPtrPtr,
                       sizeof(VkSampleCountFlagBits));
                *readStreamPtrPtr += sizeof(VkSampleCountFlagBits);
                memcpy((VkImageUsageFlags*)&usage, *readStreamPtrPtr, sizeof(VkImageUsageFlags));
                *readStreamPtrPtr += sizeof(VkImageUsageFlags);
                memcpy((VkImageTiling*)&tiling, *readStreamPtrPtr, sizeof(VkImageTiling));
                *readStreamPtrPtr += sizeof(VkImageTiling);
                // Begin manual dispatchable handle unboxing for pPropertyCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pPropertyCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPropertyCount);
                *readStreamPtrPtr += 8;
                if (pPropertyCount) {
                    vkReadStream->alloc((void**)&pPropertyCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pPropertyCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkSparseImageFormatProperties**)&pProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pProperties);
                *readStreamPtrPtr += 8;
                if (pProperties) {
                    vkReadStream->alloc(
                        (void**)&pProperties,
                        (*(pPropertyCount)) * sizeof(VkSparseImageFormatProperties));
                    for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                        reservedunmarshal_VkSparseImageFormatProperties(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkSparseImageFormatProperties*)(pProperties + i), readStreamPtrPtr);
                    }
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_tohost_VkSparseImageFormatProperties(
                                m_state, (VkSparseImageFormatProperties*)(pProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceSparseImageFormatProperties 0x%llx "
                            "0x%llx 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)format, (unsigned long long)type,
                            (unsigned long long)samples, (unsigned long long)usage,
                            (unsigned long long)tiling, (unsigned long long)pPropertyCount,
                            (unsigned long long)pProperties);
                }
                vk->vkGetPhysicalDeviceSparseImageFormatProperties(unboxed_physicalDevice, format,
                                                                   type, samples, usage, tiling,
                                                                   pPropertyCount, pProperties);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pPropertyCount;
                vkStream->putBe64(cgen_var_3);
                if (pPropertyCount) {
                    vkStream->write((uint32_t*)pPropertyCount, sizeof(uint32_t));
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_fromhost_VkSparseImageFormatProperties(
                                m_state, (VkSparseImageFormatProperties*)(pProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pProperties;
                vkStream->putBe64(cgen_var_4);
                if (pProperties) {
                    if (pPropertyCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            marshal_VkSparseImageFormatProperties(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkSparseImageFormatProperties*)(pProperties + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceSparseImageFormatProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice, format,
                        type, samples, usage, tiling, pPropertyCount, pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkQueueBindSparse: {
                android::base::beginTrace("vkQueueBindSparse decode");
                VkQueue queue;
                uint32_t bindInfoCount;
                const VkBindSparseInfo* pBindInfo;
                VkFence fence;
                // Begin global wrapped dispatchable handle unboxing for queue;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueue*)&queue = (VkQueue)(VkQueue)((VkQueue)(*&cgen_var_0));
                memcpy((uint32_t*)&bindInfoCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pBindInfo,
                                    ((bindInfoCount)) * sizeof(const VkBindSparseInfo));
                for (uint32_t i = 0; i < (uint32_t)((bindInfoCount)); ++i) {
                    reservedunmarshal_VkBindSparseInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                       (VkBindSparseInfo*)(pBindInfo + i),
                                                       readStreamPtrPtr);
                }
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkFence*)&fence = (VkFence)unbox_VkFence((VkFence)(*&cgen_var_1));
                if (pBindInfo) {
                    for (uint32_t i = 0; i < (uint32_t)((bindInfoCount)); ++i) {
                        transform_tohost_VkBindSparseInfo(m_state,
                                                          (VkBindSparseInfo*)(pBindInfo + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkQueueBindSparse 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)queue, (unsigned long long)bindInfoCount,
                            (unsigned long long)pBindInfo, (unsigned long long)fence);
                }
                VkResult vkQueueBindSparse_VkResult_return = (VkResult)0;
                vkQueueBindSparse_VkResult_return =
                    m_state->on_vkQueueBindSparse(&m_pool, queue, bindInfoCount, pBindInfo, fence);
                if ((vkQueueBindSparse_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkQueueBindSparse_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkQueueBindSparse_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkQueueBindSparse(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkQueueBindSparse_VkResult_return, queue, bindInfoCount, pBindInfo, fence);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateFence: {
                android::base::beginTrace("vkCreateFence decode");
                VkDevice device;
                const VkFenceCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkFence* pFence;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkFenceCreateInfo));
                reservedunmarshal_VkFenceCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkFenceCreateInfo*)(pCreateInfo),
                                                    readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pFence;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pFence, sizeof(VkFence));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkFence*)pFence = (VkFence)(VkFence)((VkFence)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkFenceCreateInfo(m_state, (VkFenceCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCreateFence 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pFence);
                }
                VkResult vkCreateFence_VkResult_return = (VkResult)0;
                vkCreateFence_VkResult_return =
                    m_state->on_vkCreateFence(&m_pool, device, pCreateInfo, pAllocator, pFence);
                if ((vkCreateFence_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateFence_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pFence;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkFence),
                              "handle map overwrite requires VkFence to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkFence((VkFence*)pFence, 1);
                vkStream->write((VkFence*)pFence, 8 * 1);
                // Begin manual non dispatchable handle create for pFence;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateFence_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateFence(snapshotTraceBegin, snapshotTraceBytes,
                                                       &m_pool, vkCreateFence_VkResult_return,
                                                       device, pCreateInfo, pAllocator, pFence);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyFence: {
                android::base::beginTrace("vkDestroyFence decode");
                VkDevice device;
                VkFence fence;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for fence;
                VkFence boxed_fence_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkFence*)&fence = (VkFence)(VkFence)((VkFence)(*&cgen_var_1));
                boxed_fence_preserve = fence;
                fence = unbox_VkFence(fence);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyFence 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)fence,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyFence(&m_pool, device, fence, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyFence(snapshotTraceBegin, snapshotTraceBytes,
                                                        &m_pool, device, boxed_fence_preserve,
                                                        pAllocator);
                }
                delete_VkFence(boxed_fence_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkResetFences: {
                android::base::beginTrace("vkResetFences decode");
                VkDevice device;
                uint32_t fenceCount;
                const VkFence* pFences;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                memcpy((uint32_t*)&fenceCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pFences, ((fenceCount)) * sizeof(const VkFence));
                if (((fenceCount))) {
                    uint8_t* cgen_var_1_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((fenceCount));
                    for (uint32_t k = 0; k < ((fenceCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_1_ptr + k * 8, sizeof(uint64_t));
                        *(((VkFence*)pFences) + k) = (VkFence)unbox_VkFence((VkFence)tmpval);
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkResetFences 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)fenceCount,
                            (unsigned long long)pFences);
                }
                VkResult vkResetFences_VkResult_return = (VkResult)0;
                vkResetFences_VkResult_return =
                    m_state->on_vkResetFences(&m_pool, device, fenceCount, pFences);
                if ((vkResetFences_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkResetFences_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkResetFences_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkResetFences(snapshotTraceBegin, snapshotTraceBytes,
                                                       &m_pool, vkResetFences_VkResult_return,
                                                       device, fenceCount, pFences);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetFenceStatus: {
                android::base::beginTrace("vkGetFenceStatus decode");
                VkDevice device;
                VkFence fence;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkFence*)&fence = (VkFence)unbox_VkFence((VkFence)(*&cgen_var_1));
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetFenceStatus 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)device, (unsigned long long)fence);
                }
                VkResult vkGetFenceStatus_VkResult_return = (VkResult)0;
                vkGetFenceStatus_VkResult_return = vk->vkGetFenceStatus(unboxed_device, fence);
                if ((vkGetFenceStatus_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkGetFenceStatus_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkGetFenceStatus_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetFenceStatus(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, vkGetFenceStatus_VkResult_return,
                                                          device, fence);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkWaitForFences: {
                android::base::beginTrace("vkWaitForFences decode");
                VkDevice device;
                uint32_t fenceCount;
                const VkFence* pFences;
                VkBool32 waitAll;
                uint64_t timeout;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                memcpy((uint32_t*)&fenceCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pFences, ((fenceCount)) * sizeof(const VkFence));
                if (((fenceCount))) {
                    uint8_t* cgen_var_1_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((fenceCount));
                    for (uint32_t k = 0; k < ((fenceCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_1_ptr + k * 8, sizeof(uint64_t));
                        *(((VkFence*)pFences) + k) = (VkFence)unbox_VkFence((VkFence)tmpval);
                    }
                }
                memcpy((VkBool32*)&waitAll, *readStreamPtrPtr, sizeof(VkBool32));
                *readStreamPtrPtr += sizeof(VkBool32);
                memcpy((uint64_t*)&timeout, *readStreamPtrPtr, sizeof(uint64_t));
                *readStreamPtrPtr += sizeof(uint64_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkWaitForFences 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)fenceCount,
                            (unsigned long long)pFences, (unsigned long long)waitAll,
                            (unsigned long long)timeout);
                }
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                VkResult vkWaitForFences_VkResult_return = (VkResult)0;
                vkWaitForFences_VkResult_return =
                    vk->vkWaitForFences(unboxed_device, fenceCount, pFences, waitAll, timeout);
                if ((vkWaitForFences_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkWaitForFences_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkWaitForFences_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkWaitForFences(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, vkWaitForFences_VkResult_return,
                                                         device, fenceCount, pFences, waitAll,
                                                         timeout);
                }
                vkReadStream->clearPool();
                android::base::endTrace();
                break;
            }
            case OP_vkCreateSemaphore: {
                android::base::beginTrace("vkCreateSemaphore decode");
                VkDevice device;
                const VkSemaphoreCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkSemaphore* pSemaphore;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkSemaphoreCreateInfo));
                reservedunmarshal_VkSemaphoreCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                        (VkSemaphoreCreateInfo*)(pCreateInfo),
                                                        readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pSemaphore;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pSemaphore, sizeof(VkSemaphore));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkSemaphore*)pSemaphore = (VkSemaphore)(VkSemaphore)((VkSemaphore)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkSemaphoreCreateInfo(m_state,
                                                           (VkSemaphoreCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateSemaphore 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pSemaphore);
                }
                VkResult vkCreateSemaphore_VkResult_return = (VkResult)0;
                vkCreateSemaphore_VkResult_return = m_state->on_vkCreateSemaphore(
                    &m_pool, device, pCreateInfo, pAllocator, pSemaphore);
                if ((vkCreateSemaphore_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateSemaphore_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pSemaphore;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkSemaphore),
                              "handle map overwrite requires VkSemaphore to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkSemaphore((VkSemaphore*)pSemaphore, 1);
                vkStream->write((VkSemaphore*)pSemaphore, 8 * 1);
                // Begin manual non dispatchable handle create for pSemaphore;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateSemaphore_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateSemaphore(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateSemaphore_VkResult_return, device, pCreateInfo, pAllocator,
                        pSemaphore);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroySemaphore: {
                android::base::beginTrace("vkDestroySemaphore decode");
                VkDevice device;
                VkSemaphore semaphore;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for semaphore;
                VkSemaphore boxed_semaphore_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkSemaphore*)&semaphore = (VkSemaphore)(VkSemaphore)((VkSemaphore)(*&cgen_var_1));
                boxed_semaphore_preserve = semaphore;
                semaphore = unbox_VkSemaphore(semaphore);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroySemaphore 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)semaphore,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroySemaphore(&m_pool, device, semaphore, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroySemaphore(snapshotTraceBegin, snapshotTraceBytes,
                                                            &m_pool, device,
                                                            boxed_semaphore_preserve, pAllocator);
                }
                delete_VkSemaphore(boxed_semaphore_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateEvent: {
                android::base::beginTrace("vkCreateEvent decode");
                VkDevice device;
                const VkEventCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkEvent* pEvent;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkEventCreateInfo));
                reservedunmarshal_VkEventCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkEventCreateInfo*)(pCreateInfo),
                                                    readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pEvent;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pEvent, sizeof(VkEvent));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkEvent*)pEvent = (VkEvent)(VkEvent)((VkEvent)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkEventCreateInfo(m_state, (VkEventCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCreateEvent 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pEvent);
                }
                VkResult vkCreateEvent_VkResult_return = (VkResult)0;
                vkCreateEvent_VkResult_return =
                    vk->vkCreateEvent(unboxed_device, pCreateInfo, pAllocator, pEvent);
                if ((vkCreateEvent_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateEvent_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin auto non dispatchable handle create for pEvent;
                if (vkCreateEvent_VkResult_return == VK_SUCCESS)
                    vkStream->setHandleMapping(&m_boxedHandleCreateMapping);
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkEvent),
                              "handle map overwrite requires VkEvent to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkEvent((VkEvent*)pEvent, 1);
                vkStream->write((VkEvent*)pEvent, 8 * 1);
                // Begin auto non dispatchable handle create for pEvent;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateEvent_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateEvent(snapshotTraceBegin, snapshotTraceBytes,
                                                       &m_pool, vkCreateEvent_VkResult_return,
                                                       device, pCreateInfo, pAllocator, pEvent);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyEvent: {
                android::base::beginTrace("vkDestroyEvent decode");
                VkDevice device;
                VkEvent event;
                const VkAllocationCallbacks* pAllocator;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                // Begin manual non dispatchable handle destroy unboxing for event;
                VkEvent boxed_event_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkEvent*)&event = (VkEvent)(VkEvent)((VkEvent)(*&cgen_var_1));
                boxed_event_preserve = event;
                event = unbox_VkEvent(event);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyEvent 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)event,
                            (unsigned long long)pAllocator);
                }
                vk->vkDestroyEvent(unboxed_device, event, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyEvent(snapshotTraceBegin, snapshotTraceBytes,
                                                        &m_pool, device, boxed_event_preserve,
                                                        pAllocator);
                }
                delete_VkEvent(boxed_event_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetEventStatus: {
                android::base::beginTrace("vkGetEventStatus decode");
                VkDevice device;
                VkEvent event;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkEvent*)&event = (VkEvent)unbox_VkEvent((VkEvent)(*&cgen_var_1));
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetEventStatus 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)device, (unsigned long long)event);
                }
                VkResult vkGetEventStatus_VkResult_return = (VkResult)0;
                vkGetEventStatus_VkResult_return = vk->vkGetEventStatus(unboxed_device, event);
                if ((vkGetEventStatus_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkGetEventStatus_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkGetEventStatus_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetEventStatus(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, vkGetEventStatus_VkResult_return,
                                                          device, event);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkSetEvent: {
                android::base::beginTrace("vkSetEvent decode");
                VkDevice device;
                VkEvent event;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkEvent*)&event = (VkEvent)unbox_VkEvent((VkEvent)(*&cgen_var_1));
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkSetEvent 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)device, (unsigned long long)event);
                }
                VkResult vkSetEvent_VkResult_return = (VkResult)0;
                vkSetEvent_VkResult_return = vk->vkSetEvent(unboxed_device, event);
                if ((vkSetEvent_VkResult_return) == VK_ERROR_DEVICE_LOST) m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkSetEvent_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkSetEvent_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkSetEvent(snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                                                    vkSetEvent_VkResult_return, device, event);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkResetEvent: {
                android::base::beginTrace("vkResetEvent decode");
                VkDevice device;
                VkEvent event;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkEvent*)&event = (VkEvent)unbox_VkEvent((VkEvent)(*&cgen_var_1));
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkResetEvent 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)device, (unsigned long long)event);
                }
                VkResult vkResetEvent_VkResult_return = (VkResult)0;
                vkResetEvent_VkResult_return = vk->vkResetEvent(unboxed_device, event);
                if ((vkResetEvent_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkResetEvent_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkResetEvent_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkResetEvent(snapshotTraceBegin, snapshotTraceBytes,
                                                      &m_pool, vkResetEvent_VkResult_return, device,
                                                      event);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateQueryPool: {
                android::base::beginTrace("vkCreateQueryPool decode");
                VkDevice device;
                const VkQueryPoolCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkQueryPool* pQueryPool;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkQueryPoolCreateInfo));
                reservedunmarshal_VkQueryPoolCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                        (VkQueryPoolCreateInfo*)(pCreateInfo),
                                                        readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pQueryPool;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pQueryPool, sizeof(VkQueryPool));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkQueryPool*)pQueryPool = (VkQueryPool)(VkQueryPool)((VkQueryPool)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkQueryPoolCreateInfo(m_state,
                                                           (VkQueryPoolCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateQueryPool 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pQueryPool);
                }
                VkResult vkCreateQueryPool_VkResult_return = (VkResult)0;
                vkCreateQueryPool_VkResult_return =
                    vk->vkCreateQueryPool(unboxed_device, pCreateInfo, pAllocator, pQueryPool);
                if ((vkCreateQueryPool_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateQueryPool_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin auto non dispatchable handle create for pQueryPool;
                if (vkCreateQueryPool_VkResult_return == VK_SUCCESS)
                    vkStream->setHandleMapping(&m_boxedHandleCreateMapping);
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkQueryPool),
                              "handle map overwrite requires VkQueryPool to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkQueryPool((VkQueryPool*)pQueryPool, 1);
                vkStream->write((VkQueryPool*)pQueryPool, 8 * 1);
                // Begin auto non dispatchable handle create for pQueryPool;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateQueryPool_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateQueryPool(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateQueryPool_VkResult_return, device, pCreateInfo, pAllocator,
                        pQueryPool);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyQueryPool: {
                android::base::beginTrace("vkDestroyQueryPool decode");
                VkDevice device;
                VkQueryPool queryPool;
                const VkAllocationCallbacks* pAllocator;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                // Begin manual non dispatchable handle destroy unboxing for queryPool;
                VkQueryPool boxed_queryPool_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueryPool*)&queryPool = (VkQueryPool)(VkQueryPool)((VkQueryPool)(*&cgen_var_1));
                boxed_queryPool_preserve = queryPool;
                queryPool = unbox_VkQueryPool(queryPool);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyQueryPool 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)queryPool,
                            (unsigned long long)pAllocator);
                }
                vk->vkDestroyQueryPool(unboxed_device, queryPool, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyQueryPool(snapshotTraceBegin, snapshotTraceBytes,
                                                            &m_pool, device,
                                                            boxed_queryPool_preserve, pAllocator);
                }
                delete_VkQueryPool(boxed_queryPool_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetQueryPoolResults: {
                android::base::beginTrace("vkGetQueryPoolResults decode");
                VkDevice device;
                VkQueryPool queryPool;
                uint32_t firstQuery;
                uint32_t queryCount;
                size_t dataSize;
                void* pData;
                VkDeviceSize stride;
                VkQueryResultFlags flags;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueryPool*)&queryPool =
                    (VkQueryPool)unbox_VkQueryPool((VkQueryPool)(*&cgen_var_1));
                memcpy((uint32_t*)&firstQuery, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&queryCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((size_t*)&dataSize, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&dataSize);
                *readStreamPtrPtr += 8;
                // Begin manual dispatchable handle unboxing for pData;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pData, ((dataSize)) * sizeof(uint8_t));
                memcpy((void*)pData, *readStreamPtrPtr, ((dataSize)) * sizeof(uint8_t));
                *readStreamPtrPtr += ((dataSize)) * sizeof(uint8_t);
                memcpy((VkDeviceSize*)&stride, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((VkQueryResultFlags*)&flags, *readStreamPtrPtr, sizeof(VkQueryResultFlags));
                *readStreamPtrPtr += sizeof(VkQueryResultFlags);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetQueryPoolResults 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)queryPool,
                            (unsigned long long)firstQuery, (unsigned long long)queryCount,
                            (unsigned long long)dataSize, (unsigned long long)pData,
                            (unsigned long long)stride, (unsigned long long)flags);
                }
                VkResult vkGetQueryPoolResults_VkResult_return = (VkResult)0;
                vkGetQueryPoolResults_VkResult_return =
                    vk->vkGetQueryPoolResults(unboxed_device, queryPool, firstQuery, queryCount,
                                              dataSize, pData, stride, flags);
                if ((vkGetQueryPoolResults_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkGetQueryPoolResults_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                vkStream->write((void*)pData, ((dataSize)) * sizeof(uint8_t));
                vkStream->write(&vkGetQueryPoolResults_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetQueryPoolResults(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetQueryPoolResults_VkResult_return, device, queryPool, firstQuery,
                        queryCount, dataSize, pData, stride, flags);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateBuffer: {
                android::base::beginTrace("vkCreateBuffer decode");
                VkDevice device;
                const VkBufferCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkBuffer* pBuffer;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkBufferCreateInfo));
                reservedunmarshal_VkBufferCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkBufferCreateInfo*)(pCreateInfo),
                                                     readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pBuffer;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pBuffer, sizeof(VkBuffer));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkBuffer*)pBuffer = (VkBuffer)(VkBuffer)((VkBuffer)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkBufferCreateInfo(m_state,
                                                        (VkBufferCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCreateBuffer 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pBuffer);
                }
                VkResult vkCreateBuffer_VkResult_return = (VkResult)0;
                vkCreateBuffer_VkResult_return =
                    m_state->on_vkCreateBuffer(&m_pool, device, pCreateInfo, pAllocator, pBuffer);
                if ((vkCreateBuffer_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateBuffer_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pBuffer;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkBuffer),
                              "handle map overwrite requires VkBuffer to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkBuffer((VkBuffer*)pBuffer, 1);
                vkStream->write((VkBuffer*)pBuffer, 8 * 1);
                // Begin manual non dispatchable handle create for pBuffer;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateBuffer_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateBuffer(snapshotTraceBegin, snapshotTraceBytes,
                                                        &m_pool, vkCreateBuffer_VkResult_return,
                                                        device, pCreateInfo, pAllocator, pBuffer);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyBuffer: {
                android::base::beginTrace("vkDestroyBuffer decode");
                VkDevice device;
                VkBuffer buffer;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for buffer;
                VkBuffer boxed_buffer_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&buffer = (VkBuffer)(VkBuffer)((VkBuffer)(*&cgen_var_1));
                boxed_buffer_preserve = buffer;
                buffer = unbox_VkBuffer(buffer);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyBuffer 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)buffer,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyBuffer(&m_pool, device, buffer, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyBuffer(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, device, boxed_buffer_preserve,
                                                         pAllocator);
                }
                delete_VkBuffer(boxed_buffer_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateBufferView: {
                android::base::beginTrace("vkCreateBufferView decode");
                VkDevice device;
                const VkBufferViewCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkBufferView* pView;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkBufferViewCreateInfo));
                reservedunmarshal_VkBufferViewCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                         (VkBufferViewCreateInfo*)(pCreateInfo),
                                                         readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pView;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pView, sizeof(VkBufferView));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkBufferView*)pView = (VkBufferView)(VkBufferView)((VkBufferView)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkBufferViewCreateInfo(m_state,
                                                            (VkBufferViewCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateBufferView 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pView);
                }
                VkResult vkCreateBufferView_VkResult_return = (VkResult)0;
                vkCreateBufferView_VkResult_return =
                    vk->vkCreateBufferView(unboxed_device, pCreateInfo, pAllocator, pView);
                if ((vkCreateBufferView_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateBufferView_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin auto non dispatchable handle create for pView;
                if (vkCreateBufferView_VkResult_return == VK_SUCCESS)
                    vkStream->setHandleMapping(&m_boxedHandleCreateMapping);
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkBufferView),
                              "handle map overwrite requires VkBufferView to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkBufferView((VkBufferView*)pView, 1);
                vkStream->write((VkBufferView*)pView, 8 * 1);
                // Begin auto non dispatchable handle create for pView;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateBufferView_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateBufferView(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateBufferView_VkResult_return, device, pCreateInfo, pAllocator, pView);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyBufferView: {
                android::base::beginTrace("vkDestroyBufferView decode");
                VkDevice device;
                VkBufferView bufferView;
                const VkAllocationCallbacks* pAllocator;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                // Begin manual non dispatchable handle destroy unboxing for bufferView;
                VkBufferView boxed_bufferView_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBufferView*)&bufferView =
                    (VkBufferView)(VkBufferView)((VkBufferView)(*&cgen_var_1));
                boxed_bufferView_preserve = bufferView;
                bufferView = unbox_VkBufferView(bufferView);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyBufferView 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)bufferView,
                            (unsigned long long)pAllocator);
                }
                vk->vkDestroyBufferView(unboxed_device, bufferView, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyBufferView(snapshotTraceBegin, snapshotTraceBytes,
                                                             &m_pool, device,
                                                             boxed_bufferView_preserve, pAllocator);
                }
                delete_VkBufferView(boxed_bufferView_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateImage: {
                android::base::beginTrace("vkCreateImage decode");
                VkDevice device;
                const VkImageCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkImage* pImage;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkImageCreateInfo));
                reservedunmarshal_VkImageCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkImageCreateInfo*)(pCreateInfo),
                                                    readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pImage;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pImage, sizeof(VkImage));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkImage*)pImage = (VkImage)(VkImage)((VkImage)(*&cgen_var_2));
                if (pCreateInfo) {
                    m_state->transformImpl_VkImageCreateInfo_tohost(pCreateInfo, 1);
                    transform_tohost_VkImageCreateInfo(m_state, (VkImageCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCreateImage 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pImage);
                }
                VkResult vkCreateImage_VkResult_return = (VkResult)0;
                vkCreateImage_VkResult_return =
                    m_state->on_vkCreateImage(&m_pool, device, pCreateInfo, pAllocator, pImage);
                if ((vkCreateImage_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateImage_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pImage;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkImage),
                              "handle map overwrite requires VkImage to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkImage((VkImage*)pImage, 1);
                vkStream->write((VkImage*)pImage, 8 * 1);
                // Begin manual non dispatchable handle create for pImage;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateImage_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateImage(snapshotTraceBegin, snapshotTraceBytes,
                                                       &m_pool, vkCreateImage_VkResult_return,
                                                       device, pCreateInfo, pAllocator, pImage);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyImage: {
                android::base::beginTrace("vkDestroyImage decode");
                VkDevice device;
                VkImage image;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for image;
                VkImage boxed_image_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&image = (VkImage)(VkImage)((VkImage)(*&cgen_var_1));
                boxed_image_preserve = image;
                image = unbox_VkImage(image);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyImage 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)image,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyImage(&m_pool, device, image, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyImage(snapshotTraceBegin, snapshotTraceBytes,
                                                        &m_pool, device, boxed_image_preserve,
                                                        pAllocator);
                }
                delete_VkImage(boxed_image_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetImageSubresourceLayout: {
                android::base::beginTrace("vkGetImageSubresourceLayout decode");
                VkDevice device;
                VkImage image;
                const VkImageSubresource* pSubresource;
                VkSubresourceLayout* pLayout;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&image = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_1));
                vkReadStream->alloc((void**)&pSubresource, sizeof(const VkImageSubresource));
                reservedunmarshal_VkImageSubresource(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkImageSubresource*)(pSubresource),
                                                     readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pLayout;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pLayout, sizeof(VkSubresourceLayout));
                reservedunmarshal_VkSubresourceLayout(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkSubresourceLayout*)(pLayout),
                                                      readStreamPtrPtr);
                if (pSubresource) {
                    transform_tohost_VkImageSubresource(m_state,
                                                        (VkImageSubresource*)(pSubresource));
                }
                if (pLayout) {
                    transform_tohost_VkSubresourceLayout(m_state, (VkSubresourceLayout*)(pLayout));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetImageSubresourceLayout 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)image,
                            (unsigned long long)pSubresource, (unsigned long long)pLayout);
                }
                vk->vkGetImageSubresourceLayout(unboxed_device, image, pSubresource, pLayout);
                vkStream->unsetHandleMapping();
                if (pLayout) {
                    transform_fromhost_VkSubresourceLayout(m_state,
                                                           (VkSubresourceLayout*)(pLayout));
                }
                marshal_VkSubresourceLayout(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                            (VkSubresourceLayout*)(pLayout));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetImageSubresourceLayout(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, image,
                        pSubresource, pLayout);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateImageView: {
                android::base::beginTrace("vkCreateImageView decode");
                VkDevice device;
                const VkImageViewCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkImageView* pView;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkImageViewCreateInfo));
                reservedunmarshal_VkImageViewCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                        (VkImageViewCreateInfo*)(pCreateInfo),
                                                        readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pView;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pView, sizeof(VkImageView));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkImageView*)pView = (VkImageView)(VkImageView)((VkImageView)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkImageViewCreateInfo(m_state,
                                                           (VkImageViewCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateImageView 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pView);
                }
                VkResult vkCreateImageView_VkResult_return = (VkResult)0;
                vkCreateImageView_VkResult_return =
                    m_state->on_vkCreateImageView(&m_pool, device, pCreateInfo, pAllocator, pView);
                if ((vkCreateImageView_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateImageView_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pView;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkImageView),
                              "handle map overwrite requires VkImageView to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkImageView((VkImageView*)pView, 1);
                vkStream->write((VkImageView*)pView, 8 * 1);
                // Begin manual non dispatchable handle create for pView;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateImageView_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateImageView(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateImageView_VkResult_return, device, pCreateInfo, pAllocator, pView);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyImageView: {
                android::base::beginTrace("vkDestroyImageView decode");
                VkDevice device;
                VkImageView imageView;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for imageView;
                VkImageView boxed_imageView_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImageView*)&imageView = (VkImageView)(VkImageView)((VkImageView)(*&cgen_var_1));
                boxed_imageView_preserve = imageView;
                imageView = unbox_VkImageView(imageView);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyImageView 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)imageView,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyImageView(&m_pool, device, imageView, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyImageView(snapshotTraceBegin, snapshotTraceBytes,
                                                            &m_pool, device,
                                                            boxed_imageView_preserve, pAllocator);
                }
                delete_VkImageView(boxed_imageView_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateShaderModule: {
                android::base::beginTrace("vkCreateShaderModule decode");
                VkDevice device;
                const VkShaderModuleCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkShaderModule* pShaderModule;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkShaderModuleCreateInfo));
                reservedunmarshal_VkShaderModuleCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                           (VkShaderModuleCreateInfo*)(pCreateInfo),
                                                           readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pShaderModule;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pShaderModule, sizeof(VkShaderModule));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkShaderModule*)pShaderModule =
                    (VkShaderModule)(VkShaderModule)((VkShaderModule)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkShaderModuleCreateInfo(
                        m_state, (VkShaderModuleCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateShaderModule 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pShaderModule);
                }
                VkResult vkCreateShaderModule_VkResult_return = (VkResult)0;
                vkCreateShaderModule_VkResult_return = m_state->on_vkCreateShaderModule(
                    &m_pool, device, pCreateInfo, pAllocator, pShaderModule);
                if ((vkCreateShaderModule_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateShaderModule_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pShaderModule;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkShaderModule),
                              "handle map overwrite requires VkShaderModule to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkShaderModule((VkShaderModule*)pShaderModule,
                                                                     1);
                vkStream->write((VkShaderModule*)pShaderModule, 8 * 1);
                // Begin manual non dispatchable handle create for pShaderModule;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateShaderModule_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateShaderModule(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateShaderModule_VkResult_return, device, pCreateInfo, pAllocator,
                        pShaderModule);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyShaderModule: {
                android::base::beginTrace("vkDestroyShaderModule decode");
                VkDevice device;
                VkShaderModule shaderModule;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for shaderModule;
                VkShaderModule boxed_shaderModule_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkShaderModule*)&shaderModule =
                    (VkShaderModule)(VkShaderModule)((VkShaderModule)(*&cgen_var_1));
                boxed_shaderModule_preserve = shaderModule;
                shaderModule = unbox_VkShaderModule(shaderModule);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyShaderModule 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)shaderModule,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyShaderModule(&m_pool, device, shaderModule, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyShaderModule(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_shaderModule_preserve, pAllocator);
                }
                delete_VkShaderModule(boxed_shaderModule_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreatePipelineCache: {
                android::base::beginTrace("vkCreatePipelineCache decode");
                VkDevice device;
                const VkPipelineCacheCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkPipelineCache* pPipelineCache;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkPipelineCacheCreateInfo));
                reservedunmarshal_VkPipelineCacheCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPipelineCacheCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pPipelineCache;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pPipelineCache, sizeof(VkPipelineCache));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkPipelineCache*)pPipelineCache =
                    (VkPipelineCache)(VkPipelineCache)((VkPipelineCache)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkPipelineCacheCreateInfo(
                        m_state, (VkPipelineCacheCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreatePipelineCache 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pPipelineCache);
                }
                VkResult vkCreatePipelineCache_VkResult_return = (VkResult)0;
                vkCreatePipelineCache_VkResult_return = m_state->on_vkCreatePipelineCache(
                    &m_pool, device, pCreateInfo, pAllocator, pPipelineCache);
                if ((vkCreatePipelineCache_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreatePipelineCache_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pPipelineCache;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkPipelineCache),
                              "handle map overwrite requires VkPipelineCache to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkPipelineCache(
                    (VkPipelineCache*)pPipelineCache, 1);
                vkStream->write((VkPipelineCache*)pPipelineCache, 8 * 1);
                // Begin manual non dispatchable handle create for pPipelineCache;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreatePipelineCache_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreatePipelineCache(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreatePipelineCache_VkResult_return, device, pCreateInfo, pAllocator,
                        pPipelineCache);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyPipelineCache: {
                android::base::beginTrace("vkDestroyPipelineCache decode");
                VkDevice device;
                VkPipelineCache pipelineCache;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for pipelineCache;
                VkPipelineCache boxed_pipelineCache_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPipelineCache*)&pipelineCache =
                    (VkPipelineCache)(VkPipelineCache)((VkPipelineCache)(*&cgen_var_1));
                boxed_pipelineCache_preserve = pipelineCache;
                pipelineCache = unbox_VkPipelineCache(pipelineCache);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkDestroyPipelineCache 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pipelineCache,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyPipelineCache(&m_pool, device, pipelineCache, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyPipelineCache(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_pipelineCache_preserve, pAllocator);
                }
                delete_VkPipelineCache(boxed_pipelineCache_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPipelineCacheData: {
                android::base::beginTrace("vkGetPipelineCacheData decode");
                VkDevice device;
                VkPipelineCache pipelineCache;
                size_t* pDataSize;
                void* pData;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPipelineCache*)&pipelineCache =
                    (VkPipelineCache)unbox_VkPipelineCache((VkPipelineCache)(*&cgen_var_1));
                // Begin manual dispatchable handle unboxing for pDataSize;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((size_t**)&pDataSize, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pDataSize);
                *readStreamPtrPtr += 8;
                if (pDataSize) {
                    vkReadStream->alloc((void**)&pDataSize, sizeof(size_t));
                    memcpy((size_t*)&(*pDataSize), (*readStreamPtrPtr), 8);
                    android::base::Stream::fromBe64((uint8_t*)&(*pDataSize));
                    *readStreamPtrPtr += 8;
                }
                // Begin manual dispatchable handle unboxing for pData;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((void**)&pData, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pData);
                *readStreamPtrPtr += 8;
                if (pData) {
                    vkReadStream->alloc((void**)&pData, (*(pDataSize)) * sizeof(uint8_t));
                    memcpy((void*)pData, *readStreamPtrPtr, (*(pDataSize)) * sizeof(uint8_t));
                    *readStreamPtrPtr += (*(pDataSize)) * sizeof(uint8_t);
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPipelineCacheData 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pipelineCache,
                            (unsigned long long)pDataSize, (unsigned long long)pData);
                }
                VkResult vkGetPipelineCacheData_VkResult_return = (VkResult)0;
                vkGetPipelineCacheData_VkResult_return =
                    vk->vkGetPipelineCacheData(unboxed_device, pipelineCache, pDataSize, pData);
                if ((vkGetPipelineCacheData_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkGetPipelineCacheData_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pDataSize;
                vkStream->putBe64(cgen_var_4);
                if (pDataSize) {
                    uint64_t cgen_var_4_0 = (uint64_t)(*pDataSize);
                    vkStream->putBe64(cgen_var_4_0);
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_5 = (uint64_t)(uintptr_t)pData;
                vkStream->putBe64(cgen_var_5);
                if (pData) {
                    vkStream->write((void*)pData, (*(pDataSize)) * sizeof(uint8_t));
                }
                vkStream->write(&vkGetPipelineCacheData_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPipelineCacheData(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetPipelineCacheData_VkResult_return, device, pipelineCache, pDataSize,
                        pData);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkMergePipelineCaches: {
                android::base::beginTrace("vkMergePipelineCaches decode");
                VkDevice device;
                VkPipelineCache dstCache;
                uint32_t srcCacheCount;
                const VkPipelineCache* pSrcCaches;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPipelineCache*)&dstCache =
                    (VkPipelineCache)unbox_VkPipelineCache((VkPipelineCache)(*&cgen_var_1));
                memcpy((uint32_t*)&srcCacheCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pSrcCaches,
                                    ((srcCacheCount)) * sizeof(const VkPipelineCache));
                if (((srcCacheCount))) {
                    uint8_t* cgen_var_2_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((srcCacheCount));
                    for (uint32_t k = 0; k < ((srcCacheCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_2_ptr + k * 8, sizeof(uint64_t));
                        *(((VkPipelineCache*)pSrcCaches) + k) =
                            (VkPipelineCache)unbox_VkPipelineCache((VkPipelineCache)tmpval);
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkMergePipelineCaches 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)dstCache,
                            (unsigned long long)srcCacheCount, (unsigned long long)pSrcCaches);
                }
                VkResult vkMergePipelineCaches_VkResult_return = (VkResult)0;
                vkMergePipelineCaches_VkResult_return =
                    vk->vkMergePipelineCaches(unboxed_device, dstCache, srcCacheCount, pSrcCaches);
                if ((vkMergePipelineCaches_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkMergePipelineCaches_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkMergePipelineCaches_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkMergePipelineCaches(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkMergePipelineCaches_VkResult_return, device, dstCache, srcCacheCount,
                        pSrcCaches);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateGraphicsPipelines: {
                android::base::beginTrace("vkCreateGraphicsPipelines decode");
                VkDevice device;
                VkPipelineCache pipelineCache;
                uint32_t createInfoCount;
                const VkGraphicsPipelineCreateInfo* pCreateInfos;
                const VkAllocationCallbacks* pAllocator;
                VkPipeline* pPipelines;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPipelineCache*)&pipelineCache =
                    (VkPipelineCache)unbox_VkPipelineCache((VkPipelineCache)(*&cgen_var_1));
                memcpy((uint32_t*)&createInfoCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc(
                    (void**)&pCreateInfos,
                    ((createInfoCount)) * sizeof(const VkGraphicsPipelineCreateInfo));
                for (uint32_t i = 0; i < (uint32_t)((createInfoCount)); ++i) {
                    reservedunmarshal_VkGraphicsPipelineCreateInfo(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkGraphicsPipelineCreateInfo*)(pCreateInfos + i), readStreamPtrPtr);
                }
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pPipelines;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pPipelines, ((createInfoCount)) * sizeof(VkPipeline));
                if (((createInfoCount))) {
                    uint8_t* cgen_var_3_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((createInfoCount));
                    for (uint32_t k = 0; k < ((createInfoCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_3_ptr + k * 8, sizeof(uint64_t));
                        *(((VkPipeline*)pPipelines) + k) =
                            (VkPipeline)(VkPipeline)((VkPipeline)tmpval);
                    }
                }
                if (pCreateInfos) {
                    for (uint32_t i = 0; i < (uint32_t)((createInfoCount)); ++i) {
                        transform_tohost_VkGraphicsPipelineCreateInfo(
                            m_state, (VkGraphicsPipelineCreateInfo*)(pCreateInfos + i));
                    }
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateGraphicsPipelines 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pipelineCache,
                            (unsigned long long)createInfoCount, (unsigned long long)pCreateInfos,
                            (unsigned long long)pAllocator, (unsigned long long)pPipelines);
                }
                VkResult vkCreateGraphicsPipelines_VkResult_return = (VkResult)0;
                vkCreateGraphicsPipelines_VkResult_return = m_state->on_vkCreateGraphicsPipelines(
                    &m_pool, device, pipelineCache, createInfoCount, pCreateInfos, pAllocator,
                    pPipelines);
                if ((vkCreateGraphicsPipelines_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateGraphicsPipelines_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pPipelines;
                vkStream->unsetHandleMapping();
                if (((createInfoCount))) {
                    uint64_t* cgen_var_4;
                    vkStream->alloc((void**)&cgen_var_4, ((createInfoCount)) * 8);
                    static_assert(8 == sizeof(VkPipeline),
                                  "handle map overwrite requires VkPipeline to be 8 bytes long");
                    vkStream->handleMapping()->mapHandles_VkPipeline((VkPipeline*)pPipelines,
                                                                     ((createInfoCount)));
                    vkStream->write((VkPipeline*)pPipelines, 8 * ((createInfoCount)));
                }
                // Begin manual non dispatchable handle create for pPipelines;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateGraphicsPipelines_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateGraphicsPipelines(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateGraphicsPipelines_VkResult_return, device, pipelineCache,
                        createInfoCount, pCreateInfos, pAllocator, pPipelines);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateComputePipelines: {
                android::base::beginTrace("vkCreateComputePipelines decode");
                VkDevice device;
                VkPipelineCache pipelineCache;
                uint32_t createInfoCount;
                const VkComputePipelineCreateInfo* pCreateInfos;
                const VkAllocationCallbacks* pAllocator;
                VkPipeline* pPipelines;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPipelineCache*)&pipelineCache =
                    (VkPipelineCache)unbox_VkPipelineCache((VkPipelineCache)(*&cgen_var_1));
                memcpy((uint32_t*)&createInfoCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc(
                    (void**)&pCreateInfos,
                    ((createInfoCount)) * sizeof(const VkComputePipelineCreateInfo));
                for (uint32_t i = 0; i < (uint32_t)((createInfoCount)); ++i) {
                    reservedunmarshal_VkComputePipelineCreateInfo(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkComputePipelineCreateInfo*)(pCreateInfos + i), readStreamPtrPtr);
                }
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pPipelines;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pPipelines, ((createInfoCount)) * sizeof(VkPipeline));
                if (((createInfoCount))) {
                    uint8_t* cgen_var_3_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((createInfoCount));
                    for (uint32_t k = 0; k < ((createInfoCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_3_ptr + k * 8, sizeof(uint64_t));
                        *(((VkPipeline*)pPipelines) + k) =
                            (VkPipeline)(VkPipeline)((VkPipeline)tmpval);
                    }
                }
                if (pCreateInfos) {
                    for (uint32_t i = 0; i < (uint32_t)((createInfoCount)); ++i) {
                        transform_tohost_VkComputePipelineCreateInfo(
                            m_state, (VkComputePipelineCreateInfo*)(pCreateInfos + i));
                    }
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateComputePipelines 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pipelineCache,
                            (unsigned long long)createInfoCount, (unsigned long long)pCreateInfos,
                            (unsigned long long)pAllocator, (unsigned long long)pPipelines);
                }
                VkResult vkCreateComputePipelines_VkResult_return = (VkResult)0;
                vkCreateComputePipelines_VkResult_return =
                    vk->vkCreateComputePipelines(unboxed_device, pipelineCache, createInfoCount,
                                                 pCreateInfos, pAllocator, pPipelines);
                if ((vkCreateComputePipelines_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateComputePipelines_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                // Begin auto non dispatchable handle create for pPipelines;
                if (vkCreateComputePipelines_VkResult_return == VK_SUCCESS)
                    vkStream->setHandleMapping(&m_boxedHandleCreateMapping);
                if (((createInfoCount))) {
                    uint64_t* cgen_var_4;
                    vkStream->alloc((void**)&cgen_var_4, ((createInfoCount)) * 8);
                    static_assert(8 == sizeof(VkPipeline),
                                  "handle map overwrite requires VkPipeline to be 8 bytes long");
                    vkStream->handleMapping()->mapHandles_VkPipeline((VkPipeline*)pPipelines,
                                                                     ((createInfoCount)));
                    vkStream->write((VkPipeline*)pPipelines, 8 * ((createInfoCount)));
                }
                // Begin auto non dispatchable handle create for pPipelines;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateComputePipelines_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateComputePipelines(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateComputePipelines_VkResult_return, device, pipelineCache,
                        createInfoCount, pCreateInfos, pAllocator, pPipelines);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyPipeline: {
                android::base::beginTrace("vkDestroyPipeline decode");
                VkDevice device;
                VkPipeline pipeline;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for pipeline;
                VkPipeline boxed_pipeline_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPipeline*)&pipeline = (VkPipeline)(VkPipeline)((VkPipeline)(*&cgen_var_1));
                boxed_pipeline_preserve = pipeline;
                pipeline = unbox_VkPipeline(pipeline);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyPipeline 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pipeline,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyPipeline(&m_pool, device, pipeline, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyPipeline(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, device, boxed_pipeline_preserve,
                                                           pAllocator);
                }
                delete_VkPipeline(boxed_pipeline_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreatePipelineLayout: {
                android::base::beginTrace("vkCreatePipelineLayout decode");
                VkDevice device;
                const VkPipelineLayoutCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkPipelineLayout* pPipelineLayout;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkPipelineLayoutCreateInfo));
                reservedunmarshal_VkPipelineLayoutCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPipelineLayoutCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pPipelineLayout;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pPipelineLayout, sizeof(VkPipelineLayout));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkPipelineLayout*)pPipelineLayout =
                    (VkPipelineLayout)(VkPipelineLayout)((VkPipelineLayout)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkPipelineLayoutCreateInfo(
                        m_state, (VkPipelineLayoutCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreatePipelineLayout 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pPipelineLayout);
                }
                m_state->lock();
                VkResult vkCreatePipelineLayout_VkResult_return = (VkResult)0;
                vkCreatePipelineLayout_VkResult_return = vk->vkCreatePipelineLayout(
                    unboxed_device, pCreateInfo, pAllocator, pPipelineLayout);
                if ((vkCreatePipelineLayout_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreatePipelineLayout_VkResult_return, opcode,
                                             context);
                m_state->unlock();
                vkStream->unsetHandleMapping();
                // Begin auto non dispatchable handle create for pPipelineLayout;
                if (vkCreatePipelineLayout_VkResult_return == VK_SUCCESS)
                    vkStream->setHandleMapping(&m_boxedHandleCreateMapping);
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkPipelineLayout),
                              "handle map overwrite requires VkPipelineLayout to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkPipelineLayout(
                    (VkPipelineLayout*)pPipelineLayout, 1);
                vkStream->write((VkPipelineLayout*)pPipelineLayout, 8 * 1);
                // Begin auto non dispatchable handle create for pPipelineLayout;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreatePipelineLayout_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreatePipelineLayout(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreatePipelineLayout_VkResult_return, device, pCreateInfo, pAllocator,
                        pPipelineLayout);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyPipelineLayout: {
                android::base::beginTrace("vkDestroyPipelineLayout decode");
                VkDevice device;
                VkPipelineLayout pipelineLayout;
                const VkAllocationCallbacks* pAllocator;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                // Begin manual non dispatchable handle destroy unboxing for pipelineLayout;
                VkPipelineLayout boxed_pipelineLayout_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPipelineLayout*)&pipelineLayout =
                    (VkPipelineLayout)(VkPipelineLayout)((VkPipelineLayout)(*&cgen_var_1));
                boxed_pipelineLayout_preserve = pipelineLayout;
                pipelineLayout = unbox_VkPipelineLayout(pipelineLayout);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkDestroyPipelineLayout 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)pipelineLayout, (unsigned long long)pAllocator);
                }
                std::function<void()> delayed_remove_callback = [vk, unboxed_device, pipelineLayout,
                                                                 pAllocator]() {
                    auto state = VkDecoderGlobalState::get();
                    // state already locked;
                    vk->vkDestroyPipelineLayout(unboxed_device, pipelineLayout, pAllocator);
                };
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyPipelineLayout(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_pipelineLayout_preserve, pAllocator);
                }
                delayed_delete_VkPipelineLayout(boxed_pipelineLayout_preserve, unboxed_device,
                                                delayed_remove_callback);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateSampler: {
                android::base::beginTrace("vkCreateSampler decode");
                VkDevice device;
                const VkSamplerCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkSampler* pSampler;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkSamplerCreateInfo));
                reservedunmarshal_VkSamplerCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkSamplerCreateInfo*)(pCreateInfo),
                                                      readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pSampler;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pSampler, sizeof(VkSampler));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkSampler*)pSampler = (VkSampler)(VkSampler)((VkSampler)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkSamplerCreateInfo(m_state,
                                                         (VkSamplerCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateSampler 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pSampler);
                }
                VkResult vkCreateSampler_VkResult_return = (VkResult)0;
                vkCreateSampler_VkResult_return =
                    m_state->on_vkCreateSampler(&m_pool, device, pCreateInfo, pAllocator, pSampler);
                if ((vkCreateSampler_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateSampler_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pSampler;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkSampler),
                              "handle map overwrite requires VkSampler to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkSampler((VkSampler*)pSampler, 1);
                vkStream->write((VkSampler*)pSampler, 8 * 1);
                // Begin manual non dispatchable handle create for pSampler;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateSampler_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateSampler(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, vkCreateSampler_VkResult_return,
                                                         device, pCreateInfo, pAllocator, pSampler);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroySampler: {
                android::base::beginTrace("vkDestroySampler decode");
                VkDevice device;
                VkSampler sampler;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for sampler;
                VkSampler boxed_sampler_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkSampler*)&sampler = (VkSampler)(VkSampler)((VkSampler)(*&cgen_var_1));
                boxed_sampler_preserve = sampler;
                sampler = unbox_VkSampler(sampler);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroySampler 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)sampler,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroySampler(&m_pool, device, sampler, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroySampler(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, device, boxed_sampler_preserve,
                                                          pAllocator);
                }
                delete_VkSampler(boxed_sampler_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateDescriptorSetLayout: {
                android::base::beginTrace("vkCreateDescriptorSetLayout decode");
                VkDevice device;
                const VkDescriptorSetLayoutCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkDescriptorSetLayout* pSetLayout;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo,
                                    sizeof(const VkDescriptorSetLayoutCreateInfo));
                reservedunmarshal_VkDescriptorSetLayoutCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDescriptorSetLayoutCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pSetLayout;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pSetLayout, sizeof(VkDescriptorSetLayout));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkDescriptorSetLayout*)pSetLayout =
                    (VkDescriptorSetLayout)(VkDescriptorSetLayout)((
                        VkDescriptorSetLayout)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkDescriptorSetLayoutCreateInfo(
                        m_state, (VkDescriptorSetLayoutCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateDescriptorSetLayout 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pSetLayout);
                }
                VkResult vkCreateDescriptorSetLayout_VkResult_return = (VkResult)0;
                vkCreateDescriptorSetLayout_VkResult_return =
                    m_state->on_vkCreateDescriptorSetLayout(&m_pool, device, pCreateInfo,
                                                            pAllocator, pSetLayout);
                if ((vkCreateDescriptorSetLayout_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateDescriptorSetLayout_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pSetLayout;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(
                    8 == sizeof(VkDescriptorSetLayout),
                    "handle map overwrite requires VkDescriptorSetLayout to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkDescriptorSetLayout(
                    (VkDescriptorSetLayout*)pSetLayout, 1);
                vkStream->write((VkDescriptorSetLayout*)pSetLayout, 8 * 1);
                // Begin manual non dispatchable handle create for pSetLayout;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateDescriptorSetLayout_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateDescriptorSetLayout(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateDescriptorSetLayout_VkResult_return, device, pCreateInfo,
                        pAllocator, pSetLayout);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyDescriptorSetLayout: {
                android::base::beginTrace("vkDestroyDescriptorSetLayout decode");
                VkDevice device;
                VkDescriptorSetLayout descriptorSetLayout;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for descriptorSetLayout;
                VkDescriptorSetLayout boxed_descriptorSetLayout_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDescriptorSetLayout*)&descriptorSetLayout =
                    (VkDescriptorSetLayout)(VkDescriptorSetLayout)((
                        VkDescriptorSetLayout)(*&cgen_var_1));
                boxed_descriptorSetLayout_preserve = descriptorSetLayout;
                descriptorSetLayout = unbox_VkDescriptorSetLayout(descriptorSetLayout);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkDestroyDescriptorSetLayout 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)descriptorSetLayout,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyDescriptorSetLayout(&m_pool, device, descriptorSetLayout,
                                                         pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyDescriptorSetLayout(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_descriptorSetLayout_preserve, pAllocator);
                }
                delete_VkDescriptorSetLayout(boxed_descriptorSetLayout_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateDescriptorPool: {
                android::base::beginTrace("vkCreateDescriptorPool decode");
                VkDevice device;
                const VkDescriptorPoolCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkDescriptorPool* pDescriptorPool;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkDescriptorPoolCreateInfo));
                reservedunmarshal_VkDescriptorPoolCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDescriptorPoolCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pDescriptorPool;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pDescriptorPool, sizeof(VkDescriptorPool));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkDescriptorPool*)pDescriptorPool =
                    (VkDescriptorPool)(VkDescriptorPool)((VkDescriptorPool)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkDescriptorPoolCreateInfo(
                        m_state, (VkDescriptorPoolCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateDescriptorPool 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pDescriptorPool);
                }
                VkResult vkCreateDescriptorPool_VkResult_return = (VkResult)0;
                vkCreateDescriptorPool_VkResult_return = m_state->on_vkCreateDescriptorPool(
                    &m_pool, device, pCreateInfo, pAllocator, pDescriptorPool);
                if ((vkCreateDescriptorPool_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateDescriptorPool_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pDescriptorPool;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkDescriptorPool),
                              "handle map overwrite requires VkDescriptorPool to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkDescriptorPool(
                    (VkDescriptorPool*)pDescriptorPool, 1);
                vkStream->write((VkDescriptorPool*)pDescriptorPool, 8 * 1);
                // Begin manual non dispatchable handle create for pDescriptorPool;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateDescriptorPool_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateDescriptorPool(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateDescriptorPool_VkResult_return, device, pCreateInfo, pAllocator,
                        pDescriptorPool);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyDescriptorPool: {
                android::base::beginTrace("vkDestroyDescriptorPool decode");
                VkDevice device;
                VkDescriptorPool descriptorPool;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for descriptorPool;
                VkDescriptorPool boxed_descriptorPool_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDescriptorPool*)&descriptorPool =
                    (VkDescriptorPool)(VkDescriptorPool)((VkDescriptorPool)(*&cgen_var_1));
                boxed_descriptorPool_preserve = descriptorPool;
                descriptorPool = unbox_VkDescriptorPool(descriptorPool);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkDestroyDescriptorPool 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)descriptorPool, (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyDescriptorPool(&m_pool, device, descriptorPool, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyDescriptorPool(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_descriptorPool_preserve, pAllocator);
                }
                delete_VkDescriptorPool(boxed_descriptorPool_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkResetDescriptorPool: {
                android::base::beginTrace("vkResetDescriptorPool decode");
                VkDevice device;
                VkDescriptorPool descriptorPool;
                VkDescriptorPoolResetFlags flags;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDescriptorPool*)&descriptorPool =
                    (VkDescriptorPool)unbox_VkDescriptorPool((VkDescriptorPool)(*&cgen_var_1));
                memcpy((VkDescriptorPoolResetFlags*)&flags, *readStreamPtrPtr,
                       sizeof(VkDescriptorPoolResetFlags));
                *readStreamPtrPtr += sizeof(VkDescriptorPoolResetFlags);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkResetDescriptorPool 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)descriptorPool, (unsigned long long)flags);
                }
                VkResult vkResetDescriptorPool_VkResult_return = (VkResult)0;
                vkResetDescriptorPool_VkResult_return =
                    m_state->on_vkResetDescriptorPool(&m_pool, device, descriptorPool, flags);
                if ((vkResetDescriptorPool_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkResetDescriptorPool_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkResetDescriptorPool_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkResetDescriptorPool(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkResetDescriptorPool_VkResult_return, device, descriptorPool, flags);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkAllocateDescriptorSets: {
                android::base::beginTrace("vkAllocateDescriptorSets decode");
                VkDevice device;
                const VkDescriptorSetAllocateInfo* pAllocateInfo;
                VkDescriptorSet* pDescriptorSets;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pAllocateInfo,
                                    sizeof(const VkDescriptorSetAllocateInfo));
                reservedunmarshal_VkDescriptorSetAllocateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDescriptorSetAllocateInfo*)(pAllocateInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pDescriptorSets;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pDescriptorSets,
                                    pAllocateInfo->descriptorSetCount * sizeof(VkDescriptorSet));
                if (pAllocateInfo->descriptorSetCount) {
                    uint8_t* cgen_var_1_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * pAllocateInfo->descriptorSetCount;
                    for (uint32_t k = 0; k < pAllocateInfo->descriptorSetCount; ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_1_ptr + k * 8, sizeof(uint64_t));
                        *(((VkDescriptorSet*)pDescriptorSets) + k) =
                            (VkDescriptorSet)(VkDescriptorSet)((VkDescriptorSet)tmpval);
                    }
                }
                if (pAllocateInfo) {
                    transform_tohost_VkDescriptorSetAllocateInfo(
                        m_state, (VkDescriptorSetAllocateInfo*)(pAllocateInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkAllocateDescriptorSets 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pAllocateInfo,
                            (unsigned long long)pDescriptorSets);
                }
                VkResult vkAllocateDescriptorSets_VkResult_return = (VkResult)0;
                vkAllocateDescriptorSets_VkResult_return = m_state->on_vkAllocateDescriptorSets(
                    &m_pool, device, pAllocateInfo, pDescriptorSets);
                if ((vkAllocateDescriptorSets_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkAllocateDescriptorSets_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pDescriptorSets;
                vkStream->unsetHandleMapping();
                if (pAllocateInfo->descriptorSetCount) {
                    uint64_t* cgen_var_2;
                    vkStream->alloc((void**)&cgen_var_2, pAllocateInfo->descriptorSetCount * 8);
                    static_assert(
                        8 == sizeof(VkDescriptorSet),
                        "handle map overwrite requires VkDescriptorSet to be 8 bytes long");
                    vkStream->handleMapping()->mapHandles_VkDescriptorSet(
                        (VkDescriptorSet*)pDescriptorSets, pAllocateInfo->descriptorSetCount);
                    vkStream->write((VkDescriptorSet*)pDescriptorSets,
                                    8 * pAllocateInfo->descriptorSetCount);
                }
                // Begin manual non dispatchable handle create for pDescriptorSets;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkAllocateDescriptorSets_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkAllocateDescriptorSets(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkAllocateDescriptorSets_VkResult_return, device, pAllocateInfo,
                        pDescriptorSets);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkFreeDescriptorSets: {
                android::base::beginTrace("vkFreeDescriptorSets decode");
                VkDevice device;
                VkDescriptorPool descriptorPool;
                uint32_t descriptorSetCount;
                const VkDescriptorSet* pDescriptorSets;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDescriptorPool*)&descriptorPool =
                    (VkDescriptorPool)unbox_VkDescriptorPool((VkDescriptorPool)(*&cgen_var_1));
                memcpy((uint32_t*)&descriptorSetCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                // Begin manual non dispatchable handle destroy unboxing for pDescriptorSets;
                VkDescriptorSet* boxed_pDescriptorSets_preserve;
                vkReadStream->alloc((void**)&boxed_pDescriptorSets_preserve,
                                    ((descriptorSetCount)) * sizeof(VkDescriptorSet));
                // WARNING PTR CHECK
                memcpy((VkDescriptorSet**)&pDescriptorSets, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pDescriptorSets);
                *readStreamPtrPtr += 8;
                if (pDescriptorSets) {
                    vkReadStream->alloc((void**)&pDescriptorSets,
                                        ((descriptorSetCount)) * sizeof(const VkDescriptorSet));
                    if (((descriptorSetCount))) {
                        uint8_t* cgen_var_2_0_ptr = (uint8_t*)(*readStreamPtrPtr);
                        *readStreamPtrPtr += 8 * ((descriptorSetCount));
                        for (uint32_t k = 0; k < ((descriptorSetCount)); ++k) {
                            uint64_t tmpval;
                            memcpy(&tmpval, cgen_var_2_0_ptr + k * 8, sizeof(uint64_t));
                            *(((VkDescriptorSet*)pDescriptorSets) + k) =
                                (VkDescriptorSet)(VkDescriptorSet)((VkDescriptorSet)tmpval);
                        }
                    }
                }
                for (uint32_t i = 0; i < ((descriptorSetCount)); ++i) {
                    boxed_pDescriptorSets_preserve[i] = pDescriptorSets[i];
                    ((VkDescriptorSet*)(pDescriptorSets))[i] =
                        unbox_VkDescriptorSet(pDescriptorSets[i]);
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkFreeDescriptorSets 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)descriptorPool,
                            (unsigned long long)descriptorSetCount,
                            (unsigned long long)pDescriptorSets);
                }
                VkResult vkFreeDescriptorSets_VkResult_return = (VkResult)0;
                vkFreeDescriptorSets_VkResult_return = m_state->on_vkFreeDescriptorSets(
                    &m_pool, device, descriptorPool, descriptorSetCount, pDescriptorSets);
                if ((vkFreeDescriptorSets_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkFreeDescriptorSets_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkFreeDescriptorSets_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkFreeDescriptorSets(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkFreeDescriptorSets_VkResult_return, device, descriptorPool,
                        descriptorSetCount, boxed_pDescriptorSets_preserve);
                }
                // Skipping handle cleanup for vkFreeDescriptorSets
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkUpdateDescriptorSets: {
                android::base::beginTrace("vkUpdateDescriptorSets decode");
                VkDevice device;
                uint32_t descriptorWriteCount;
                const VkWriteDescriptorSet* pDescriptorWrites;
                uint32_t descriptorCopyCount;
                const VkCopyDescriptorSet* pDescriptorCopies;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                memcpy((uint32_t*)&descriptorWriteCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pDescriptorWrites,
                                    ((descriptorWriteCount)) * sizeof(const VkWriteDescriptorSet));
                for (uint32_t i = 0; i < (uint32_t)((descriptorWriteCount)); ++i) {
                    reservedunmarshal_VkWriteDescriptorSet(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkWriteDescriptorSet*)(pDescriptorWrites + i), readStreamPtrPtr);
                }
                memcpy((uint32_t*)&descriptorCopyCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pDescriptorCopies,
                                    ((descriptorCopyCount)) * sizeof(const VkCopyDescriptorSet));
                for (uint32_t i = 0; i < (uint32_t)((descriptorCopyCount)); ++i) {
                    reservedunmarshal_VkCopyDescriptorSet(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkCopyDescriptorSet*)(pDescriptorCopies + i), readStreamPtrPtr);
                }
                if (pDescriptorWrites) {
                    for (uint32_t i = 0; i < (uint32_t)((descriptorWriteCount)); ++i) {
                        transform_tohost_VkWriteDescriptorSet(
                            m_state, (VkWriteDescriptorSet*)(pDescriptorWrites + i));
                    }
                }
                if (pDescriptorCopies) {
                    for (uint32_t i = 0; i < (uint32_t)((descriptorCopyCount)); ++i) {
                        transform_tohost_VkCopyDescriptorSet(
                            m_state, (VkCopyDescriptorSet*)(pDescriptorCopies + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkUpdateDescriptorSets 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)descriptorWriteCount,
                            (unsigned long long)pDescriptorWrites,
                            (unsigned long long)descriptorCopyCount,
                            (unsigned long long)pDescriptorCopies);
                }
                m_state->on_vkUpdateDescriptorSets(&m_pool, device, descriptorWriteCount,
                                                   pDescriptorWrites, descriptorCopyCount,
                                                   pDescriptorCopies);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkUpdateDescriptorSets(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        descriptorWriteCount, pDescriptorWrites, descriptorCopyCount,
                        pDescriptorCopies);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateFramebuffer: {
                android::base::beginTrace("vkCreateFramebuffer decode");
                VkDevice device;
                const VkFramebufferCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkFramebuffer* pFramebuffer;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkFramebufferCreateInfo));
                reservedunmarshal_VkFramebufferCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                          (VkFramebufferCreateInfo*)(pCreateInfo),
                                                          readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pFramebuffer;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pFramebuffer, sizeof(VkFramebuffer));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkFramebuffer*)pFramebuffer =
                    (VkFramebuffer)(VkFramebuffer)((VkFramebuffer)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkFramebufferCreateInfo(
                        m_state, (VkFramebufferCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateFramebuffer 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pFramebuffer);
                }
                VkResult vkCreateFramebuffer_VkResult_return = (VkResult)0;
                vkCreateFramebuffer_VkResult_return = m_state->on_vkCreateFramebuffer(
                    &m_pool, device, pCreateInfo, pAllocator, pFramebuffer);
                if ((vkCreateFramebuffer_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateFramebuffer_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pFramebuffer;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkFramebuffer),
                              "handle map overwrite requires VkFramebuffer to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkFramebuffer((VkFramebuffer*)pFramebuffer,
                                                                    1);
                vkStream->write((VkFramebuffer*)pFramebuffer, 8 * 1);
                // Begin manual non dispatchable handle create for pFramebuffer;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateFramebuffer_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateFramebuffer(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateFramebuffer_VkResult_return, device, pCreateInfo, pAllocator,
                        pFramebuffer);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyFramebuffer: {
                android::base::beginTrace("vkDestroyFramebuffer decode");
                VkDevice device;
                VkFramebuffer framebuffer;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for framebuffer;
                VkFramebuffer boxed_framebuffer_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkFramebuffer*)&framebuffer =
                    (VkFramebuffer)(VkFramebuffer)((VkFramebuffer)(*&cgen_var_1));
                boxed_framebuffer_preserve = framebuffer;
                framebuffer = unbox_VkFramebuffer(framebuffer);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyFramebuffer 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)framebuffer,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyFramebuffer(&m_pool, device, framebuffer, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyFramebuffer(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_framebuffer_preserve, pAllocator);
                }
                delete_VkFramebuffer(boxed_framebuffer_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateRenderPass: {
                android::base::beginTrace("vkCreateRenderPass decode");
                VkDevice device;
                const VkRenderPassCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkRenderPass* pRenderPass;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkRenderPassCreateInfo));
                reservedunmarshal_VkRenderPassCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                         (VkRenderPassCreateInfo*)(pCreateInfo),
                                                         readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pRenderPass;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pRenderPass, sizeof(VkRenderPass));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkRenderPass*)pRenderPass =
                    (VkRenderPass)(VkRenderPass)((VkRenderPass)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkRenderPassCreateInfo(m_state,
                                                            (VkRenderPassCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateRenderPass 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pRenderPass);
                }
                VkResult vkCreateRenderPass_VkResult_return = (VkResult)0;
                vkCreateRenderPass_VkResult_return = m_state->on_vkCreateRenderPass(
                    &m_pool, device, pCreateInfo, pAllocator, pRenderPass);
                if ((vkCreateRenderPass_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateRenderPass_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pRenderPass;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkRenderPass),
                              "handle map overwrite requires VkRenderPass to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkRenderPass((VkRenderPass*)pRenderPass, 1);
                vkStream->write((VkRenderPass*)pRenderPass, 8 * 1);
                // Begin manual non dispatchable handle create for pRenderPass;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateRenderPass_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateRenderPass(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateRenderPass_VkResult_return, device, pCreateInfo, pAllocator,
                        pRenderPass);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyRenderPass: {
                android::base::beginTrace("vkDestroyRenderPass decode");
                VkDevice device;
                VkRenderPass renderPass;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for renderPass;
                VkRenderPass boxed_renderPass_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkRenderPass*)&renderPass =
                    (VkRenderPass)(VkRenderPass)((VkRenderPass)(*&cgen_var_1));
                boxed_renderPass_preserve = renderPass;
                renderPass = unbox_VkRenderPass(renderPass);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyRenderPass 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)renderPass,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyRenderPass(&m_pool, device, renderPass, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyRenderPass(snapshotTraceBegin, snapshotTraceBytes,
                                                             &m_pool, device,
                                                             boxed_renderPass_preserve, pAllocator);
                }
                delete_VkRenderPass(boxed_renderPass_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetRenderAreaGranularity: {
                android::base::beginTrace("vkGetRenderAreaGranularity decode");
                VkDevice device;
                VkRenderPass renderPass;
                VkExtent2D* pGranularity;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkRenderPass*)&renderPass =
                    (VkRenderPass)unbox_VkRenderPass((VkRenderPass)(*&cgen_var_1));
                // Begin manual dispatchable handle unboxing for pGranularity;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pGranularity, sizeof(VkExtent2D));
                reservedunmarshal_VkExtent2D(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                             (VkExtent2D*)(pGranularity), readStreamPtrPtr);
                if (pGranularity) {
                    transform_tohost_VkExtent2D(m_state, (VkExtent2D*)(pGranularity));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetRenderAreaGranularity 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)renderPass,
                            (unsigned long long)pGranularity);
                }
                vk->vkGetRenderAreaGranularity(unboxed_device, renderPass, pGranularity);
                vkStream->unsetHandleMapping();
                if (pGranularity) {
                    transform_fromhost_VkExtent2D(m_state, (VkExtent2D*)(pGranularity));
                }
                marshal_VkExtent2D(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                   (VkExtent2D*)(pGranularity));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetRenderAreaGranularity(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, renderPass,
                        pGranularity);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateCommandPool: {
                android::base::beginTrace("vkCreateCommandPool decode");
                VkDevice device;
                const VkCommandPoolCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkCommandPool* pCommandPool;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkCommandPoolCreateInfo));
                reservedunmarshal_VkCommandPoolCreateInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                          (VkCommandPoolCreateInfo*)(pCreateInfo),
                                                          readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pCommandPool;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pCommandPool, sizeof(VkCommandPool));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkCommandPool*)pCommandPool =
                    (VkCommandPool)(VkCommandPool)((VkCommandPool)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkCommandPoolCreateInfo(
                        m_state, (VkCommandPoolCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateCommandPool 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pCommandPool);
                }
                VkResult vkCreateCommandPool_VkResult_return = (VkResult)0;
                vkCreateCommandPool_VkResult_return = m_state->on_vkCreateCommandPool(
                    &m_pool, device, pCreateInfo, pAllocator, pCommandPool);
                if ((vkCreateCommandPool_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateCommandPool_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pCommandPool;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkCommandPool),
                              "handle map overwrite requires VkCommandPool to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkCommandPool((VkCommandPool*)pCommandPool,
                                                                    1);
                vkStream->write((VkCommandPool*)pCommandPool, 8 * 1);
                // Begin manual non dispatchable handle create for pCommandPool;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateCommandPool_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateCommandPool(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateCommandPool_VkResult_return, device, pCreateInfo, pAllocator,
                        pCommandPool);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyCommandPool: {
                android::base::beginTrace("vkDestroyCommandPool decode");
                VkDevice device;
                VkCommandPool commandPool;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for commandPool;
                VkCommandPool boxed_commandPool_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandPool*)&commandPool =
                    (VkCommandPool)(VkCommandPool)((VkCommandPool)(*&cgen_var_1));
                boxed_commandPool_preserve = commandPool;
                commandPool = unbox_VkCommandPool(commandPool);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkDestroyCommandPool 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)commandPool,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyCommandPool(&m_pool, device, commandPool, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyCommandPool(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_commandPool_preserve, pAllocator);
                }
                delete_VkCommandPool(boxed_commandPool_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkResetCommandPool: {
                android::base::beginTrace("vkResetCommandPool decode");
                VkDevice device;
                VkCommandPool commandPool;
                VkCommandPoolResetFlags flags;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandPool*)&commandPool =
                    (VkCommandPool)unbox_VkCommandPool((VkCommandPool)(*&cgen_var_1));
                memcpy((VkCommandPoolResetFlags*)&flags, *readStreamPtrPtr,
                       sizeof(VkCommandPoolResetFlags));
                *readStreamPtrPtr += sizeof(VkCommandPoolResetFlags);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkResetCommandPool 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)commandPool,
                            (unsigned long long)flags);
                }
                VkResult vkResetCommandPool_VkResult_return = (VkResult)0;
                vkResetCommandPool_VkResult_return =
                    m_state->on_vkResetCommandPool(&m_pool, device, commandPool, flags);
                if ((vkResetCommandPool_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkResetCommandPool_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkResetCommandPool_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkResetCommandPool(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkResetCommandPool_VkResult_return, device, commandPool, flags);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkAllocateCommandBuffers: {
                android::base::beginTrace("vkAllocateCommandBuffers decode");
                VkDevice device;
                const VkCommandBufferAllocateInfo* pAllocateInfo;
                VkCommandBuffer* pCommandBuffers;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pAllocateInfo,
                                    sizeof(const VkCommandBufferAllocateInfo));
                reservedunmarshal_VkCommandBufferAllocateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkCommandBufferAllocateInfo*)(pAllocateInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pCommandBuffers;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pCommandBuffers,
                                    pAllocateInfo->commandBufferCount * sizeof(VkCommandBuffer));
                if (pAllocateInfo->commandBufferCount) {
                    uint8_t* cgen_var_1_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * pAllocateInfo->commandBufferCount;
                    for (uint32_t k = 0; k < pAllocateInfo->commandBufferCount; ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_1_ptr + k * 8, sizeof(uint64_t));
                        *(((VkCommandBuffer*)pCommandBuffers) + k) =
                            (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)tmpval);
                    }
                }
                if (pAllocateInfo) {
                    transform_tohost_VkCommandBufferAllocateInfo(
                        m_state, (VkCommandBufferAllocateInfo*)(pAllocateInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkAllocateCommandBuffers 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pAllocateInfo,
                            (unsigned long long)pCommandBuffers);
                }
                VkResult vkAllocateCommandBuffers_VkResult_return = (VkResult)0;
                vkAllocateCommandBuffers_VkResult_return = m_state->on_vkAllocateCommandBuffers(
                    &m_pool, device, pAllocateInfo, pCommandBuffers);
                if ((vkAllocateCommandBuffers_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkAllocateCommandBuffers_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                if (pAllocateInfo->commandBufferCount) {
                    uint64_t* cgen_var_2;
                    vkStream->alloc((void**)&cgen_var_2, pAllocateInfo->commandBufferCount * 8);
                    static_assert(
                        8 == sizeof(VkCommandBuffer),
                        "handle map overwrite requires VkCommandBuffer to be 8 bytes long");
                    vkStream->handleMapping()->mapHandles_VkCommandBuffer(
                        (VkCommandBuffer*)pCommandBuffers, pAllocateInfo->commandBufferCount);
                    vkStream->write((VkCommandBuffer*)pCommandBuffers,
                                    8 * pAllocateInfo->commandBufferCount);
                }
                vkStream->write(&vkAllocateCommandBuffers_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkAllocateCommandBuffers(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkAllocateCommandBuffers_VkResult_return, device, pAllocateInfo,
                        pCommandBuffers);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkFreeCommandBuffers: {
                android::base::beginTrace("vkFreeCommandBuffers decode");
                VkDevice device;
                VkCommandPool commandPool;
                uint32_t commandBufferCount;
                const VkCommandBuffer* pCommandBuffers;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandPool*)&commandPool =
                    (VkCommandPool)unbox_VkCommandPool((VkCommandPool)(*&cgen_var_1));
                memcpy((uint32_t*)&commandBufferCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                // Begin manual non dispatchable handle destroy unboxing for pCommandBuffers;
                VkCommandBuffer* boxed_pCommandBuffers_preserve;
                vkReadStream->alloc((void**)&boxed_pCommandBuffers_preserve,
                                    ((commandBufferCount)) * sizeof(VkCommandBuffer));
                // WARNING PTR CHECK
                memcpy((VkCommandBuffer**)&pCommandBuffers, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pCommandBuffers);
                *readStreamPtrPtr += 8;
                if (pCommandBuffers) {
                    vkReadStream->alloc((void**)&pCommandBuffers,
                                        ((commandBufferCount)) * sizeof(const VkCommandBuffer));
                    if (((commandBufferCount))) {
                        uint8_t* cgen_var_2_0_ptr = (uint8_t*)(*readStreamPtrPtr);
                        *readStreamPtrPtr += 8 * ((commandBufferCount));
                        for (uint32_t k = 0; k < ((commandBufferCount)); ++k) {
                            uint64_t tmpval;
                            memcpy(&tmpval, cgen_var_2_0_ptr + k * 8, sizeof(uint64_t));
                            *(((VkCommandBuffer*)pCommandBuffers) + k) =
                                (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)tmpval);
                        }
                    }
                }
                for (uint32_t i = 0; i < ((commandBufferCount)); ++i) {
                    boxed_pCommandBuffers_preserve[i] = pCommandBuffers[i];
                    ((VkCommandBuffer*)(pCommandBuffers))[i] =
                        unbox_VkCommandBuffer(pCommandBuffers[i]);
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkFreeCommandBuffers 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)commandPool,
                            (unsigned long long)commandBufferCount,
                            (unsigned long long)pCommandBuffers);
                }
                m_state->on_vkFreeCommandBuffers(&m_pool, device, commandPool, commandBufferCount,
                                                 pCommandBuffers);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkFreeCommandBuffers(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, commandPool,
                        commandBufferCount, boxed_pCommandBuffers_preserve);
                }
                if (((commandBufferCount))) {
                    for (uint32_t i = 0; i < ((commandBufferCount)); ++i) {
                        delete_VkCommandBuffer(boxed_pCommandBuffers_preserve[i]);
                    }
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkBeginCommandBuffer: {
                android::base::beginTrace("vkBeginCommandBuffer decode");
                VkCommandBuffer commandBuffer;
                const VkCommandBufferBeginInfo* pBeginInfo;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pBeginInfo, sizeof(const VkCommandBufferBeginInfo));
                reservedunmarshal_VkCommandBufferBeginInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                           (VkCommandBufferBeginInfo*)(pBeginInfo),
                                                           readStreamPtrPtr);
                if (pBeginInfo) {
                    transform_tohost_VkCommandBufferBeginInfo(
                        m_state, (VkCommandBufferBeginInfo*)(pBeginInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkBeginCommandBuffer 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pBeginInfo);
                }
                VkResult vkBeginCommandBuffer_VkResult_return = (VkResult)0;
                vkBeginCommandBuffer_VkResult_return =
                    m_state->on_vkBeginCommandBuffer(&m_pool, commandBuffer, pBeginInfo, context);
                if ((vkBeginCommandBuffer_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkBeginCommandBuffer_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkBeginCommandBuffer_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkBeginCommandBuffer(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkBeginCommandBuffer_VkResult_return, commandBuffer, pBeginInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkEndCommandBuffer: {
                android::base::beginTrace("vkEndCommandBuffer decode");
                VkCommandBuffer commandBuffer;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkEndCommandBuffer 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer);
                }
                VkResult vkEndCommandBuffer_VkResult_return = (VkResult)0;
                vkEndCommandBuffer_VkResult_return =
                    m_state->on_vkEndCommandBuffer(&m_pool, commandBuffer, context);
                if ((vkEndCommandBuffer_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkEndCommandBuffer_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkEndCommandBuffer_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkEndCommandBuffer(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkEndCommandBuffer_VkResult_return, commandBuffer);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkResetCommandBuffer: {
                android::base::beginTrace("vkResetCommandBuffer decode");
                VkCommandBuffer commandBuffer;
                VkCommandBufferResetFlags flags;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                memcpy((VkCommandBufferResetFlags*)&flags, *readStreamPtrPtr,
                       sizeof(VkCommandBufferResetFlags));
                *readStreamPtrPtr += sizeof(VkCommandBufferResetFlags);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkResetCommandBuffer 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)flags);
                }
                VkResult vkResetCommandBuffer_VkResult_return = (VkResult)0;
                vkResetCommandBuffer_VkResult_return =
                    m_state->on_vkResetCommandBuffer(&m_pool, commandBuffer, flags);
                if ((vkResetCommandBuffer_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkResetCommandBuffer_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkResetCommandBuffer_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkResetCommandBuffer(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkResetCommandBuffer_VkResult_return, commandBuffer, flags);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBindPipeline: {
                android::base::beginTrace("vkCmdBindPipeline decode");
                VkCommandBuffer commandBuffer;
                VkPipelineBindPoint pipelineBindPoint;
                VkPipeline pipeline;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                memcpy((VkPipelineBindPoint*)&pipelineBindPoint, *readStreamPtrPtr,
                       sizeof(VkPipelineBindPoint));
                *readStreamPtrPtr += sizeof(VkPipelineBindPoint);
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPipeline*)&pipeline = (VkPipeline)unbox_VkPipeline((VkPipeline)(*&cgen_var_1));
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdBindPipeline 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pipelineBindPoint, (unsigned long long)pipeline);
                }
                m_state->on_vkCmdBindPipeline(&m_pool, commandBuffer, pipelineBindPoint, pipeline);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBindPipeline(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, commandBuffer,
                                                           pipelineBindPoint, pipeline);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetViewport: {
                android::base::beginTrace("vkCmdSetViewport decode");
                VkCommandBuffer commandBuffer;
                uint32_t firstViewport;
                uint32_t viewportCount;
                const VkViewport* pViewports;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&firstViewport, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&viewportCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pViewports,
                                    ((viewportCount)) * sizeof(const VkViewport));
                for (uint32_t i = 0; i < (uint32_t)((viewportCount)); ++i) {
                    reservedunmarshal_VkViewport(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                 (VkViewport*)(pViewports + i), readStreamPtrPtr);
                }
                if (pViewports) {
                    for (uint32_t i = 0; i < (uint32_t)((viewportCount)); ++i) {
                        transform_tohost_VkViewport(m_state, (VkViewport*)(pViewports + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetViewport 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)firstViewport, (unsigned long long)viewportCount,
                            (unsigned long long)pViewports);
                }
                vk->vkCmdSetViewport(unboxed_commandBuffer, firstViewport, viewportCount,
                                     pViewports);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetViewport(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, commandBuffer, firstViewport,
                                                          viewportCount, pViewports);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetScissor: {
                android::base::beginTrace("vkCmdSetScissor decode");
                VkCommandBuffer commandBuffer;
                uint32_t firstScissor;
                uint32_t scissorCount;
                const VkRect2D* pScissors;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&firstScissor, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&scissorCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pScissors, ((scissorCount)) * sizeof(const VkRect2D));
                for (uint32_t i = 0; i < (uint32_t)((scissorCount)); ++i) {
                    reservedunmarshal_VkRect2D(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                               (VkRect2D*)(pScissors + i), readStreamPtrPtr);
                }
                if (pScissors) {
                    for (uint32_t i = 0; i < (uint32_t)((scissorCount)); ++i) {
                        transform_tohost_VkRect2D(m_state, (VkRect2D*)(pScissors + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetScissor 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)firstScissor, (unsigned long long)scissorCount,
                            (unsigned long long)pScissors);
                }
                vk->vkCmdSetScissor(unboxed_commandBuffer, firstScissor, scissorCount, pScissors);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetScissor(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, commandBuffer, firstScissor,
                                                         scissorCount, pScissors);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetLineWidth: {
                android::base::beginTrace("vkCmdSetLineWidth decode");
                VkCommandBuffer commandBuffer;
                float lineWidth;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((float*)&lineWidth, *readStreamPtrPtr, sizeof(float));
                *readStreamPtrPtr += sizeof(float);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetLineWidth 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer, (unsigned long long)lineWidth);
                }
                vk->vkCmdSetLineWidth(unboxed_commandBuffer, lineWidth);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetLineWidth(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, commandBuffer, lineWidth);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetDepthBias: {
                android::base::beginTrace("vkCmdSetDepthBias decode");
                VkCommandBuffer commandBuffer;
                float depthBiasConstantFactor;
                float depthBiasClamp;
                float depthBiasSlopeFactor;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((float*)&depthBiasConstantFactor, *readStreamPtrPtr, sizeof(float));
                *readStreamPtrPtr += sizeof(float);
                memcpy((float*)&depthBiasClamp, *readStreamPtrPtr, sizeof(float));
                *readStreamPtrPtr += sizeof(float);
                memcpy((float*)&depthBiasSlopeFactor, *readStreamPtrPtr, sizeof(float));
                *readStreamPtrPtr += sizeof(float);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetDepthBias 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)depthBiasConstantFactor,
                            (unsigned long long)depthBiasClamp,
                            (unsigned long long)depthBiasSlopeFactor);
                }
                vk->vkCmdSetDepthBias(unboxed_commandBuffer, depthBiasConstantFactor,
                                      depthBiasClamp, depthBiasSlopeFactor);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetDepthBias(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        depthBiasConstantFactor, depthBiasClamp, depthBiasSlopeFactor);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetBlendConstants: {
                android::base::beginTrace("vkCmdSetBlendConstants decode");
                VkCommandBuffer commandBuffer;
                float blendConstants[4];
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((float*)blendConstants, *readStreamPtrPtr, 4 * sizeof(const float));
                *readStreamPtrPtr += 4 * sizeof(const float);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetBlendConstants 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)blendConstants);
                }
                vk->vkCmdSetBlendConstants(unboxed_commandBuffer, blendConstants);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetBlendConstants(snapshotTraceBegin,
                                                                snapshotTraceBytes, &m_pool,
                                                                commandBuffer, blendConstants);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetDepthBounds: {
                android::base::beginTrace("vkCmdSetDepthBounds decode");
                VkCommandBuffer commandBuffer;
                float minDepthBounds;
                float maxDepthBounds;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((float*)&minDepthBounds, *readStreamPtrPtr, sizeof(float));
                *readStreamPtrPtr += sizeof(float);
                memcpy((float*)&maxDepthBounds, *readStreamPtrPtr, sizeof(float));
                *readStreamPtrPtr += sizeof(float);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetDepthBounds 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)minDepthBounds, (unsigned long long)maxDepthBounds);
                }
                vk->vkCmdSetDepthBounds(unboxed_commandBuffer, minDepthBounds, maxDepthBounds);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetDepthBounds(snapshotTraceBegin, snapshotTraceBytes,
                                                             &m_pool, commandBuffer, minDepthBounds,
                                                             maxDepthBounds);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetStencilCompareMask: {
                android::base::beginTrace("vkCmdSetStencilCompareMask decode");
                VkCommandBuffer commandBuffer;
                VkStencilFaceFlags faceMask;
                uint32_t compareMask;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkStencilFaceFlags*)&faceMask, *readStreamPtrPtr,
                       sizeof(VkStencilFaceFlags));
                *readStreamPtrPtr += sizeof(VkStencilFaceFlags);
                memcpy((uint32_t*)&compareMask, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetStencilCompareMask 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)faceMask, (unsigned long long)compareMask);
                }
                vk->vkCmdSetStencilCompareMask(unboxed_commandBuffer, faceMask, compareMask);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetStencilCompareMask(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, faceMask,
                        compareMask);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetStencilWriteMask: {
                android::base::beginTrace("vkCmdSetStencilWriteMask decode");
                VkCommandBuffer commandBuffer;
                VkStencilFaceFlags faceMask;
                uint32_t writeMask;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkStencilFaceFlags*)&faceMask, *readStreamPtrPtr,
                       sizeof(VkStencilFaceFlags));
                *readStreamPtrPtr += sizeof(VkStencilFaceFlags);
                memcpy((uint32_t*)&writeMask, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetStencilWriteMask 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)faceMask, (unsigned long long)writeMask);
                }
                vk->vkCmdSetStencilWriteMask(unboxed_commandBuffer, faceMask, writeMask);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetStencilWriteMask(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, faceMask,
                        writeMask);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetStencilReference: {
                android::base::beginTrace("vkCmdSetStencilReference decode");
                VkCommandBuffer commandBuffer;
                VkStencilFaceFlags faceMask;
                uint32_t reference;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkStencilFaceFlags*)&faceMask, *readStreamPtrPtr,
                       sizeof(VkStencilFaceFlags));
                *readStreamPtrPtr += sizeof(VkStencilFaceFlags);
                memcpy((uint32_t*)&reference, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetStencilReference 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)faceMask, (unsigned long long)reference);
                }
                vk->vkCmdSetStencilReference(unboxed_commandBuffer, faceMask, reference);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetStencilReference(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, faceMask,
                        reference);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBindDescriptorSets: {
                android::base::beginTrace("vkCmdBindDescriptorSets decode");
                VkCommandBuffer commandBuffer;
                VkPipelineBindPoint pipelineBindPoint;
                VkPipelineLayout layout;
                uint32_t firstSet;
                uint32_t descriptorSetCount;
                const VkDescriptorSet* pDescriptorSets;
                uint32_t dynamicOffsetCount;
                const uint32_t* pDynamicOffsets;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                memcpy((VkPipelineBindPoint*)&pipelineBindPoint, *readStreamPtrPtr,
                       sizeof(VkPipelineBindPoint));
                *readStreamPtrPtr += sizeof(VkPipelineBindPoint);
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPipelineLayout*)&layout =
                    (VkPipelineLayout)unbox_VkPipelineLayout((VkPipelineLayout)(*&cgen_var_1));
                memcpy((uint32_t*)&firstSet, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&descriptorSetCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pDescriptorSets,
                                    ((descriptorSetCount)) * sizeof(const VkDescriptorSet));
                if (((descriptorSetCount))) {
                    uint8_t* cgen_var_2_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((descriptorSetCount));
                    for (uint32_t k = 0; k < ((descriptorSetCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_2_ptr + k * 8, sizeof(uint64_t));
                        *(((VkDescriptorSet*)pDescriptorSets) + k) =
                            (VkDescriptorSet)unbox_VkDescriptorSet((VkDescriptorSet)tmpval);
                    }
                }
                memcpy((uint32_t*)&dynamicOffsetCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pDynamicOffsets,
                                    ((dynamicOffsetCount)) * sizeof(const uint32_t));
                memcpy((uint32_t*)pDynamicOffsets, *readStreamPtrPtr,
                       ((dynamicOffsetCount)) * sizeof(const uint32_t));
                *readStreamPtrPtr += ((dynamicOffsetCount)) * sizeof(const uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdBindDescriptorSets 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pipelineBindPoint, (unsigned long long)layout,
                            (unsigned long long)firstSet, (unsigned long long)descriptorSetCount,
                            (unsigned long long)pDescriptorSets,
                            (unsigned long long)dynamicOffsetCount,
                            (unsigned long long)pDynamicOffsets);
                }
                m_state->on_vkCmdBindDescriptorSets(
                    &m_pool, commandBuffer, pipelineBindPoint, layout, firstSet, descriptorSetCount,
                    pDescriptorSets, dynamicOffsetCount, pDynamicOffsets);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBindDescriptorSets(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        pipelineBindPoint, layout, firstSet, descriptorSetCount, pDescriptorSets,
                        dynamicOffsetCount, pDynamicOffsets);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBindIndexBuffer: {
                android::base::beginTrace("vkCmdBindIndexBuffer decode");
                VkCommandBuffer commandBuffer;
                VkBuffer buffer;
                VkDeviceSize offset;
                VkIndexType indexType;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&buffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                memcpy((VkDeviceSize*)&offset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((VkIndexType*)&indexType, *readStreamPtrPtr, sizeof(VkIndexType));
                *readStreamPtrPtr += sizeof(VkIndexType);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdBindIndexBuffer 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)buffer,
                            (unsigned long long)offset, (unsigned long long)indexType);
                }
                vk->vkCmdBindIndexBuffer(unboxed_commandBuffer, buffer, offset, indexType);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBindIndexBuffer(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, buffer,
                        offset, indexType);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBindVertexBuffers: {
                android::base::beginTrace("vkCmdBindVertexBuffers decode");
                VkCommandBuffer commandBuffer;
                uint32_t firstBinding;
                uint32_t bindingCount;
                const VkBuffer* pBuffers;
                const VkDeviceSize* pOffsets;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&firstBinding, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&bindingCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pBuffers, ((bindingCount)) * sizeof(const VkBuffer));
                if (((bindingCount))) {
                    uint8_t* cgen_var_1_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((bindingCount));
                    for (uint32_t k = 0; k < ((bindingCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_1_ptr + k * 8, sizeof(uint64_t));
                        *(((VkBuffer*)pBuffers) + k) = (VkBuffer)unbox_VkBuffer((VkBuffer)tmpval);
                    }
                }
                vkReadStream->alloc((void**)&pOffsets,
                                    ((bindingCount)) * sizeof(const VkDeviceSize));
                memcpy((VkDeviceSize*)pOffsets, *readStreamPtrPtr,
                       ((bindingCount)) * sizeof(const VkDeviceSize));
                *readStreamPtrPtr += ((bindingCount)) * sizeof(const VkDeviceSize);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdBindVertexBuffers 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)firstBinding, (unsigned long long)bindingCount,
                            (unsigned long long)pBuffers, (unsigned long long)pOffsets);
                }
                vk->vkCmdBindVertexBuffers(unboxed_commandBuffer, firstBinding, bindingCount,
                                           pBuffers, pOffsets);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBindVertexBuffers(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        firstBinding, bindingCount, pBuffers, pOffsets);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdDraw: {
                android::base::beginTrace("vkCmdDraw decode");
                VkCommandBuffer commandBuffer;
                uint32_t vertexCount;
                uint32_t instanceCount;
                uint32_t firstVertex;
                uint32_t firstInstance;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&vertexCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&instanceCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&firstVertex, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&firstInstance, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdDraw 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)vertexCount, (unsigned long long)instanceCount,
                            (unsigned long long)firstVertex, (unsigned long long)firstInstance);
                }
                vk->vkCmdDraw(unboxed_commandBuffer, vertexCount, instanceCount, firstVertex,
                              firstInstance);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdDraw(snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                                                   commandBuffer, vertexCount, instanceCount,
                                                   firstVertex, firstInstance);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdDrawIndexed: {
                android::base::beginTrace("vkCmdDrawIndexed decode");
                VkCommandBuffer commandBuffer;
                uint32_t indexCount;
                uint32_t instanceCount;
                uint32_t firstIndex;
                int32_t vertexOffset;
                uint32_t firstInstance;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&indexCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&instanceCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&firstIndex, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((int32_t*)&vertexOffset, *readStreamPtrPtr, sizeof(int32_t));
                *readStreamPtrPtr += sizeof(int32_t);
                memcpy((uint32_t*)&firstInstance, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdDrawIndexed 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)indexCount, (unsigned long long)instanceCount,
                            (unsigned long long)firstIndex, (unsigned long long)vertexOffset,
                            (unsigned long long)firstInstance);
                }
                vk->vkCmdDrawIndexed(unboxed_commandBuffer, indexCount, instanceCount, firstIndex,
                                     vertexOffset, firstInstance);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdDrawIndexed(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, indexCount,
                        instanceCount, firstIndex, vertexOffset, firstInstance);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdDrawIndirect: {
                android::base::beginTrace("vkCmdDrawIndirect decode");
                VkCommandBuffer commandBuffer;
                VkBuffer buffer;
                VkDeviceSize offset;
                uint32_t drawCount;
                uint32_t stride;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&buffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                memcpy((VkDeviceSize*)&offset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((uint32_t*)&drawCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&stride, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkCmdDrawIndirect 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)commandBuffer, (unsigned long long)buffer,
                        (unsigned long long)offset, (unsigned long long)drawCount,
                        (unsigned long long)stride);
                }
                vk->vkCmdDrawIndirect(unboxed_commandBuffer, buffer, offset, drawCount, stride);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdDrawIndirect(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, commandBuffer, buffer, offset,
                                                           drawCount, stride);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdDrawIndexedIndirect: {
                android::base::beginTrace("vkCmdDrawIndexedIndirect decode");
                VkCommandBuffer commandBuffer;
                VkBuffer buffer;
                VkDeviceSize offset;
                uint32_t drawCount;
                uint32_t stride;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&buffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                memcpy((VkDeviceSize*)&offset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((uint32_t*)&drawCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&stride, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdDrawIndexedIndirect 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)buffer,
                            (unsigned long long)offset, (unsigned long long)drawCount,
                            (unsigned long long)stride);
                }
                vk->vkCmdDrawIndexedIndirect(unboxed_commandBuffer, buffer, offset, drawCount,
                                             stride);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdDrawIndexedIndirect(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, buffer,
                        offset, drawCount, stride);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdDispatch: {
                android::base::beginTrace("vkCmdDispatch decode");
                VkCommandBuffer commandBuffer;
                uint32_t groupCountX;
                uint32_t groupCountY;
                uint32_t groupCountZ;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&groupCountX, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&groupCountY, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&groupCountZ, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdDispatch 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)groupCountX, (unsigned long long)groupCountY,
                            (unsigned long long)groupCountZ);
                }
                vk->vkCmdDispatch(unboxed_commandBuffer, groupCountX, groupCountY, groupCountZ);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdDispatch(snapshotTraceBegin, snapshotTraceBytes,
                                                       &m_pool, commandBuffer, groupCountX,
                                                       groupCountY, groupCountZ);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdDispatchIndirect: {
                android::base::beginTrace("vkCmdDispatchIndirect decode");
                VkCommandBuffer commandBuffer;
                VkBuffer buffer;
                VkDeviceSize offset;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&buffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                memcpy((VkDeviceSize*)&offset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdDispatchIndirect 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)buffer,
                            (unsigned long long)offset);
                }
                vk->vkCmdDispatchIndirect(unboxed_commandBuffer, buffer, offset);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdDispatchIndirect(snapshotTraceBegin,
                                                               snapshotTraceBytes, &m_pool,
                                                               commandBuffer, buffer, offset);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyBuffer: {
                android::base::beginTrace("vkCmdCopyBuffer decode");
                VkCommandBuffer commandBuffer;
                VkBuffer srcBuffer;
                VkBuffer dstBuffer;
                uint32_t regionCount;
                const VkBufferCopy* pRegions;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&srcBuffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&dstBuffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_2));
                memcpy((uint32_t*)&regionCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pRegions,
                                    ((regionCount)) * sizeof(const VkBufferCopy));
                for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                    reservedunmarshal_VkBufferCopy(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkBufferCopy*)(pRegions + i), readStreamPtrPtr);
                }
                if (pRegions) {
                    for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                        transform_tohost_VkBufferCopy(m_state, (VkBufferCopy*)(pRegions + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdCopyBuffer 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)srcBuffer, (unsigned long long)dstBuffer,
                            (unsigned long long)regionCount, (unsigned long long)pRegions);
                }
                vk->vkCmdCopyBuffer(unboxed_commandBuffer, srcBuffer, dstBuffer, regionCount,
                                    pRegions);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyBuffer(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, commandBuffer, srcBuffer,
                                                         dstBuffer, regionCount, pRegions);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyImage: {
                android::base::beginTrace("vkCmdCopyImage decode");
                VkCommandBuffer commandBuffer;
                VkImage srcImage;
                VkImageLayout srcImageLayout;
                VkImage dstImage;
                VkImageLayout dstImageLayout;
                uint32_t regionCount;
                const VkImageCopy* pRegions;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&srcImage = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_1));
                memcpy((VkImageLayout*)&srcImageLayout, *readStreamPtrPtr, sizeof(VkImageLayout));
                *readStreamPtrPtr += sizeof(VkImageLayout);
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&dstImage = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_2));
                memcpy((VkImageLayout*)&dstImageLayout, *readStreamPtrPtr, sizeof(VkImageLayout));
                *readStreamPtrPtr += sizeof(VkImageLayout);
                memcpy((uint32_t*)&regionCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pRegions, ((regionCount)) * sizeof(const VkImageCopy));
                for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                    reservedunmarshal_VkImageCopy(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                  (VkImageCopy*)(pRegions + i), readStreamPtrPtr);
                }
                if (pRegions) {
                    for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                        transform_tohost_VkImageCopy(m_state, (VkImageCopy*)(pRegions + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdCopyImage 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)srcImage, (unsigned long long)srcImageLayout,
                            (unsigned long long)dstImage, (unsigned long long)dstImageLayout,
                            (unsigned long long)regionCount, (unsigned long long)pRegions);
                }
                m_state->on_vkCmdCopyImage(&m_pool, commandBuffer, srcImage, srcImageLayout,
                                           dstImage, dstImageLayout, regionCount, pRegions);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyImage(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, srcImage,
                        srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBlitImage: {
                android::base::beginTrace("vkCmdBlitImage decode");
                VkCommandBuffer commandBuffer;
                VkImage srcImage;
                VkImageLayout srcImageLayout;
                VkImage dstImage;
                VkImageLayout dstImageLayout;
                uint32_t regionCount;
                const VkImageBlit* pRegions;
                VkFilter filter;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&srcImage = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_1));
                memcpy((VkImageLayout*)&srcImageLayout, *readStreamPtrPtr, sizeof(VkImageLayout));
                *readStreamPtrPtr += sizeof(VkImageLayout);
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&dstImage = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_2));
                memcpy((VkImageLayout*)&dstImageLayout, *readStreamPtrPtr, sizeof(VkImageLayout));
                *readStreamPtrPtr += sizeof(VkImageLayout);
                memcpy((uint32_t*)&regionCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pRegions, ((regionCount)) * sizeof(const VkImageBlit));
                for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                    reservedunmarshal_VkImageBlit(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                  (VkImageBlit*)(pRegions + i), readStreamPtrPtr);
                }
                memcpy((VkFilter*)&filter, *readStreamPtrPtr, sizeof(VkFilter));
                *readStreamPtrPtr += sizeof(VkFilter);
                if (pRegions) {
                    for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                        transform_tohost_VkImageBlit(m_state, (VkImageBlit*)(pRegions + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdBlitImage 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)srcImage, (unsigned long long)srcImageLayout,
                            (unsigned long long)dstImage, (unsigned long long)dstImageLayout,
                            (unsigned long long)regionCount, (unsigned long long)pRegions,
                            (unsigned long long)filter);
                }
                vk->vkCmdBlitImage(unboxed_commandBuffer, srcImage, srcImageLayout, dstImage,
                                   dstImageLayout, regionCount, pRegions, filter);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBlitImage(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, srcImage,
                        srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions, filter);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyBufferToImage: {
                android::base::beginTrace("vkCmdCopyBufferToImage decode");
                VkCommandBuffer commandBuffer;
                VkBuffer srcBuffer;
                VkImage dstImage;
                VkImageLayout dstImageLayout;
                uint32_t regionCount;
                const VkBufferImageCopy* pRegions;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&srcBuffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&dstImage = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_2));
                memcpy((VkImageLayout*)&dstImageLayout, *readStreamPtrPtr, sizeof(VkImageLayout));
                *readStreamPtrPtr += sizeof(VkImageLayout);
                memcpy((uint32_t*)&regionCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pRegions,
                                    ((regionCount)) * sizeof(const VkBufferImageCopy));
                for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                    reservedunmarshal_VkBufferImageCopy(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                        (VkBufferImageCopy*)(pRegions + i),
                                                        readStreamPtrPtr);
                }
                if (pRegions) {
                    for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                        transform_tohost_VkBufferImageCopy(m_state,
                                                           (VkBufferImageCopy*)(pRegions + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdCopyBufferToImage 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)srcBuffer, (unsigned long long)dstImage,
                            (unsigned long long)dstImageLayout, (unsigned long long)regionCount,
                            (unsigned long long)pRegions);
                }
                m_state->on_vkCmdCopyBufferToImage(&m_pool, commandBuffer, srcBuffer, dstImage,
                                                   dstImageLayout, regionCount, pRegions, context);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyBufferToImage(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, srcBuffer,
                        dstImage, dstImageLayout, regionCount, pRegions);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyImageToBuffer: {
                android::base::beginTrace("vkCmdCopyImageToBuffer decode");
                VkCommandBuffer commandBuffer;
                VkImage srcImage;
                VkImageLayout srcImageLayout;
                VkBuffer dstBuffer;
                uint32_t regionCount;
                const VkBufferImageCopy* pRegions;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&srcImage = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_1));
                memcpy((VkImageLayout*)&srcImageLayout, *readStreamPtrPtr, sizeof(VkImageLayout));
                *readStreamPtrPtr += sizeof(VkImageLayout);
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&dstBuffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_2));
                memcpy((uint32_t*)&regionCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pRegions,
                                    ((regionCount)) * sizeof(const VkBufferImageCopy));
                for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                    reservedunmarshal_VkBufferImageCopy(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                        (VkBufferImageCopy*)(pRegions + i),
                                                        readStreamPtrPtr);
                }
                if (pRegions) {
                    for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                        transform_tohost_VkBufferImageCopy(m_state,
                                                           (VkBufferImageCopy*)(pRegions + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdCopyImageToBuffer 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)srcImage, (unsigned long long)srcImageLayout,
                            (unsigned long long)dstBuffer, (unsigned long long)regionCount,
                            (unsigned long long)pRegions);
                }
                m_state->on_vkCmdCopyImageToBuffer(&m_pool, commandBuffer, srcImage, srcImageLayout,
                                                   dstBuffer, regionCount, pRegions);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyImageToBuffer(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, srcImage,
                        srcImageLayout, dstBuffer, regionCount, pRegions);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdUpdateBuffer: {
                android::base::beginTrace("vkCmdUpdateBuffer decode");
                VkCommandBuffer commandBuffer;
                VkBuffer dstBuffer;
                VkDeviceSize dstOffset;
                VkDeviceSize dataSize;
                const void* pData;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&dstBuffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                memcpy((VkDeviceSize*)&dstOffset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((VkDeviceSize*)&dataSize, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                vkReadStream->alloc((void**)&pData, ((dataSize)) * sizeof(const uint8_t));
                memcpy((void*)pData, *readStreamPtrPtr, ((dataSize)) * sizeof(const uint8_t));
                *readStreamPtrPtr += ((dataSize)) * sizeof(const uint8_t);
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkCmdUpdateBuffer 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)commandBuffer, (unsigned long long)dstBuffer,
                        (unsigned long long)dstOffset, (unsigned long long)dataSize,
                        (unsigned long long)pData);
                }
                vk->vkCmdUpdateBuffer(unboxed_commandBuffer, dstBuffer, dstOffset, dataSize, pData);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdUpdateBuffer(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, commandBuffer, dstBuffer,
                                                           dstOffset, dataSize, pData);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdFillBuffer: {
                android::base::beginTrace("vkCmdFillBuffer decode");
                VkCommandBuffer commandBuffer;
                VkBuffer dstBuffer;
                VkDeviceSize dstOffset;
                VkDeviceSize size;
                uint32_t data;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&dstBuffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                memcpy((VkDeviceSize*)&dstOffset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((VkDeviceSize*)&size, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((uint32_t*)&data, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdFillBuffer 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)dstBuffer, (unsigned long long)dstOffset,
                            (unsigned long long)size, (unsigned long long)data);
                }
                vk->vkCmdFillBuffer(unboxed_commandBuffer, dstBuffer, dstOffset, size, data);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdFillBuffer(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, commandBuffer, dstBuffer,
                                                         dstOffset, size, data);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdClearColorImage: {
                android::base::beginTrace("vkCmdClearColorImage decode");
                VkCommandBuffer commandBuffer;
                VkImage image;
                VkImageLayout imageLayout;
                const VkClearColorValue* pColor;
                uint32_t rangeCount;
                const VkImageSubresourceRange* pRanges;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&image = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_1));
                memcpy((VkImageLayout*)&imageLayout, *readStreamPtrPtr, sizeof(VkImageLayout));
                *readStreamPtrPtr += sizeof(VkImageLayout);
                vkReadStream->alloc((void**)&pColor, sizeof(const VkClearColorValue));
                reservedunmarshal_VkClearColorValue(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkClearColorValue*)(pColor), readStreamPtrPtr);
                memcpy((uint32_t*)&rangeCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pRanges,
                                    ((rangeCount)) * sizeof(const VkImageSubresourceRange));
                for (uint32_t i = 0; i < (uint32_t)((rangeCount)); ++i) {
                    reservedunmarshal_VkImageSubresourceRange(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkImageSubresourceRange*)(pRanges + i), readStreamPtrPtr);
                }
                if (pColor) {
                    transform_tohost_VkClearColorValue(m_state, (VkClearColorValue*)(pColor));
                }
                if (pRanges) {
                    for (uint32_t i = 0; i < (uint32_t)((rangeCount)); ++i) {
                        transform_tohost_VkImageSubresourceRange(
                            m_state, (VkImageSubresourceRange*)(pRanges + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdClearColorImage 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)image,
                            (unsigned long long)imageLayout, (unsigned long long)pColor,
                            (unsigned long long)rangeCount, (unsigned long long)pRanges);
                }
                vk->vkCmdClearColorImage(unboxed_commandBuffer, image, imageLayout, pColor,
                                         rangeCount, pRanges);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdClearColorImage(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, image,
                        imageLayout, pColor, rangeCount, pRanges);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdClearDepthStencilImage: {
                android::base::beginTrace("vkCmdClearDepthStencilImage decode");
                VkCommandBuffer commandBuffer;
                VkImage image;
                VkImageLayout imageLayout;
                const VkClearDepthStencilValue* pDepthStencil;
                uint32_t rangeCount;
                const VkImageSubresourceRange* pRanges;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&image = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_1));
                memcpy((VkImageLayout*)&imageLayout, *readStreamPtrPtr, sizeof(VkImageLayout));
                *readStreamPtrPtr += sizeof(VkImageLayout);
                vkReadStream->alloc((void**)&pDepthStencil, sizeof(const VkClearDepthStencilValue));
                reservedunmarshal_VkClearDepthStencilValue(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkClearDepthStencilValue*)(pDepthStencil), readStreamPtrPtr);
                memcpy((uint32_t*)&rangeCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pRanges,
                                    ((rangeCount)) * sizeof(const VkImageSubresourceRange));
                for (uint32_t i = 0; i < (uint32_t)((rangeCount)); ++i) {
                    reservedunmarshal_VkImageSubresourceRange(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkImageSubresourceRange*)(pRanges + i), readStreamPtrPtr);
                }
                if (pDepthStencil) {
                    transform_tohost_VkClearDepthStencilValue(
                        m_state, (VkClearDepthStencilValue*)(pDepthStencil));
                }
                if (pRanges) {
                    for (uint32_t i = 0; i < (uint32_t)((rangeCount)); ++i) {
                        transform_tohost_VkImageSubresourceRange(
                            m_state, (VkImageSubresourceRange*)(pRanges + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdClearDepthStencilImage 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)image,
                            (unsigned long long)imageLayout, (unsigned long long)pDepthStencil,
                            (unsigned long long)rangeCount, (unsigned long long)pRanges);
                }
                vk->vkCmdClearDepthStencilImage(unboxed_commandBuffer, image, imageLayout,
                                                pDepthStencil, rangeCount, pRanges);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdClearDepthStencilImage(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, image,
                        imageLayout, pDepthStencil, rangeCount, pRanges);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdClearAttachments: {
                android::base::beginTrace("vkCmdClearAttachments decode");
                VkCommandBuffer commandBuffer;
                uint32_t attachmentCount;
                const VkClearAttachment* pAttachments;
                uint32_t rectCount;
                const VkClearRect* pRects;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&attachmentCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pAttachments,
                                    ((attachmentCount)) * sizeof(const VkClearAttachment));
                for (uint32_t i = 0; i < (uint32_t)((attachmentCount)); ++i) {
                    reservedunmarshal_VkClearAttachment(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                        (VkClearAttachment*)(pAttachments + i),
                                                        readStreamPtrPtr);
                }
                memcpy((uint32_t*)&rectCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pRects, ((rectCount)) * sizeof(const VkClearRect));
                for (uint32_t i = 0; i < (uint32_t)((rectCount)); ++i) {
                    reservedunmarshal_VkClearRect(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                  (VkClearRect*)(pRects + i), readStreamPtrPtr);
                }
                if (pAttachments) {
                    for (uint32_t i = 0; i < (uint32_t)((attachmentCount)); ++i) {
                        transform_tohost_VkClearAttachment(m_state,
                                                           (VkClearAttachment*)(pAttachments + i));
                    }
                }
                if (pRects) {
                    for (uint32_t i = 0; i < (uint32_t)((rectCount)); ++i) {
                        transform_tohost_VkClearRect(m_state, (VkClearRect*)(pRects + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdClearAttachments 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)attachmentCount, (unsigned long long)pAttachments,
                            (unsigned long long)rectCount, (unsigned long long)pRects);
                }
                vk->vkCmdClearAttachments(unboxed_commandBuffer, attachmentCount, pAttachments,
                                          rectCount, pRects);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdClearAttachments(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        attachmentCount, pAttachments, rectCount, pRects);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdResolveImage: {
                android::base::beginTrace("vkCmdResolveImage decode");
                VkCommandBuffer commandBuffer;
                VkImage srcImage;
                VkImageLayout srcImageLayout;
                VkImage dstImage;
                VkImageLayout dstImageLayout;
                uint32_t regionCount;
                const VkImageResolve* pRegions;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&srcImage = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_1));
                memcpy((VkImageLayout*)&srcImageLayout, *readStreamPtrPtr, sizeof(VkImageLayout));
                *readStreamPtrPtr += sizeof(VkImageLayout);
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkImage*)&dstImage = (VkImage)unbox_VkImage((VkImage)(*&cgen_var_2));
                memcpy((VkImageLayout*)&dstImageLayout, *readStreamPtrPtr, sizeof(VkImageLayout));
                *readStreamPtrPtr += sizeof(VkImageLayout);
                memcpy((uint32_t*)&regionCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pRegions,
                                    ((regionCount)) * sizeof(const VkImageResolve));
                for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                    reservedunmarshal_VkImageResolve(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkImageResolve*)(pRegions + i),
                                                     readStreamPtrPtr);
                }
                if (pRegions) {
                    for (uint32_t i = 0; i < (uint32_t)((regionCount)); ++i) {
                        transform_tohost_VkImageResolve(m_state, (VkImageResolve*)(pRegions + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdResolveImage 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)srcImage, (unsigned long long)srcImageLayout,
                            (unsigned long long)dstImage, (unsigned long long)dstImageLayout,
                            (unsigned long long)regionCount, (unsigned long long)pRegions);
                }
                vk->vkCmdResolveImage(unboxed_commandBuffer, srcImage, srcImageLayout, dstImage,
                                      dstImageLayout, regionCount, pRegions);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdResolveImage(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, srcImage,
                        srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetEvent: {
                android::base::beginTrace("vkCmdSetEvent decode");
                VkCommandBuffer commandBuffer;
                VkEvent event;
                VkPipelineStageFlags stageMask;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkEvent*)&event = (VkEvent)unbox_VkEvent((VkEvent)(*&cgen_var_1));
                memcpy((VkPipelineStageFlags*)&stageMask, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetEvent 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)event,
                            (unsigned long long)stageMask);
                }
                vk->vkCmdSetEvent(unboxed_commandBuffer, event, stageMask);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetEvent(snapshotTraceBegin, snapshotTraceBytes,
                                                       &m_pool, commandBuffer, event, stageMask);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdResetEvent: {
                android::base::beginTrace("vkCmdResetEvent decode");
                VkCommandBuffer commandBuffer;
                VkEvent event;
                VkPipelineStageFlags stageMask;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkEvent*)&event = (VkEvent)unbox_VkEvent((VkEvent)(*&cgen_var_1));
                memcpy((VkPipelineStageFlags*)&stageMask, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdResetEvent 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)event,
                            (unsigned long long)stageMask);
                }
                vk->vkCmdResetEvent(unboxed_commandBuffer, event, stageMask);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdResetEvent(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, commandBuffer, event, stageMask);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdWaitEvents: {
                android::base::beginTrace("vkCmdWaitEvents decode");
                VkCommandBuffer commandBuffer;
                uint32_t eventCount;
                const VkEvent* pEvents;
                VkPipelineStageFlags srcStageMask;
                VkPipelineStageFlags dstStageMask;
                uint32_t memoryBarrierCount;
                const VkMemoryBarrier* pMemoryBarriers;
                uint32_t bufferMemoryBarrierCount;
                const VkBufferMemoryBarrier* pBufferMemoryBarriers;
                uint32_t imageMemoryBarrierCount;
                const VkImageMemoryBarrier* pImageMemoryBarriers;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&eventCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pEvents, ((eventCount)) * sizeof(const VkEvent));
                if (((eventCount))) {
                    uint8_t* cgen_var_1_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((eventCount));
                    for (uint32_t k = 0; k < ((eventCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_1_ptr + k * 8, sizeof(uint64_t));
                        *(((VkEvent*)pEvents) + k) = (VkEvent)unbox_VkEvent((VkEvent)tmpval);
                    }
                }
                memcpy((VkPipelineStageFlags*)&srcStageMask, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags);
                memcpy((VkPipelineStageFlags*)&dstStageMask, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags);
                memcpy((uint32_t*)&memoryBarrierCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pMemoryBarriers,
                                    ((memoryBarrierCount)) * sizeof(const VkMemoryBarrier));
                for (uint32_t i = 0; i < (uint32_t)((memoryBarrierCount)); ++i) {
                    reservedunmarshal_VkMemoryBarrier(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkMemoryBarrier*)(pMemoryBarriers + i),
                                                      readStreamPtrPtr);
                }
                memcpy((uint32_t*)&bufferMemoryBarrierCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc(
                    (void**)&pBufferMemoryBarriers,
                    ((bufferMemoryBarrierCount)) * sizeof(const VkBufferMemoryBarrier));
                for (uint32_t i = 0; i < (uint32_t)((bufferMemoryBarrierCount)); ++i) {
                    reservedunmarshal_VkBufferMemoryBarrier(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkBufferMemoryBarrier*)(pBufferMemoryBarriers + i), readStreamPtrPtr);
                }
                memcpy((uint32_t*)&imageMemoryBarrierCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc(
                    (void**)&pImageMemoryBarriers,
                    ((imageMemoryBarrierCount)) * sizeof(const VkImageMemoryBarrier));
                for (uint32_t i = 0; i < (uint32_t)((imageMemoryBarrierCount)); ++i) {
                    reservedunmarshal_VkImageMemoryBarrier(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkImageMemoryBarrier*)(pImageMemoryBarriers + i), readStreamPtrPtr);
                }
                if (pMemoryBarriers) {
                    for (uint32_t i = 0; i < (uint32_t)((memoryBarrierCount)); ++i) {
                        transform_tohost_VkMemoryBarrier(m_state,
                                                         (VkMemoryBarrier*)(pMemoryBarriers + i));
                    }
                }
                if (pBufferMemoryBarriers) {
                    for (uint32_t i = 0; i < (uint32_t)((bufferMemoryBarrierCount)); ++i) {
                        transform_tohost_VkBufferMemoryBarrier(
                            m_state, (VkBufferMemoryBarrier*)(pBufferMemoryBarriers + i));
                    }
                }
                if (pImageMemoryBarriers) {
                    for (uint32_t i = 0; i < (uint32_t)((imageMemoryBarrierCount)); ++i) {
                        transform_tohost_VkImageMemoryBarrier(
                            m_state, (VkImageMemoryBarrier*)(pImageMemoryBarriers + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdWaitEvents 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)eventCount, (unsigned long long)pEvents,
                            (unsigned long long)srcStageMask, (unsigned long long)dstStageMask,
                            (unsigned long long)memoryBarrierCount,
                            (unsigned long long)pMemoryBarriers,
                            (unsigned long long)bufferMemoryBarrierCount,
                            (unsigned long long)pBufferMemoryBarriers,
                            (unsigned long long)imageMemoryBarrierCount,
                            (unsigned long long)pImageMemoryBarriers);
                }
                vk->vkCmdWaitEvents(unboxed_commandBuffer, eventCount, pEvents, srcStageMask,
                                    dstStageMask, memoryBarrierCount, pMemoryBarriers,
                                    bufferMemoryBarrierCount, pBufferMemoryBarriers,
                                    imageMemoryBarrierCount, pImageMemoryBarriers);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdWaitEvents(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, eventCount,
                        pEvents, srcStageMask, dstStageMask, memoryBarrierCount, pMemoryBarriers,
                        bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount,
                        pImageMemoryBarriers);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdPipelineBarrier: {
                android::base::beginTrace("vkCmdPipelineBarrier decode");
                VkCommandBuffer commandBuffer;
                VkPipelineStageFlags srcStageMask;
                VkPipelineStageFlags dstStageMask;
                VkDependencyFlags dependencyFlags;
                uint32_t memoryBarrierCount;
                const VkMemoryBarrier* pMemoryBarriers;
                uint32_t bufferMemoryBarrierCount;
                const VkBufferMemoryBarrier* pBufferMemoryBarriers;
                uint32_t imageMemoryBarrierCount;
                const VkImageMemoryBarrier* pImageMemoryBarriers;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                memcpy((VkPipelineStageFlags*)&srcStageMask, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags);
                memcpy((VkPipelineStageFlags*)&dstStageMask, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags);
                memcpy((VkDependencyFlags*)&dependencyFlags, *readStreamPtrPtr,
                       sizeof(VkDependencyFlags));
                *readStreamPtrPtr += sizeof(VkDependencyFlags);
                memcpy((uint32_t*)&memoryBarrierCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pMemoryBarriers,
                                    ((memoryBarrierCount)) * sizeof(const VkMemoryBarrier));
                for (uint32_t i = 0; i < (uint32_t)((memoryBarrierCount)); ++i) {
                    reservedunmarshal_VkMemoryBarrier(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkMemoryBarrier*)(pMemoryBarriers + i),
                                                      readStreamPtrPtr);
                }
                memcpy((uint32_t*)&bufferMemoryBarrierCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc(
                    (void**)&pBufferMemoryBarriers,
                    ((bufferMemoryBarrierCount)) * sizeof(const VkBufferMemoryBarrier));
                for (uint32_t i = 0; i < (uint32_t)((bufferMemoryBarrierCount)); ++i) {
                    reservedunmarshal_VkBufferMemoryBarrier(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkBufferMemoryBarrier*)(pBufferMemoryBarriers + i), readStreamPtrPtr);
                }
                memcpy((uint32_t*)&imageMemoryBarrierCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc(
                    (void**)&pImageMemoryBarriers,
                    ((imageMemoryBarrierCount)) * sizeof(const VkImageMemoryBarrier));
                for (uint32_t i = 0; i < (uint32_t)((imageMemoryBarrierCount)); ++i) {
                    reservedunmarshal_VkImageMemoryBarrier(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkImageMemoryBarrier*)(pImageMemoryBarriers + i), readStreamPtrPtr);
                }
                if (pMemoryBarriers) {
                    for (uint32_t i = 0; i < (uint32_t)((memoryBarrierCount)); ++i) {
                        transform_tohost_VkMemoryBarrier(m_state,
                                                         (VkMemoryBarrier*)(pMemoryBarriers + i));
                    }
                }
                if (pBufferMemoryBarriers) {
                    for (uint32_t i = 0; i < (uint32_t)((bufferMemoryBarrierCount)); ++i) {
                        transform_tohost_VkBufferMemoryBarrier(
                            m_state, (VkBufferMemoryBarrier*)(pBufferMemoryBarriers + i));
                    }
                }
                if (pImageMemoryBarriers) {
                    for (uint32_t i = 0; i < (uint32_t)((imageMemoryBarrierCount)); ++i) {
                        transform_tohost_VkImageMemoryBarrier(
                            m_state, (VkImageMemoryBarrier*)(pImageMemoryBarriers + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdPipelineBarrier 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)srcStageMask, (unsigned long long)dstStageMask,
                            (unsigned long long)dependencyFlags,
                            (unsigned long long)memoryBarrierCount,
                            (unsigned long long)pMemoryBarriers,
                            (unsigned long long)bufferMemoryBarrierCount,
                            (unsigned long long)pBufferMemoryBarriers,
                            (unsigned long long)imageMemoryBarrierCount,
                            (unsigned long long)pImageMemoryBarriers);
                }
                m_state->on_vkCmdPipelineBarrier(
                    &m_pool, commandBuffer, srcStageMask, dstStageMask, dependencyFlags,
                    memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount,
                    pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdPipelineBarrier(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        srcStageMask, dstStageMask, dependencyFlags, memoryBarrierCount,
                        pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers,
                        imageMemoryBarrierCount, pImageMemoryBarriers);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBeginQuery: {
                android::base::beginTrace("vkCmdBeginQuery decode");
                VkCommandBuffer commandBuffer;
                VkQueryPool queryPool;
                uint32_t query;
                VkQueryControlFlags flags;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueryPool*)&queryPool =
                    (VkQueryPool)unbox_VkQueryPool((VkQueryPool)(*&cgen_var_1));
                memcpy((uint32_t*)&query, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((VkQueryControlFlags*)&flags, *readStreamPtrPtr,
                       sizeof(VkQueryControlFlags));
                *readStreamPtrPtr += sizeof(VkQueryControlFlags);
                if (m_logCalls) {
                    fprintf(
                        stderr, "stream %p: call vkCmdBeginQuery 0x%llx 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)commandBuffer, (unsigned long long)queryPool,
                        (unsigned long long)query, (unsigned long long)flags);
                }
                vk->vkCmdBeginQuery(unboxed_commandBuffer, queryPool, query, flags);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBeginQuery(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, commandBuffer, queryPool, query,
                                                         flags);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdEndQuery: {
                android::base::beginTrace("vkCmdEndQuery decode");
                VkCommandBuffer commandBuffer;
                VkQueryPool queryPool;
                uint32_t query;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueryPool*)&queryPool =
                    (VkQueryPool)unbox_VkQueryPool((VkQueryPool)(*&cgen_var_1));
                memcpy((uint32_t*)&query, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdEndQuery 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)queryPool, (unsigned long long)query);
                }
                vk->vkCmdEndQuery(unboxed_commandBuffer, queryPool, query);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdEndQuery(snapshotTraceBegin, snapshotTraceBytes,
                                                       &m_pool, commandBuffer, queryPool, query);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdResetQueryPool: {
                android::base::beginTrace("vkCmdResetQueryPool decode");
                VkCommandBuffer commandBuffer;
                VkQueryPool queryPool;
                uint32_t firstQuery;
                uint32_t queryCount;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueryPool*)&queryPool =
                    (VkQueryPool)unbox_VkQueryPool((VkQueryPool)(*&cgen_var_1));
                memcpy((uint32_t*)&firstQuery, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&queryCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdResetQueryPool 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)queryPool, (unsigned long long)firstQuery,
                            (unsigned long long)queryCount);
                }
                vk->vkCmdResetQueryPool(unboxed_commandBuffer, queryPool, firstQuery, queryCount);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdResetQueryPool(snapshotTraceBegin, snapshotTraceBytes,
                                                             &m_pool, commandBuffer, queryPool,
                                                             firstQuery, queryCount);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdWriteTimestamp: {
                android::base::beginTrace("vkCmdWriteTimestamp decode");
                VkCommandBuffer commandBuffer;
                VkPipelineStageFlagBits pipelineStage;
                VkQueryPool queryPool;
                uint32_t query;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkPipelineStageFlagBits*)&pipelineStage, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlagBits));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlagBits);
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueryPool*)&queryPool =
                    (VkQueryPool)unbox_VkQueryPool((VkQueryPool)(*&cgen_var_1));
                memcpy((uint32_t*)&query, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdWriteTimestamp 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pipelineStage, (unsigned long long)queryPool,
                            (unsigned long long)query);
                }
                vk->vkCmdWriteTimestamp(unboxed_commandBuffer, pipelineStage, queryPool, query);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdWriteTimestamp(snapshotTraceBegin, snapshotTraceBytes,
                                                             &m_pool, commandBuffer, pipelineStage,
                                                             queryPool, query);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyQueryPoolResults: {
                android::base::beginTrace("vkCmdCopyQueryPoolResults decode");
                VkCommandBuffer commandBuffer;
                VkQueryPool queryPool;
                uint32_t firstQuery;
                uint32_t queryCount;
                VkBuffer dstBuffer;
                VkDeviceSize dstOffset;
                VkDeviceSize stride;
                VkQueryResultFlags flags;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueryPool*)&queryPool =
                    (VkQueryPool)unbox_VkQueryPool((VkQueryPool)(*&cgen_var_1));
                memcpy((uint32_t*)&firstQuery, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&queryCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&dstBuffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_2));
                memcpy((VkDeviceSize*)&dstOffset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((VkDeviceSize*)&stride, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((VkQueryResultFlags*)&flags, *readStreamPtrPtr, sizeof(VkQueryResultFlags));
                *readStreamPtrPtr += sizeof(VkQueryResultFlags);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdCopyQueryPoolResults 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)queryPool, (unsigned long long)firstQuery,
                            (unsigned long long)queryCount, (unsigned long long)dstBuffer,
                            (unsigned long long)dstOffset, (unsigned long long)stride,
                            (unsigned long long)flags);
                }
                vk->vkCmdCopyQueryPoolResults(unboxed_commandBuffer, queryPool, firstQuery,
                                              queryCount, dstBuffer, dstOffset, stride, flags);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyQueryPoolResults(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, queryPool,
                        firstQuery, queryCount, dstBuffer, dstOffset, stride, flags);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdPushConstants: {
                android::base::beginTrace("vkCmdPushConstants decode");
                VkCommandBuffer commandBuffer;
                VkPipelineLayout layout;
                VkShaderStageFlags stageFlags;
                uint32_t offset;
                uint32_t size;
                const void* pValues;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPipelineLayout*)&layout =
                    (VkPipelineLayout)unbox_VkPipelineLayout((VkPipelineLayout)(*&cgen_var_1));
                memcpy((VkShaderStageFlags*)&stageFlags, *readStreamPtrPtr,
                       sizeof(VkShaderStageFlags));
                *readStreamPtrPtr += sizeof(VkShaderStageFlags);
                memcpy((uint32_t*)&offset, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&size, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pValues, ((size)) * sizeof(const uint8_t));
                memcpy((void*)pValues, *readStreamPtrPtr, ((size)) * sizeof(const uint8_t));
                *readStreamPtrPtr += ((size)) * sizeof(const uint8_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdPushConstants 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)layout,
                            (unsigned long long)stageFlags, (unsigned long long)offset,
                            (unsigned long long)size, (unsigned long long)pValues);
                }
                vk->vkCmdPushConstants(unboxed_commandBuffer, layout, stageFlags, offset, size,
                                       pValues);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdPushConstants(snapshotTraceBegin, snapshotTraceBytes,
                                                            &m_pool, commandBuffer, layout,
                                                            stageFlags, offset, size, pValues);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBeginRenderPass: {
                android::base::beginTrace("vkCmdBeginRenderPass decode");
                VkCommandBuffer commandBuffer;
                const VkRenderPassBeginInfo* pRenderPassBegin;
                VkSubpassContents contents;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pRenderPassBegin, sizeof(const VkRenderPassBeginInfo));
                reservedunmarshal_VkRenderPassBeginInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                        (VkRenderPassBeginInfo*)(pRenderPassBegin),
                                                        readStreamPtrPtr);
                memcpy((VkSubpassContents*)&contents, *readStreamPtrPtr, sizeof(VkSubpassContents));
                *readStreamPtrPtr += sizeof(VkSubpassContents);
                if (pRenderPassBegin) {
                    transform_tohost_VkRenderPassBeginInfo(
                        m_state, (VkRenderPassBeginInfo*)(pRenderPassBegin));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdBeginRenderPass 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pRenderPassBegin, (unsigned long long)contents);
                }
                vk->vkCmdBeginRenderPass(unboxed_commandBuffer, pRenderPassBegin, contents);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBeginRenderPass(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        pRenderPassBegin, contents);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdNextSubpass: {
                android::base::beginTrace("vkCmdNextSubpass decode");
                VkCommandBuffer commandBuffer;
                VkSubpassContents contents;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkSubpassContents*)&contents, *readStreamPtrPtr, sizeof(VkSubpassContents));
                *readStreamPtrPtr += sizeof(VkSubpassContents);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdNextSubpass 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer, (unsigned long long)contents);
                }
                vk->vkCmdNextSubpass(unboxed_commandBuffer, contents);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdNextSubpass(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, commandBuffer, contents);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdEndRenderPass: {
                android::base::beginTrace("vkCmdEndRenderPass decode");
                VkCommandBuffer commandBuffer;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdEndRenderPass 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer);
                }
                vk->vkCmdEndRenderPass(unboxed_commandBuffer);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdEndRenderPass(snapshotTraceBegin, snapshotTraceBytes,
                                                            &m_pool, commandBuffer);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdExecuteCommands: {
                android::base::beginTrace("vkCmdExecuteCommands decode");
                VkCommandBuffer commandBuffer;
                uint32_t commandBufferCount;
                const VkCommandBuffer* pCommandBuffers;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                memcpy((uint32_t*)&commandBufferCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pCommandBuffers,
                                    ((commandBufferCount)) * sizeof(const VkCommandBuffer));
                if (((commandBufferCount))) {
                    uint8_t* cgen_var_1_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((commandBufferCount));
                    for (uint32_t k = 0; k < ((commandBufferCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_1_ptr + k * 8, sizeof(uint64_t));
                        *(((VkCommandBuffer*)pCommandBuffers) + k) =
                            (VkCommandBuffer)unbox_VkCommandBuffer((VkCommandBuffer)tmpval);
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdExecuteCommands 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)commandBufferCount,
                            (unsigned long long)pCommandBuffers);
                }
                m_state->on_vkCmdExecuteCommands(&m_pool, commandBuffer, commandBufferCount,
                                                 pCommandBuffers);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdExecuteCommands(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        commandBufferCount, pCommandBuffers);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_VERSION_1_1
            case OP_vkEnumerateInstanceVersion: {
                android::base::beginTrace("vkEnumerateInstanceVersion decode");
                uint32_t* pApiVersion;
                // Begin manual dispatchable handle unboxing for pApiVersion;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pApiVersion, sizeof(uint32_t));
                memcpy((uint32_t*)pApiVersion, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkEnumerateInstanceVersion 0x%llx \n",
                            ioStream, (unsigned long long)pApiVersion);
                }
                VkResult vkEnumerateInstanceVersion_VkResult_return = (VkResult)0;
                vkEnumerateInstanceVersion_VkResult_return =
                    m_state->on_vkEnumerateInstanceVersion(&m_pool, pApiVersion);
                if ((vkEnumerateInstanceVersion_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkEnumerateInstanceVersion_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                vkStream->write((uint32_t*)pApiVersion, sizeof(uint32_t));
                vkStream->write(&vkEnumerateInstanceVersion_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkEnumerateInstanceVersion(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkEnumerateInstanceVersion_VkResult_return, pApiVersion);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkBindBufferMemory2: {
                android::base::beginTrace("vkBindBufferMemory2 decode");
                VkDevice device;
                uint32_t bindInfoCount;
                const VkBindBufferMemoryInfo* pBindInfos;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                memcpy((uint32_t*)&bindInfoCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pBindInfos,
                                    ((bindInfoCount)) * sizeof(const VkBindBufferMemoryInfo));
                for (uint32_t i = 0; i < (uint32_t)((bindInfoCount)); ++i) {
                    reservedunmarshal_VkBindBufferMemoryInfo(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkBindBufferMemoryInfo*)(pBindInfos + i), readStreamPtrPtr);
                }
                if (pBindInfos) {
                    for (uint32_t i = 0; i < (uint32_t)((bindInfoCount)); ++i) {
                        transform_tohost_VkBindBufferMemoryInfo(
                            m_state, (VkBindBufferMemoryInfo*)(pBindInfos + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkBindBufferMemory2 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)bindInfoCount,
                            (unsigned long long)pBindInfos);
                }
                VkResult vkBindBufferMemory2_VkResult_return = (VkResult)0;
                vkBindBufferMemory2_VkResult_return =
                    m_state->on_vkBindBufferMemory2(&m_pool, device, bindInfoCount, pBindInfos);
                if ((vkBindBufferMemory2_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkBindBufferMemory2_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkBindBufferMemory2_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkBindBufferMemory2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkBindBufferMemory2_VkResult_return, device, bindInfoCount, pBindInfos);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkBindImageMemory2: {
                android::base::beginTrace("vkBindImageMemory2 decode");
                VkDevice device;
                uint32_t bindInfoCount;
                const VkBindImageMemoryInfo* pBindInfos;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                memcpy((uint32_t*)&bindInfoCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pBindInfos,
                                    ((bindInfoCount)) * sizeof(const VkBindImageMemoryInfo));
                for (uint32_t i = 0; i < (uint32_t)((bindInfoCount)); ++i) {
                    reservedunmarshal_VkBindImageMemoryInfo(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkBindImageMemoryInfo*)(pBindInfos + i), readStreamPtrPtr);
                }
                if (pBindInfos) {
                    for (uint32_t i = 0; i < (uint32_t)((bindInfoCount)); ++i) {
                        transform_tohost_VkBindImageMemoryInfo(
                            m_state, (VkBindImageMemoryInfo*)(pBindInfos + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkBindImageMemory2 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)bindInfoCount,
                            (unsigned long long)pBindInfos);
                }
                VkResult vkBindImageMemory2_VkResult_return = (VkResult)0;
                vkBindImageMemory2_VkResult_return =
                    m_state->on_vkBindImageMemory2(&m_pool, device, bindInfoCount, pBindInfos);
                if ((vkBindImageMemory2_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkBindImageMemory2_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkBindImageMemory2_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkBindImageMemory2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkBindImageMemory2_VkResult_return, device, bindInfoCount, pBindInfos);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceGroupPeerMemoryFeatures: {
                android::base::beginTrace("vkGetDeviceGroupPeerMemoryFeatures decode");
                VkDevice device;
                uint32_t heapIndex;
                uint32_t localDeviceIndex;
                uint32_t remoteDeviceIndex;
                VkPeerMemoryFeatureFlags* pPeerMemoryFeatures;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                memcpy((uint32_t*)&heapIndex, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&localDeviceIndex, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&remoteDeviceIndex, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                // Begin manual dispatchable handle unboxing for pPeerMemoryFeatures;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pPeerMemoryFeatures, sizeof(VkPeerMemoryFeatureFlags));
                memcpy((VkPeerMemoryFeatureFlags*)pPeerMemoryFeatures, *readStreamPtrPtr,
                       sizeof(VkPeerMemoryFeatureFlags));
                *readStreamPtrPtr += sizeof(VkPeerMemoryFeatureFlags);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetDeviceGroupPeerMemoryFeatures 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)heapIndex,
                            (unsigned long long)localDeviceIndex,
                            (unsigned long long)remoteDeviceIndex,
                            (unsigned long long)pPeerMemoryFeatures);
                }
                vk->vkGetDeviceGroupPeerMemoryFeatures(unboxed_device, heapIndex, localDeviceIndex,
                                                       remoteDeviceIndex, pPeerMemoryFeatures);
                vkStream->unsetHandleMapping();
                vkStream->write((VkPeerMemoryFeatureFlags*)pPeerMemoryFeatures,
                                sizeof(VkPeerMemoryFeatureFlags));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceGroupPeerMemoryFeatures(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, heapIndex,
                        localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetDeviceMask: {
                android::base::beginTrace("vkCmdSetDeviceMask decode");
                VkCommandBuffer commandBuffer;
                uint32_t deviceMask;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&deviceMask, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetDeviceMask 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer, (unsigned long long)deviceMask);
                }
                vk->vkCmdSetDeviceMask(unboxed_commandBuffer, deviceMask);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetDeviceMask(snapshotTraceBegin, snapshotTraceBytes,
                                                            &m_pool, commandBuffer, deviceMask);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdDispatchBase: {
                android::base::beginTrace("vkCmdDispatchBase decode");
                VkCommandBuffer commandBuffer;
                uint32_t baseGroupX;
                uint32_t baseGroupY;
                uint32_t baseGroupZ;
                uint32_t groupCountX;
                uint32_t groupCountY;
                uint32_t groupCountZ;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&baseGroupX, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&baseGroupY, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&baseGroupZ, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&groupCountX, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&groupCountY, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&groupCountZ, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdDispatchBase 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)baseGroupX, (unsigned long long)baseGroupY,
                            (unsigned long long)baseGroupZ, (unsigned long long)groupCountX,
                            (unsigned long long)groupCountY, (unsigned long long)groupCountZ);
                }
                vk->vkCmdDispatchBase(unboxed_commandBuffer, baseGroupX, baseGroupY, baseGroupZ,
                                      groupCountX, groupCountY, groupCountZ);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdDispatchBase(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, baseGroupX,
                        baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkEnumeratePhysicalDeviceGroups: {
                android::base::beginTrace("vkEnumeratePhysicalDeviceGroups decode");
                VkInstance instance;
                uint32_t* pPhysicalDeviceGroupCount;
                VkPhysicalDeviceGroupProperties* pPhysicalDeviceGroupProperties;
                // Begin non wrapped dispatchable handle unboxing for instance;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkInstance*)&instance = (VkInstance)(VkInstance)((VkInstance)(*&cgen_var_0));
                auto unboxed_instance = unbox_VkInstance(instance);
                auto vk = dispatch_VkInstance(instance);
                // End manual dispatchable handle unboxing for instance;
                // Begin manual dispatchable handle unboxing for pPhysicalDeviceGroupCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pPhysicalDeviceGroupCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPhysicalDeviceGroupCount);
                *readStreamPtrPtr += 8;
                if (pPhysicalDeviceGroupCount) {
                    vkReadStream->alloc((void**)&pPhysicalDeviceGroupCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pPhysicalDeviceGroupCount, *readStreamPtrPtr,
                           sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pPhysicalDeviceGroupProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkPhysicalDeviceGroupProperties**)&pPhysicalDeviceGroupProperties,
                       (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPhysicalDeviceGroupProperties);
                *readStreamPtrPtr += 8;
                if (pPhysicalDeviceGroupProperties) {
                    vkReadStream->alloc(
                        (void**)&pPhysicalDeviceGroupProperties,
                        (*(pPhysicalDeviceGroupCount)) * sizeof(VkPhysicalDeviceGroupProperties));
                    for (uint32_t i = 0; i < (uint32_t)(*(pPhysicalDeviceGroupCount)); ++i) {
                        reservedunmarshal_VkPhysicalDeviceGroupProperties(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkPhysicalDeviceGroupProperties*)(pPhysicalDeviceGroupProperties + i),
                            readStreamPtrPtr);
                    }
                }
                if (pPhysicalDeviceGroupCount) {
                    if (pPhysicalDeviceGroupProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPhysicalDeviceGroupCount)); ++i) {
                            transform_tohost_VkPhysicalDeviceGroupProperties(
                                m_state,
                                (VkPhysicalDeviceGroupProperties*)(pPhysicalDeviceGroupProperties +
                                                                   i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkEnumeratePhysicalDeviceGroups 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)instance,
                        (unsigned long long)pPhysicalDeviceGroupCount,
                        (unsigned long long)pPhysicalDeviceGroupProperties);
                }
                VkResult vkEnumeratePhysicalDeviceGroups_VkResult_return = (VkResult)0;
                vkEnumeratePhysicalDeviceGroups_VkResult_return =
                    vk->vkEnumeratePhysicalDeviceGroups(unboxed_instance, pPhysicalDeviceGroupCount,
                                                        pPhysicalDeviceGroupProperties);
                if ((vkEnumeratePhysicalDeviceGroups_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkEnumeratePhysicalDeviceGroups_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pPhysicalDeviceGroupCount;
                vkStream->putBe64(cgen_var_3);
                if (pPhysicalDeviceGroupCount) {
                    vkStream->write((uint32_t*)pPhysicalDeviceGroupCount, sizeof(uint32_t));
                }
                if (pPhysicalDeviceGroupCount) {
                    if (pPhysicalDeviceGroupProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPhysicalDeviceGroupCount)); ++i) {
                            transform_fromhost_VkPhysicalDeviceGroupProperties(
                                m_state,
                                (VkPhysicalDeviceGroupProperties*)(pPhysicalDeviceGroupProperties +
                                                                   i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pPhysicalDeviceGroupProperties;
                vkStream->putBe64(cgen_var_4);
                if (pPhysicalDeviceGroupProperties) {
                    if (pPhysicalDeviceGroupCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPhysicalDeviceGroupCount)); ++i) {
                            marshal_VkPhysicalDeviceGroupProperties(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkPhysicalDeviceGroupProperties*)(pPhysicalDeviceGroupProperties +
                                                                   i));
                        }
                    }
                }
                vkStream->write(&vkEnumeratePhysicalDeviceGroups_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkEnumeratePhysicalDeviceGroups(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkEnumeratePhysicalDeviceGroups_VkResult_return, instance,
                        pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetImageMemoryRequirements2: {
                android::base::beginTrace("vkGetImageMemoryRequirements2 decode");
                VkDevice device;
                const VkImageMemoryRequirementsInfo2* pInfo;
                VkMemoryRequirements2* pMemoryRequirements;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkImageMemoryRequirementsInfo2));
                reservedunmarshal_VkImageMemoryRequirementsInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImageMemoryRequirementsInfo2*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryRequirements, sizeof(VkMemoryRequirements2));
                reservedunmarshal_VkMemoryRequirements2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkMemoryRequirements2*)(pMemoryRequirements), readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkImageMemoryRequirementsInfo2(
                        m_state, (VkImageMemoryRequirementsInfo2*)(pInfo));
                }
                if (pMemoryRequirements) {
                    transform_tohost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetImageMemoryRequirements2 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                            (unsigned long long)pMemoryRequirements);
                }
                m_state->on_vkGetImageMemoryRequirements2(&m_pool, device, pInfo,
                                                          pMemoryRequirements);
                vkStream->unsetHandleMapping();
                if (pMemoryRequirements) {
                    transform_fromhost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                marshal_VkMemoryRequirements2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                              (VkMemoryRequirements2*)(pMemoryRequirements));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetImageMemoryRequirements2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetBufferMemoryRequirements2: {
                android::base::beginTrace("vkGetBufferMemoryRequirements2 decode");
                VkDevice device;
                const VkBufferMemoryRequirementsInfo2* pInfo;
                VkMemoryRequirements2* pMemoryRequirements;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkBufferMemoryRequirementsInfo2));
                reservedunmarshal_VkBufferMemoryRequirementsInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkBufferMemoryRequirementsInfo2*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryRequirements, sizeof(VkMemoryRequirements2));
                reservedunmarshal_VkMemoryRequirements2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkMemoryRequirements2*)(pMemoryRequirements), readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkBufferMemoryRequirementsInfo2(
                        m_state, (VkBufferMemoryRequirementsInfo2*)(pInfo));
                }
                if (pMemoryRequirements) {
                    transform_tohost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkGetBufferMemoryRequirements2 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                        (unsigned long long)pMemoryRequirements);
                }
                m_state->on_vkGetBufferMemoryRequirements2(&m_pool, device, pInfo,
                                                           pMemoryRequirements);
                vkStream->unsetHandleMapping();
                if (pMemoryRequirements) {
                    transform_fromhost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                marshal_VkMemoryRequirements2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                              (VkMemoryRequirements2*)(pMemoryRequirements));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetBufferMemoryRequirements2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetImageSparseMemoryRequirements2: {
                android::base::beginTrace("vkGetImageSparseMemoryRequirements2 decode");
                VkDevice device;
                const VkImageSparseMemoryRequirementsInfo2* pInfo;
                uint32_t* pSparseMemoryRequirementCount;
                VkSparseImageMemoryRequirements2* pSparseMemoryRequirements;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo,
                                    sizeof(const VkImageSparseMemoryRequirementsInfo2));
                reservedunmarshal_VkImageSparseMemoryRequirementsInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImageSparseMemoryRequirementsInfo2*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pSparseMemoryRequirementCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pSparseMemoryRequirementCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSparseMemoryRequirementCount);
                *readStreamPtrPtr += 8;
                if (pSparseMemoryRequirementCount) {
                    vkReadStream->alloc((void**)&pSparseMemoryRequirementCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pSparseMemoryRequirementCount, *readStreamPtrPtr,
                           sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pSparseMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkSparseImageMemoryRequirements2**)&pSparseMemoryRequirements,
                       (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSparseMemoryRequirements);
                *readStreamPtrPtr += 8;
                if (pSparseMemoryRequirements) {
                    vkReadStream->alloc((void**)&pSparseMemoryRequirements,
                                        (*(pSparseMemoryRequirementCount)) *
                                            sizeof(VkSparseImageMemoryRequirements2));
                    for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount)); ++i) {
                        reservedunmarshal_VkSparseImageMemoryRequirements2(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i),
                            readStreamPtrPtr);
                    }
                }
                if (pInfo) {
                    transform_tohost_VkImageSparseMemoryRequirementsInfo2(
                        m_state, (VkImageSparseMemoryRequirementsInfo2*)(pInfo));
                }
                if (pSparseMemoryRequirementCount) {
                    if (pSparseMemoryRequirements) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            transform_tohost_VkSparseImageMemoryRequirements2(
                                m_state,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetImageSparseMemoryRequirements2 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                            (unsigned long long)pSparseMemoryRequirementCount,
                            (unsigned long long)pSparseMemoryRequirements);
                }
                vk->vkGetImageSparseMemoryRequirements2(unboxed_device, pInfo,
                                                        pSparseMemoryRequirementCount,
                                                        pSparseMemoryRequirements);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pSparseMemoryRequirementCount;
                vkStream->putBe64(cgen_var_3);
                if (pSparseMemoryRequirementCount) {
                    vkStream->write((uint32_t*)pSparseMemoryRequirementCount, sizeof(uint32_t));
                }
                if (pSparseMemoryRequirementCount) {
                    if (pSparseMemoryRequirements) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            transform_fromhost_VkSparseImageMemoryRequirements2(
                                m_state,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pSparseMemoryRequirements;
                vkStream->putBe64(cgen_var_4);
                if (pSparseMemoryRequirements) {
                    if (pSparseMemoryRequirementCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            marshal_VkSparseImageMemoryRequirements2(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetImageSparseMemoryRequirements2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pSparseMemoryRequirementCount, pSparseMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceFeatures2: {
                android::base::beginTrace("vkGetPhysicalDeviceFeatures2 decode");
                VkPhysicalDevice physicalDevice;
                VkPhysicalDeviceFeatures2* pFeatures;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                // Begin manual dispatchable handle unboxing for pFeatures;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pFeatures, sizeof(VkPhysicalDeviceFeatures2));
                reservedunmarshal_VkPhysicalDeviceFeatures2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceFeatures2*)(pFeatures), readStreamPtrPtr);
                if (pFeatures) {
                    transform_tohost_VkPhysicalDeviceFeatures2(
                        m_state, (VkPhysicalDeviceFeatures2*)(pFeatures));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetPhysicalDeviceFeatures2 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pFeatures);
                }
                m_state->on_vkGetPhysicalDeviceFeatures2(&m_pool, physicalDevice, pFeatures);
                vkStream->unsetHandleMapping();
                if (pFeatures) {
                    transform_fromhost_VkPhysicalDeviceFeatures2(
                        m_state, (VkPhysicalDeviceFeatures2*)(pFeatures));
                }
                marshal_VkPhysicalDeviceFeatures2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                  (VkPhysicalDeviceFeatures2*)(pFeatures));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceFeatures2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice, pFeatures);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceProperties2: {
                android::base::beginTrace("vkGetPhysicalDeviceProperties2 decode");
                VkPhysicalDevice physicalDevice;
                VkPhysicalDeviceProperties2* pProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pProperties, sizeof(VkPhysicalDeviceProperties2));
                reservedunmarshal_VkPhysicalDeviceProperties2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceProperties2*)(pProperties), readStreamPtrPtr);
                if (pProperties) {
                    transform_tohost_VkPhysicalDeviceProperties2(
                        m_state, (VkPhysicalDeviceProperties2*)(pProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceProperties2 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pProperties);
                }
                m_state->on_vkGetPhysicalDeviceProperties2(&m_pool, physicalDevice, pProperties);
                vkStream->unsetHandleMapping();
                if (pProperties) {
                    transform_fromhost_VkPhysicalDeviceProperties2(
                        m_state, (VkPhysicalDeviceProperties2*)(pProperties));
                }
                marshal_VkPhysicalDeviceProperties2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkPhysicalDeviceProperties2*)(pProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceProperties2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceFormatProperties2: {
                android::base::beginTrace("vkGetPhysicalDeviceFormatProperties2 decode");
                VkPhysicalDevice physicalDevice;
                VkFormat format;
                VkFormatProperties2* pFormatProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                memcpy((VkFormat*)&format, *readStreamPtrPtr, sizeof(VkFormat));
                *readStreamPtrPtr += sizeof(VkFormat);
                // Begin manual dispatchable handle unboxing for pFormatProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pFormatProperties, sizeof(VkFormatProperties2));
                reservedunmarshal_VkFormatProperties2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkFormatProperties2*)(pFormatProperties),
                                                      readStreamPtrPtr);
                if (pFormatProperties) {
                    transform_tohost_VkFormatProperties2(m_state,
                                                         (VkFormatProperties2*)(pFormatProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceFormatProperties2 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)format, (unsigned long long)pFormatProperties);
                }
                m_state->on_vkGetPhysicalDeviceFormatProperties2(&m_pool, physicalDevice, format,
                                                                 pFormatProperties);
                vkStream->unsetHandleMapping();
                if (pFormatProperties) {
                    transform_fromhost_VkFormatProperties2(
                        m_state, (VkFormatProperties2*)(pFormatProperties));
                }
                marshal_VkFormatProperties2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                            (VkFormatProperties2*)(pFormatProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceFormatProperties2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice, format,
                        pFormatProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceImageFormatProperties2: {
                android::base::beginTrace("vkGetPhysicalDeviceImageFormatProperties2 decode");
                VkPhysicalDevice physicalDevice;
                const VkPhysicalDeviceImageFormatInfo2* pImageFormatInfo;
                VkImageFormatProperties2* pImageFormatProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pImageFormatInfo,
                                    sizeof(const VkPhysicalDeviceImageFormatInfo2));
                reservedunmarshal_VkPhysicalDeviceImageFormatInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceImageFormatInfo2*)(pImageFormatInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pImageFormatProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pImageFormatProperties,
                                    sizeof(VkImageFormatProperties2));
                reservedunmarshal_VkImageFormatProperties2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImageFormatProperties2*)(pImageFormatProperties), readStreamPtrPtr);
                if (pImageFormatInfo) {
                    transform_tohost_VkPhysicalDeviceImageFormatInfo2(
                        m_state, (VkPhysicalDeviceImageFormatInfo2*)(pImageFormatInfo));
                }
                if (pImageFormatProperties) {
                    transform_tohost_VkImageFormatProperties2(
                        m_state, (VkImageFormatProperties2*)(pImageFormatProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceImageFormatProperties2 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pImageFormatInfo,
                            (unsigned long long)pImageFormatProperties);
                }
                VkResult vkGetPhysicalDeviceImageFormatProperties2_VkResult_return = (VkResult)0;
                vkGetPhysicalDeviceImageFormatProperties2_VkResult_return =
                    m_state->on_vkGetPhysicalDeviceImageFormatProperties2(
                        &m_pool, physicalDevice, pImageFormatInfo, pImageFormatProperties);
                if ((vkGetPhysicalDeviceImageFormatProperties2_VkResult_return) ==
                    VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(
                    vkGetPhysicalDeviceImageFormatProperties2_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                if (pImageFormatProperties) {
                    transform_fromhost_VkImageFormatProperties2(
                        m_state, (VkImageFormatProperties2*)(pImageFormatProperties));
                }
                marshal_VkImageFormatProperties2(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImageFormatProperties2*)(pImageFormatProperties));
                vkStream->write(&vkGetPhysicalDeviceImageFormatProperties2_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceImageFormatProperties2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetPhysicalDeviceImageFormatProperties2_VkResult_return, physicalDevice,
                        pImageFormatInfo, pImageFormatProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceQueueFamilyProperties2: {
                android::base::beginTrace("vkGetPhysicalDeviceQueueFamilyProperties2 decode");
                VkPhysicalDevice physicalDevice;
                uint32_t* pQueueFamilyPropertyCount;
                VkQueueFamilyProperties2* pQueueFamilyProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                // Begin manual dispatchable handle unboxing for pQueueFamilyPropertyCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pQueueFamilyPropertyCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pQueueFamilyPropertyCount);
                *readStreamPtrPtr += 8;
                if (pQueueFamilyPropertyCount) {
                    vkReadStream->alloc((void**)&pQueueFamilyPropertyCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pQueueFamilyPropertyCount, *readStreamPtrPtr,
                           sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pQueueFamilyProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkQueueFamilyProperties2**)&pQueueFamilyProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pQueueFamilyProperties);
                *readStreamPtrPtr += 8;
                if (pQueueFamilyProperties) {
                    vkReadStream->alloc(
                        (void**)&pQueueFamilyProperties,
                        (*(pQueueFamilyPropertyCount)) * sizeof(VkQueueFamilyProperties2));
                    for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                        reservedunmarshal_VkQueueFamilyProperties2(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkQueueFamilyProperties2*)(pQueueFamilyProperties + i),
                            readStreamPtrPtr);
                    }
                }
                if (pQueueFamilyPropertyCount) {
                    if (pQueueFamilyProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                            transform_tohost_VkQueueFamilyProperties2(
                                m_state, (VkQueueFamilyProperties2*)(pQueueFamilyProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceQueueFamilyProperties2 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pQueueFamilyPropertyCount,
                            (unsigned long long)pQueueFamilyProperties);
                }
                vk->vkGetPhysicalDeviceQueueFamilyProperties2(
                    unboxed_physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pQueueFamilyPropertyCount;
                vkStream->putBe64(cgen_var_3);
                if (pQueueFamilyPropertyCount) {
                    vkStream->write((uint32_t*)pQueueFamilyPropertyCount, sizeof(uint32_t));
                }
                if (pQueueFamilyPropertyCount) {
                    if (pQueueFamilyProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                            transform_fromhost_VkQueueFamilyProperties2(
                                m_state, (VkQueueFamilyProperties2*)(pQueueFamilyProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pQueueFamilyProperties;
                vkStream->putBe64(cgen_var_4);
                if (pQueueFamilyProperties) {
                    if (pQueueFamilyPropertyCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                            marshal_VkQueueFamilyProperties2(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkQueueFamilyProperties2*)(pQueueFamilyProperties + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceQueueFamilyProperties2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pQueueFamilyPropertyCount, pQueueFamilyProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceMemoryProperties2: {
                android::base::beginTrace("vkGetPhysicalDeviceMemoryProperties2 decode");
                VkPhysicalDevice physicalDevice;
                VkPhysicalDeviceMemoryProperties2* pMemoryProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                // Begin manual dispatchable handle unboxing for pMemoryProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryProperties,
                                    sizeof(VkPhysicalDeviceMemoryProperties2));
                reservedunmarshal_VkPhysicalDeviceMemoryProperties2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceMemoryProperties2*)(pMemoryProperties), readStreamPtrPtr);
                if (pMemoryProperties) {
                    transform_tohost_VkPhysicalDeviceMemoryProperties2(
                        m_state, (VkPhysicalDeviceMemoryProperties2*)(pMemoryProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceMemoryProperties2 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pMemoryProperties);
                }
                m_state->on_vkGetPhysicalDeviceMemoryProperties2(&m_pool, physicalDevice,
                                                                 pMemoryProperties);
                vkStream->unsetHandleMapping();
                if (pMemoryProperties) {
                    transform_fromhost_VkPhysicalDeviceMemoryProperties2(
                        m_state, (VkPhysicalDeviceMemoryProperties2*)(pMemoryProperties));
                }
                marshal_VkPhysicalDeviceMemoryProperties2(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceMemoryProperties2*)(pMemoryProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceMemoryProperties2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pMemoryProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceSparseImageFormatProperties2: {
                android::base::beginTrace("vkGetPhysicalDeviceSparseImageFormatProperties2 decode");
                VkPhysicalDevice physicalDevice;
                const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo;
                uint32_t* pPropertyCount;
                VkSparseImageFormatProperties2* pProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                vkReadStream->alloc((void**)&pFormatInfo,
                                    sizeof(const VkPhysicalDeviceSparseImageFormatInfo2));
                reservedunmarshal_VkPhysicalDeviceSparseImageFormatInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceSparseImageFormatInfo2*)(pFormatInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pPropertyCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pPropertyCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPropertyCount);
                *readStreamPtrPtr += 8;
                if (pPropertyCount) {
                    vkReadStream->alloc((void**)&pPropertyCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pPropertyCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkSparseImageFormatProperties2**)&pProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pProperties);
                *readStreamPtrPtr += 8;
                if (pProperties) {
                    vkReadStream->alloc(
                        (void**)&pProperties,
                        (*(pPropertyCount)) * sizeof(VkSparseImageFormatProperties2));
                    for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                        reservedunmarshal_VkSparseImageFormatProperties2(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkSparseImageFormatProperties2*)(pProperties + i), readStreamPtrPtr);
                    }
                }
                if (pFormatInfo) {
                    transform_tohost_VkPhysicalDeviceSparseImageFormatInfo2(
                        m_state, (VkPhysicalDeviceSparseImageFormatInfo2*)(pFormatInfo));
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_tohost_VkSparseImageFormatProperties2(
                                m_state, (VkSparseImageFormatProperties2*)(pProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceSparseImageFormatProperties2 "
                            "0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pFormatInfo, (unsigned long long)pPropertyCount,
                            (unsigned long long)pProperties);
                }
                vk->vkGetPhysicalDeviceSparseImageFormatProperties2(
                    unboxed_physicalDevice, pFormatInfo, pPropertyCount, pProperties);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pPropertyCount;
                vkStream->putBe64(cgen_var_3);
                if (pPropertyCount) {
                    vkStream->write((uint32_t*)pPropertyCount, sizeof(uint32_t));
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_fromhost_VkSparseImageFormatProperties2(
                                m_state, (VkSparseImageFormatProperties2*)(pProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pProperties;
                vkStream->putBe64(cgen_var_4);
                if (pProperties) {
                    if (pPropertyCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            marshal_VkSparseImageFormatProperties2(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkSparseImageFormatProperties2*)(pProperties + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceSparseImageFormatProperties2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pFormatInfo, pPropertyCount, pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkTrimCommandPool: {
                android::base::beginTrace("vkTrimCommandPool decode");
                VkDevice device;
                VkCommandPool commandPool;
                VkCommandPoolTrimFlags flags;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandPool*)&commandPool =
                    (VkCommandPool)unbox_VkCommandPool((VkCommandPool)(*&cgen_var_1));
                memcpy((VkCommandPoolTrimFlags*)&flags, *readStreamPtrPtr,
                       sizeof(VkCommandPoolTrimFlags));
                *readStreamPtrPtr += sizeof(VkCommandPoolTrimFlags);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkTrimCommandPool 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)commandPool,
                            (unsigned long long)flags);
                }
                vk->vkTrimCommandPool(unboxed_device, commandPool, flags);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkTrimCommandPool(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, device, commandPool, flags);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceQueue2: {
                android::base::beginTrace("vkGetDeviceQueue2 decode");
                VkDevice device;
                const VkDeviceQueueInfo2* pQueueInfo;
                VkQueue* pQueue;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pQueueInfo, sizeof(const VkDeviceQueueInfo2));
                reservedunmarshal_VkDeviceQueueInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkDeviceQueueInfo2*)(pQueueInfo),
                                                     readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pQueue;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pQueue, sizeof(VkQueue));
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkQueue*)pQueue = (VkQueue)(VkQueue)((VkQueue)(*&cgen_var_1));
                if (pQueueInfo) {
                    transform_tohost_VkDeviceQueueInfo2(m_state, (VkDeviceQueueInfo2*)(pQueueInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetDeviceQueue2 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pQueueInfo,
                            (unsigned long long)pQueue);
                }
                m_state->on_vkGetDeviceQueue2(&m_pool, device, pQueueInfo, pQueue);
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_2;
                static_assert(8 == sizeof(VkQueue),
                              "handle map overwrite requires VkQueue to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkQueue((VkQueue*)pQueue, 1);
                vkStream->write((VkQueue*)pQueue, 8 * 1);
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceQueue2(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, device, pQueueInfo, pQueue);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateSamplerYcbcrConversion: {
                android::base::beginTrace("vkCreateSamplerYcbcrConversion decode");
                VkDevice device;
                const VkSamplerYcbcrConversionCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkSamplerYcbcrConversion* pYcbcrConversion;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo,
                                    sizeof(const VkSamplerYcbcrConversionCreateInfo));
                reservedunmarshal_VkSamplerYcbcrConversionCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkSamplerYcbcrConversionCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pYcbcrConversion;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pYcbcrConversion, sizeof(VkSamplerYcbcrConversion));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkSamplerYcbcrConversion*)pYcbcrConversion =
                    (VkSamplerYcbcrConversion)(VkSamplerYcbcrConversion)((
                        VkSamplerYcbcrConversion)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkSamplerYcbcrConversionCreateInfo(
                        m_state, (VkSamplerYcbcrConversionCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateSamplerYcbcrConversion 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pYcbcrConversion);
                }
                VkResult vkCreateSamplerYcbcrConversion_VkResult_return = (VkResult)0;
                vkCreateSamplerYcbcrConversion_VkResult_return =
                    m_state->on_vkCreateSamplerYcbcrConversion(&m_pool, device, pCreateInfo,
                                                               pAllocator, pYcbcrConversion);
                if ((vkCreateSamplerYcbcrConversion_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateSamplerYcbcrConversion_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pYcbcrConversion;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(
                    8 == sizeof(VkSamplerYcbcrConversion),
                    "handle map overwrite requires VkSamplerYcbcrConversion to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkSamplerYcbcrConversion(
                    (VkSamplerYcbcrConversion*)pYcbcrConversion, 1);
                vkStream->write((VkSamplerYcbcrConversion*)pYcbcrConversion, 8 * 1);
                // Begin manual non dispatchable handle create for pYcbcrConversion;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateSamplerYcbcrConversion_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateSamplerYcbcrConversion(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateSamplerYcbcrConversion_VkResult_return, device, pCreateInfo,
                        pAllocator, pYcbcrConversion);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroySamplerYcbcrConversion: {
                android::base::beginTrace("vkDestroySamplerYcbcrConversion decode");
                VkDevice device;
                VkSamplerYcbcrConversion ycbcrConversion;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for ycbcrConversion;
                VkSamplerYcbcrConversion boxed_ycbcrConversion_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkSamplerYcbcrConversion*)&ycbcrConversion =
                    (VkSamplerYcbcrConversion)(VkSamplerYcbcrConversion)((
                        VkSamplerYcbcrConversion)(*&cgen_var_1));
                boxed_ycbcrConversion_preserve = ycbcrConversion;
                ycbcrConversion = unbox_VkSamplerYcbcrConversion(ycbcrConversion);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkDestroySamplerYcbcrConversion 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)ycbcrConversion,
                        (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroySamplerYcbcrConversion(&m_pool, device, ycbcrConversion,
                                                            pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroySamplerYcbcrConversion(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_ycbcrConversion_preserve, pAllocator);
                }
                delete_VkSamplerYcbcrConversion(boxed_ycbcrConversion_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateDescriptorUpdateTemplate: {
                android::base::beginTrace("vkCreateDescriptorUpdateTemplate decode");
                VkDevice device;
                const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkDescriptorUpdateTemplate* pDescriptorUpdateTemplate;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo,
                                    sizeof(const VkDescriptorUpdateTemplateCreateInfo));
                reservedunmarshal_VkDescriptorUpdateTemplateCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDescriptorUpdateTemplateCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pDescriptorUpdateTemplate;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pDescriptorUpdateTemplate,
                                    sizeof(VkDescriptorUpdateTemplate));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkDescriptorUpdateTemplate*)pDescriptorUpdateTemplate =
                    (VkDescriptorUpdateTemplate)(VkDescriptorUpdateTemplate)((
                        VkDescriptorUpdateTemplate)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkDescriptorUpdateTemplateCreateInfo(
                        m_state, (VkDescriptorUpdateTemplateCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateDescriptorUpdateTemplate 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator,
                            (unsigned long long)pDescriptorUpdateTemplate);
                }
                VkResult vkCreateDescriptorUpdateTemplate_VkResult_return = (VkResult)0;
                vkCreateDescriptorUpdateTemplate_VkResult_return =
                    m_state->on_vkCreateDescriptorUpdateTemplate(
                        &m_pool, device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate);
                if ((vkCreateDescriptorUpdateTemplate_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateDescriptorUpdateTemplate_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pDescriptorUpdateTemplate;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(
                    8 == sizeof(VkDescriptorUpdateTemplate),
                    "handle map overwrite requires VkDescriptorUpdateTemplate to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkDescriptorUpdateTemplate(
                    (VkDescriptorUpdateTemplate*)pDescriptorUpdateTemplate, 1);
                vkStream->write((VkDescriptorUpdateTemplate*)pDescriptorUpdateTemplate, 8 * 1);
                // Begin manual non dispatchable handle create for pDescriptorUpdateTemplate;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateDescriptorUpdateTemplate_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateDescriptorUpdateTemplate(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateDescriptorUpdateTemplate_VkResult_return, device, pCreateInfo,
                        pAllocator, pDescriptorUpdateTemplate);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyDescriptorUpdateTemplate: {
                android::base::beginTrace("vkDestroyDescriptorUpdateTemplate decode");
                VkDevice device;
                VkDescriptorUpdateTemplate descriptorUpdateTemplate;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for
                // descriptorUpdateTemplate;
                VkDescriptorUpdateTemplate boxed_descriptorUpdateTemplate_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDescriptorUpdateTemplate*)&descriptorUpdateTemplate =
                    (VkDescriptorUpdateTemplate)(VkDescriptorUpdateTemplate)((
                        VkDescriptorUpdateTemplate)(*&cgen_var_1));
                boxed_descriptorUpdateTemplate_preserve = descriptorUpdateTemplate;
                descriptorUpdateTemplate =
                    unbox_VkDescriptorUpdateTemplate(descriptorUpdateTemplate);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkDestroyDescriptorUpdateTemplate 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device,
                        (unsigned long long)descriptorUpdateTemplate,
                        (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyDescriptorUpdateTemplate(&m_pool, device,
                                                              descriptorUpdateTemplate, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyDescriptorUpdateTemplate(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_descriptorUpdateTemplate_preserve, pAllocator);
                }
                delete_VkDescriptorUpdateTemplate(boxed_descriptorUpdateTemplate_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkUpdateDescriptorSetWithTemplate: {
                android::base::beginTrace("vkUpdateDescriptorSetWithTemplate decode");
                VkDevice device;
                VkDescriptorSet descriptorSet;
                VkDescriptorUpdateTemplate descriptorUpdateTemplate;
                const void* pData;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDescriptorSet*)&descriptorSet =
                    (VkDescriptorSet)unbox_VkDescriptorSet((VkDescriptorSet)(*&cgen_var_1));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDescriptorUpdateTemplate*)&descriptorUpdateTemplate =
                    (VkDescriptorUpdateTemplate)unbox_VkDescriptorUpdateTemplate(
                        (VkDescriptorUpdateTemplate)(*&cgen_var_2));
                // WARNING PTR CHECK
                memcpy((void**)&pData, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pData);
                *readStreamPtrPtr += 8;
                if (pData) {
                    vkReadStream->alloc((void**)&pData, sizeof(const uint8_t));
                    memcpy((void*)pData, *readStreamPtrPtr, sizeof(const uint8_t));
                    *readStreamPtrPtr += sizeof(const uint8_t);
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkUpdateDescriptorSetWithTemplate 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)descriptorSet,
                            (unsigned long long)descriptorUpdateTemplate,
                            (unsigned long long)pData);
                }
                vk->vkUpdateDescriptorSetWithTemplate(unboxed_device, descriptorSet,
                                                      descriptorUpdateTemplate, pData);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkUpdateDescriptorSetWithTemplate(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, descriptorSet,
                        descriptorUpdateTemplate, pData);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceExternalBufferProperties: {
                android::base::beginTrace("vkGetPhysicalDeviceExternalBufferProperties decode");
                VkPhysicalDevice physicalDevice;
                const VkPhysicalDeviceExternalBufferInfo* pExternalBufferInfo;
                VkExternalBufferProperties* pExternalBufferProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                vkReadStream->alloc((void**)&pExternalBufferInfo,
                                    sizeof(const VkPhysicalDeviceExternalBufferInfo));
                reservedunmarshal_VkPhysicalDeviceExternalBufferInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceExternalBufferInfo*)(pExternalBufferInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pExternalBufferProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pExternalBufferProperties,
                                    sizeof(VkExternalBufferProperties));
                reservedunmarshal_VkExternalBufferProperties(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalBufferProperties*)(pExternalBufferProperties), readStreamPtrPtr);
                if (pExternalBufferInfo) {
                    m_state->transformImpl_VkPhysicalDeviceExternalBufferInfo_tohost(
                        pExternalBufferInfo, 1);
                    transform_tohost_VkPhysicalDeviceExternalBufferInfo(
                        m_state, (VkPhysicalDeviceExternalBufferInfo*)(pExternalBufferInfo));
                }
                if (pExternalBufferProperties) {
                    m_state->transformImpl_VkExternalBufferProperties_tohost(
                        pExternalBufferProperties, 1);
                    transform_tohost_VkExternalBufferProperties(
                        m_state, (VkExternalBufferProperties*)(pExternalBufferProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceExternalBufferProperties 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pExternalBufferInfo,
                            (unsigned long long)pExternalBufferProperties);
                }
                vk->vkGetPhysicalDeviceExternalBufferProperties(
                    unboxed_physicalDevice, pExternalBufferInfo, pExternalBufferProperties);
                vkStream->unsetHandleMapping();
                if (pExternalBufferProperties) {
                    m_state->transformImpl_VkExternalBufferProperties_fromhost(
                        pExternalBufferProperties, 1);
                    transform_fromhost_VkExternalBufferProperties(
                        m_state, (VkExternalBufferProperties*)(pExternalBufferProperties));
                }
                marshal_VkExternalBufferProperties(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalBufferProperties*)(pExternalBufferProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceExternalBufferProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pExternalBufferInfo, pExternalBufferProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceExternalFenceProperties: {
                android::base::beginTrace("vkGetPhysicalDeviceExternalFenceProperties decode");
                VkPhysicalDevice physicalDevice;
                const VkPhysicalDeviceExternalFenceInfo* pExternalFenceInfo;
                VkExternalFenceProperties* pExternalFenceProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                vkReadStream->alloc((void**)&pExternalFenceInfo,
                                    sizeof(const VkPhysicalDeviceExternalFenceInfo));
                reservedunmarshal_VkPhysicalDeviceExternalFenceInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceExternalFenceInfo*)(pExternalFenceInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pExternalFenceProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pExternalFenceProperties,
                                    sizeof(VkExternalFenceProperties));
                reservedunmarshal_VkExternalFenceProperties(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalFenceProperties*)(pExternalFenceProperties), readStreamPtrPtr);
                if (pExternalFenceInfo) {
                    transform_tohost_VkPhysicalDeviceExternalFenceInfo(
                        m_state, (VkPhysicalDeviceExternalFenceInfo*)(pExternalFenceInfo));
                }
                if (pExternalFenceProperties) {
                    transform_tohost_VkExternalFenceProperties(
                        m_state, (VkExternalFenceProperties*)(pExternalFenceProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceExternalFenceProperties 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pExternalFenceInfo,
                            (unsigned long long)pExternalFenceProperties);
                }
                vk->vkGetPhysicalDeviceExternalFenceProperties(
                    unboxed_physicalDevice, pExternalFenceInfo, pExternalFenceProperties);
                vkStream->unsetHandleMapping();
                if (pExternalFenceProperties) {
                    transform_fromhost_VkExternalFenceProperties(
                        m_state, (VkExternalFenceProperties*)(pExternalFenceProperties));
                }
                marshal_VkExternalFenceProperties(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalFenceProperties*)(pExternalFenceProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceExternalFenceProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pExternalFenceInfo, pExternalFenceProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceExternalSemaphoreProperties: {
                android::base::beginTrace("vkGetPhysicalDeviceExternalSemaphoreProperties decode");
                VkPhysicalDevice physicalDevice;
                const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo;
                VkExternalSemaphoreProperties* pExternalSemaphoreProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pExternalSemaphoreInfo,
                                    sizeof(const VkPhysicalDeviceExternalSemaphoreInfo));
                reservedunmarshal_VkPhysicalDeviceExternalSemaphoreInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceExternalSemaphoreInfo*)(pExternalSemaphoreInfo),
                    readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pExternalSemaphoreProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pExternalSemaphoreProperties,
                                    sizeof(VkExternalSemaphoreProperties));
                reservedunmarshal_VkExternalSemaphoreProperties(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalSemaphoreProperties*)(pExternalSemaphoreProperties),
                    readStreamPtrPtr);
                if (pExternalSemaphoreInfo) {
                    transform_tohost_VkPhysicalDeviceExternalSemaphoreInfo(
                        m_state, (VkPhysicalDeviceExternalSemaphoreInfo*)(pExternalSemaphoreInfo));
                }
                if (pExternalSemaphoreProperties) {
                    transform_tohost_VkExternalSemaphoreProperties(
                        m_state, (VkExternalSemaphoreProperties*)(pExternalSemaphoreProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceExternalSemaphoreProperties 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pExternalSemaphoreInfo,
                            (unsigned long long)pExternalSemaphoreProperties);
                }
                m_state->on_vkGetPhysicalDeviceExternalSemaphoreProperties(
                    &m_pool, physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);
                vkStream->unsetHandleMapping();
                if (pExternalSemaphoreProperties) {
                    transform_fromhost_VkExternalSemaphoreProperties(
                        m_state, (VkExternalSemaphoreProperties*)(pExternalSemaphoreProperties));
                }
                marshal_VkExternalSemaphoreProperties(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalSemaphoreProperties*)(pExternalSemaphoreProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceExternalSemaphoreProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pExternalSemaphoreInfo, pExternalSemaphoreProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDescriptorSetLayoutSupport: {
                android::base::beginTrace("vkGetDescriptorSetLayoutSupport decode");
                VkDevice device;
                const VkDescriptorSetLayoutCreateInfo* pCreateInfo;
                VkDescriptorSetLayoutSupport* pSupport;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pCreateInfo,
                                    sizeof(const VkDescriptorSetLayoutCreateInfo));
                reservedunmarshal_VkDescriptorSetLayoutCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDescriptorSetLayoutCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pSupport;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pSupport, sizeof(VkDescriptorSetLayoutSupport));
                reservedunmarshal_VkDescriptorSetLayoutSupport(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDescriptorSetLayoutSupport*)(pSupport), readStreamPtrPtr);
                if (pCreateInfo) {
                    transform_tohost_VkDescriptorSetLayoutCreateInfo(
                        m_state, (VkDescriptorSetLayoutCreateInfo*)(pCreateInfo));
                }
                if (pSupport) {
                    transform_tohost_VkDescriptorSetLayoutSupport(
                        m_state, (VkDescriptorSetLayoutSupport*)(pSupport));
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkGetDescriptorSetLayoutSupport 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                        (unsigned long long)pSupport);
                }
                vk->vkGetDescriptorSetLayoutSupport(unboxed_device, pCreateInfo, pSupport);
                vkStream->unsetHandleMapping();
                if (pSupport) {
                    transform_fromhost_VkDescriptorSetLayoutSupport(
                        m_state, (VkDescriptorSetLayoutSupport*)(pSupport));
                }
                marshal_VkDescriptorSetLayoutSupport(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkDescriptorSetLayoutSupport*)(pSupport));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDescriptorSetLayoutSupport(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pCreateInfo,
                        pSupport);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_VERSION_1_2
            case OP_vkCmdDrawIndirectCount: {
                android::base::beginTrace("vkCmdDrawIndirectCount decode");
                VkCommandBuffer commandBuffer;
                VkBuffer buffer;
                VkDeviceSize offset;
                VkBuffer countBuffer;
                VkDeviceSize countBufferOffset;
                uint32_t maxDrawCount;
                uint32_t stride;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&buffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                memcpy((VkDeviceSize*)&offset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&countBuffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_2));
                memcpy((VkDeviceSize*)&countBufferOffset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((uint32_t*)&maxDrawCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&stride, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdDrawIndirectCount 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)buffer,
                            (unsigned long long)offset, (unsigned long long)countBuffer,
                            (unsigned long long)countBufferOffset, (unsigned long long)maxDrawCount,
                            (unsigned long long)stride);
                }
                vk->vkCmdDrawIndirectCount(unboxed_commandBuffer, buffer, offset, countBuffer,
                                           countBufferOffset, maxDrawCount, stride);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdDrawIndirectCount(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, buffer,
                        offset, countBuffer, countBufferOffset, maxDrawCount, stride);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdDrawIndexedIndirectCount: {
                android::base::beginTrace("vkCmdDrawIndexedIndirectCount decode");
                VkCommandBuffer commandBuffer;
                VkBuffer buffer;
                VkDeviceSize offset;
                VkBuffer countBuffer;
                VkDeviceSize countBufferOffset;
                uint32_t maxDrawCount;
                uint32_t stride;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&buffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                memcpy((VkDeviceSize*)&offset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&countBuffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_2));
                memcpy((VkDeviceSize*)&countBufferOffset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((uint32_t*)&maxDrawCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&stride, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdDrawIndexedIndirectCount 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)buffer,
                            (unsigned long long)offset, (unsigned long long)countBuffer,
                            (unsigned long long)countBufferOffset, (unsigned long long)maxDrawCount,
                            (unsigned long long)stride);
                }
                vk->vkCmdDrawIndexedIndirectCount(unboxed_commandBuffer, buffer, offset,
                                                  countBuffer, countBufferOffset, maxDrawCount,
                                                  stride);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdDrawIndexedIndirectCount(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, buffer,
                        offset, countBuffer, countBufferOffset, maxDrawCount, stride);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreateRenderPass2: {
                android::base::beginTrace("vkCreateRenderPass2 decode");
                VkDevice device;
                const VkRenderPassCreateInfo2* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkRenderPass* pRenderPass;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkRenderPassCreateInfo2));
                reservedunmarshal_VkRenderPassCreateInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                          (VkRenderPassCreateInfo2*)(pCreateInfo),
                                                          readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pRenderPass;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pRenderPass, sizeof(VkRenderPass));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkRenderPass*)pRenderPass =
                    (VkRenderPass)(VkRenderPass)((VkRenderPass)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkRenderPassCreateInfo2(
                        m_state, (VkRenderPassCreateInfo2*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateRenderPass2 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pRenderPass);
                }
                VkResult vkCreateRenderPass2_VkResult_return = (VkResult)0;
                vkCreateRenderPass2_VkResult_return = m_state->on_vkCreateRenderPass2(
                    &m_pool, device, pCreateInfo, pAllocator, pRenderPass);
                if ((vkCreateRenderPass2_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateRenderPass2_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pRenderPass;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkRenderPass),
                              "handle map overwrite requires VkRenderPass to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkRenderPass((VkRenderPass*)pRenderPass, 1);
                vkStream->write((VkRenderPass*)pRenderPass, 8 * 1);
                // Begin manual non dispatchable handle create for pRenderPass;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateRenderPass2_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateRenderPass2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateRenderPass2_VkResult_return, device, pCreateInfo, pAllocator,
                        pRenderPass);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBeginRenderPass2: {
                android::base::beginTrace("vkCmdBeginRenderPass2 decode");
                VkCommandBuffer commandBuffer;
                const VkRenderPassBeginInfo* pRenderPassBegin;
                const VkSubpassBeginInfo* pSubpassBeginInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pRenderPassBegin, sizeof(const VkRenderPassBeginInfo));
                reservedunmarshal_VkRenderPassBeginInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                        (VkRenderPassBeginInfo*)(pRenderPassBegin),
                                                        readStreamPtrPtr);
                vkReadStream->alloc((void**)&pSubpassBeginInfo, sizeof(const VkSubpassBeginInfo));
                reservedunmarshal_VkSubpassBeginInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkSubpassBeginInfo*)(pSubpassBeginInfo),
                                                     readStreamPtrPtr);
                if (pRenderPassBegin) {
                    transform_tohost_VkRenderPassBeginInfo(
                        m_state, (VkRenderPassBeginInfo*)(pRenderPassBegin));
                }
                if (pSubpassBeginInfo) {
                    transform_tohost_VkSubpassBeginInfo(m_state,
                                                        (VkSubpassBeginInfo*)(pSubpassBeginInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdBeginRenderPass2 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pRenderPassBegin,
                            (unsigned long long)pSubpassBeginInfo);
                }
                vk->vkCmdBeginRenderPass2(unboxed_commandBuffer, pRenderPassBegin,
                                          pSubpassBeginInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBeginRenderPass2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        pRenderPassBegin, pSubpassBeginInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdNextSubpass2: {
                android::base::beginTrace("vkCmdNextSubpass2 decode");
                VkCommandBuffer commandBuffer;
                const VkSubpassBeginInfo* pSubpassBeginInfo;
                const VkSubpassEndInfo* pSubpassEndInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pSubpassBeginInfo, sizeof(const VkSubpassBeginInfo));
                reservedunmarshal_VkSubpassBeginInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkSubpassBeginInfo*)(pSubpassBeginInfo),
                                                     readStreamPtrPtr);
                vkReadStream->alloc((void**)&pSubpassEndInfo, sizeof(const VkSubpassEndInfo));
                reservedunmarshal_VkSubpassEndInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkSubpassEndInfo*)(pSubpassEndInfo),
                                                   readStreamPtrPtr);
                if (pSubpassBeginInfo) {
                    transform_tohost_VkSubpassBeginInfo(m_state,
                                                        (VkSubpassBeginInfo*)(pSubpassBeginInfo));
                }
                if (pSubpassEndInfo) {
                    transform_tohost_VkSubpassEndInfo(m_state,
                                                      (VkSubpassEndInfo*)(pSubpassEndInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdNextSubpass2 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pSubpassBeginInfo,
                            (unsigned long long)pSubpassEndInfo);
                }
                vk->vkCmdNextSubpass2(unboxed_commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdNextSubpass2(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, commandBuffer,
                                                           pSubpassBeginInfo, pSubpassEndInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdEndRenderPass2: {
                android::base::beginTrace("vkCmdEndRenderPass2 decode");
                VkCommandBuffer commandBuffer;
                const VkSubpassEndInfo* pSubpassEndInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pSubpassEndInfo, sizeof(const VkSubpassEndInfo));
                reservedunmarshal_VkSubpassEndInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkSubpassEndInfo*)(pSubpassEndInfo),
                                                   readStreamPtrPtr);
                if (pSubpassEndInfo) {
                    transform_tohost_VkSubpassEndInfo(m_state,
                                                      (VkSubpassEndInfo*)(pSubpassEndInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdEndRenderPass2 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pSubpassEndInfo);
                }
                vk->vkCmdEndRenderPass2(unboxed_commandBuffer, pSubpassEndInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdEndRenderPass2(snapshotTraceBegin, snapshotTraceBytes,
                                                             &m_pool, commandBuffer,
                                                             pSubpassEndInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkResetQueryPool: {
                android::base::beginTrace("vkResetQueryPool decode");
                VkDevice device;
                VkQueryPool queryPool;
                uint32_t firstQuery;
                uint32_t queryCount;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueryPool*)&queryPool =
                    (VkQueryPool)unbox_VkQueryPool((VkQueryPool)(*&cgen_var_1));
                memcpy((uint32_t*)&firstQuery, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&queryCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkResetQueryPool 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)queryPool,
                            (unsigned long long)firstQuery, (unsigned long long)queryCount);
                }
                vk->vkResetQueryPool(unboxed_device, queryPool, firstQuery, queryCount);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkResetQueryPool(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, device, queryPool, firstQuery,
                                                          queryCount);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetSemaphoreCounterValue: {
                android::base::beginTrace("vkGetSemaphoreCounterValue decode");
                VkDevice device;
                VkSemaphore semaphore;
                uint64_t* pValue;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkSemaphore*)&semaphore =
                    (VkSemaphore)unbox_VkSemaphore((VkSemaphore)(*&cgen_var_1));
                // Begin manual dispatchable handle unboxing for pValue;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pValue, sizeof(uint64_t));
                memcpy((uint64_t*)pValue, *readStreamPtrPtr, sizeof(uint64_t));
                *readStreamPtrPtr += sizeof(uint64_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetSemaphoreCounterValue 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)semaphore,
                            (unsigned long long)pValue);
                }
                VkResult vkGetSemaphoreCounterValue_VkResult_return = (VkResult)0;
                vkGetSemaphoreCounterValue_VkResult_return =
                    vk->vkGetSemaphoreCounterValue(unboxed_device, semaphore, pValue);
                if ((vkGetSemaphoreCounterValue_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkGetSemaphoreCounterValue_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                vkStream->write((uint64_t*)pValue, sizeof(uint64_t));
                vkStream->write(&vkGetSemaphoreCounterValue_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetSemaphoreCounterValue(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetSemaphoreCounterValue_VkResult_return, device, semaphore, pValue);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkWaitSemaphores: {
                android::base::beginTrace("vkWaitSemaphores decode");
                VkDevice device;
                const VkSemaphoreWaitInfo* pWaitInfo;
                uint64_t timeout;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pWaitInfo, sizeof(const VkSemaphoreWaitInfo));
                reservedunmarshal_VkSemaphoreWaitInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkSemaphoreWaitInfo*)(pWaitInfo),
                                                      readStreamPtrPtr);
                memcpy((uint64_t*)&timeout, *readStreamPtrPtr, sizeof(uint64_t));
                *readStreamPtrPtr += sizeof(uint64_t);
                if (pWaitInfo) {
                    transform_tohost_VkSemaphoreWaitInfo(m_state,
                                                         (VkSemaphoreWaitInfo*)(pWaitInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkWaitSemaphores 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pWaitInfo,
                            (unsigned long long)timeout);
                }
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                VkResult vkWaitSemaphores_VkResult_return = (VkResult)0;
                vkWaitSemaphores_VkResult_return =
                    vk->vkWaitSemaphores(unboxed_device, pWaitInfo, timeout);
                if ((vkWaitSemaphores_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkWaitSemaphores_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkWaitSemaphores_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkWaitSemaphores(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, vkWaitSemaphores_VkResult_return,
                                                          device, pWaitInfo, timeout);
                }
                vkReadStream->clearPool();
                android::base::endTrace();
                break;
            }
            case OP_vkSignalSemaphore: {
                android::base::beginTrace("vkSignalSemaphore decode");
                VkDevice device;
                const VkSemaphoreSignalInfo* pSignalInfo;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pSignalInfo, sizeof(const VkSemaphoreSignalInfo));
                reservedunmarshal_VkSemaphoreSignalInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                        (VkSemaphoreSignalInfo*)(pSignalInfo),
                                                        readStreamPtrPtr);
                if (pSignalInfo) {
                    transform_tohost_VkSemaphoreSignalInfo(m_state,
                                                           (VkSemaphoreSignalInfo*)(pSignalInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkSignalSemaphore 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)device, (unsigned long long)pSignalInfo);
                }
                VkResult vkSignalSemaphore_VkResult_return = (VkResult)0;
                vkSignalSemaphore_VkResult_return =
                    vk->vkSignalSemaphore(unboxed_device, pSignalInfo);
                if ((vkSignalSemaphore_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkSignalSemaphore_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkSignalSemaphore_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkSignalSemaphore(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkSignalSemaphore_VkResult_return, device, pSignalInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetBufferDeviceAddress: {
                android::base::beginTrace("vkGetBufferDeviceAddress decode");
                VkDevice device;
                const VkBufferDeviceAddressInfo* pInfo;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkBufferDeviceAddressInfo));
                reservedunmarshal_VkBufferDeviceAddressInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM, (VkBufferDeviceAddressInfo*)(pInfo),
                    readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkBufferDeviceAddressInfo(m_state,
                                                               (VkBufferDeviceAddressInfo*)(pInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetBufferDeviceAddress 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo);
                }
                VkDeviceAddress vkGetBufferDeviceAddress_VkDeviceAddress_return =
                    (VkDeviceAddress)0;
                vkGetBufferDeviceAddress_VkDeviceAddress_return =
                    vk->vkGetBufferDeviceAddress(unboxed_device, pInfo);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkGetBufferDeviceAddress_VkDeviceAddress_return,
                                sizeof(VkDeviceAddress));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetBufferDeviceAddress(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetBufferDeviceAddress_VkDeviceAddress_return, device, pInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetBufferOpaqueCaptureAddress: {
                android::base::beginTrace("vkGetBufferOpaqueCaptureAddress decode");
                VkDevice device;
                const VkBufferDeviceAddressInfo* pInfo;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkBufferDeviceAddressInfo));
                reservedunmarshal_VkBufferDeviceAddressInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM, (VkBufferDeviceAddressInfo*)(pInfo),
                    readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkBufferDeviceAddressInfo(m_state,
                                                               (VkBufferDeviceAddressInfo*)(pInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetBufferOpaqueCaptureAddress 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo);
                }
                uint64_t vkGetBufferOpaqueCaptureAddress_uint64_t_return = (uint64_t)0;
                vkGetBufferOpaqueCaptureAddress_uint64_t_return =
                    vk->vkGetBufferOpaqueCaptureAddress(unboxed_device, pInfo);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkGetBufferOpaqueCaptureAddress_uint64_t_return, sizeof(uint64_t));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetBufferOpaqueCaptureAddress(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetBufferOpaqueCaptureAddress_uint64_t_return, device, pInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceMemoryOpaqueCaptureAddress: {
                android::base::beginTrace("vkGetDeviceMemoryOpaqueCaptureAddress decode");
                VkDevice device;
                const VkDeviceMemoryOpaqueCaptureAddressInfo* pInfo;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo,
                                    sizeof(const VkDeviceMemoryOpaqueCaptureAddressInfo));
                reservedunmarshal_VkDeviceMemoryOpaqueCaptureAddressInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDeviceMemoryOpaqueCaptureAddressInfo*)(pInfo), readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkDeviceMemoryOpaqueCaptureAddressInfo(
                        m_state, (VkDeviceMemoryOpaqueCaptureAddressInfo*)(pInfo));
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkGetDeviceMemoryOpaqueCaptureAddress 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)pInfo);
                }
                uint64_t vkGetDeviceMemoryOpaqueCaptureAddress_uint64_t_return = (uint64_t)0;
                vkGetDeviceMemoryOpaqueCaptureAddress_uint64_t_return =
                    vk->vkGetDeviceMemoryOpaqueCaptureAddress(unboxed_device, pInfo);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkGetDeviceMemoryOpaqueCaptureAddress_uint64_t_return,
                                sizeof(uint64_t));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceMemoryOpaqueCaptureAddress(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetDeviceMemoryOpaqueCaptureAddress_uint64_t_return, device, pInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_VERSION_1_3
            case OP_vkGetPhysicalDeviceToolProperties: {
                android::base::beginTrace("vkGetPhysicalDeviceToolProperties decode");
                VkPhysicalDevice physicalDevice;
                uint32_t* pToolCount;
                VkPhysicalDeviceToolProperties* pToolProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                // Begin manual dispatchable handle unboxing for pToolCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pToolCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pToolCount);
                *readStreamPtrPtr += 8;
                if (pToolCount) {
                    vkReadStream->alloc((void**)&pToolCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pToolCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pToolProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkPhysicalDeviceToolProperties**)&pToolProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pToolProperties);
                *readStreamPtrPtr += 8;
                if (pToolProperties) {
                    vkReadStream->alloc((void**)&pToolProperties,
                                        (*(pToolCount)) * sizeof(VkPhysicalDeviceToolProperties));
                    for (uint32_t i = 0; i < (uint32_t)(*(pToolCount)); ++i) {
                        reservedunmarshal_VkPhysicalDeviceToolProperties(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkPhysicalDeviceToolProperties*)(pToolProperties + i),
                            readStreamPtrPtr);
                    }
                }
                if (pToolCount) {
                    if (pToolProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pToolCount)); ++i) {
                            transform_tohost_VkPhysicalDeviceToolProperties(
                                m_state, (VkPhysicalDeviceToolProperties*)(pToolProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkGetPhysicalDeviceToolProperties 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)physicalDevice,
                        (unsigned long long)pToolCount, (unsigned long long)pToolProperties);
                }
                VkResult vkGetPhysicalDeviceToolProperties_VkResult_return = (VkResult)0;
                vkGetPhysicalDeviceToolProperties_VkResult_return =
                    vk->vkGetPhysicalDeviceToolProperties(unboxed_physicalDevice, pToolCount,
                                                          pToolProperties);
                if ((vkGetPhysicalDeviceToolProperties_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkGetPhysicalDeviceToolProperties_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pToolCount;
                vkStream->putBe64(cgen_var_3);
                if (pToolCount) {
                    vkStream->write((uint32_t*)pToolCount, sizeof(uint32_t));
                }
                if (pToolCount) {
                    if (pToolProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pToolCount)); ++i) {
                            transform_fromhost_VkPhysicalDeviceToolProperties(
                                m_state, (VkPhysicalDeviceToolProperties*)(pToolProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pToolProperties;
                vkStream->putBe64(cgen_var_4);
                if (pToolProperties) {
                    if (pToolCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pToolCount)); ++i) {
                            marshal_VkPhysicalDeviceToolProperties(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkPhysicalDeviceToolProperties*)(pToolProperties + i));
                        }
                    }
                }
                vkStream->write(&vkGetPhysicalDeviceToolProperties_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceToolProperties(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetPhysicalDeviceToolProperties_VkResult_return, physicalDevice,
                        pToolCount, pToolProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCreatePrivateDataSlot: {
                android::base::beginTrace("vkCreatePrivateDataSlot decode");
                VkDevice device;
                const VkPrivateDataSlotCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkPrivateDataSlot* pPrivateDataSlot;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pCreateInfo,
                                    sizeof(const VkPrivateDataSlotCreateInfo));
                reservedunmarshal_VkPrivateDataSlotCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPrivateDataSlotCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pPrivateDataSlot;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pPrivateDataSlot, sizeof(VkPrivateDataSlot));
                memcpy((VkPrivateDataSlot*)&(*pPrivateDataSlot), (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&(*pPrivateDataSlot));
                *readStreamPtrPtr += 8;
                if (pCreateInfo) {
                    transform_tohost_VkPrivateDataSlotCreateInfo(
                        m_state, (VkPrivateDataSlotCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkCreatePrivateDataSlot 0x%llx 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                        (unsigned long long)pAllocator, (unsigned long long)pPrivateDataSlot);
                }
                VkResult vkCreatePrivateDataSlot_VkResult_return = (VkResult)0;
                vkCreatePrivateDataSlot_VkResult_return = vk->vkCreatePrivateDataSlot(
                    unboxed_device, pCreateInfo, pAllocator, pPrivateDataSlot);
                if ((vkCreatePrivateDataSlot_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreatePrivateDataSlot_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3 = (uint64_t)(*pPrivateDataSlot);
                vkStream->putBe64(cgen_var_3);
                vkStream->write(&vkCreatePrivateDataSlot_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreatePrivateDataSlot(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreatePrivateDataSlot_VkResult_return, device, pCreateInfo, pAllocator,
                        pPrivateDataSlot);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyPrivateDataSlot: {
                android::base::beginTrace("vkDestroyPrivateDataSlot decode");
                VkDevice device;
                VkPrivateDataSlot privateDataSlot;
                const VkAllocationCallbacks* pAllocator;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                memcpy((VkPrivateDataSlot*)&privateDataSlot, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&privateDataSlot);
                *readStreamPtrPtr += 8;
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkDestroyPrivateDataSlot 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)privateDataSlot, (unsigned long long)pAllocator);
                }
                vk->vkDestroyPrivateDataSlot(unboxed_device, privateDataSlot, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyPrivateDataSlot(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, privateDataSlot,
                        pAllocator);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkSetPrivateData: {
                android::base::beginTrace("vkSetPrivateData decode");
                VkDevice device;
                VkObjectType objectType;
                uint64_t objectHandle;
                VkPrivateDataSlot privateDataSlot;
                uint64_t data;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                memcpy((VkObjectType*)&objectType, *readStreamPtrPtr, sizeof(VkObjectType));
                *readStreamPtrPtr += sizeof(VkObjectType);
                memcpy((uint64_t*)&objectHandle, *readStreamPtrPtr, sizeof(uint64_t));
                *readStreamPtrPtr += sizeof(uint64_t);
                memcpy((VkPrivateDataSlot*)&privateDataSlot, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&privateDataSlot);
                *readStreamPtrPtr += 8;
                memcpy((uint64_t*)&data, *readStreamPtrPtr, sizeof(uint64_t));
                *readStreamPtrPtr += sizeof(uint64_t);
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkSetPrivateData 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)objectType,
                        (unsigned long long)objectHandle, (unsigned long long)privateDataSlot,
                        (unsigned long long)data);
                }
                VkResult vkSetPrivateData_VkResult_return = (VkResult)0;
                vkSetPrivateData_VkResult_return = vk->vkSetPrivateData(
                    unboxed_device, objectType, objectHandle, privateDataSlot, data);
                if ((vkSetPrivateData_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkSetPrivateData_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkSetPrivateData_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkSetPrivateData(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, vkSetPrivateData_VkResult_return,
                                                          device, objectType, objectHandle,
                                                          privateDataSlot, data);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPrivateData: {
                android::base::beginTrace("vkGetPrivateData decode");
                VkDevice device;
                VkObjectType objectType;
                uint64_t objectHandle;
                VkPrivateDataSlot privateDataSlot;
                uint64_t* pData;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                memcpy((VkObjectType*)&objectType, *readStreamPtrPtr, sizeof(VkObjectType));
                *readStreamPtrPtr += sizeof(VkObjectType);
                memcpy((uint64_t*)&objectHandle, *readStreamPtrPtr, sizeof(uint64_t));
                *readStreamPtrPtr += sizeof(uint64_t);
                memcpy((VkPrivateDataSlot*)&privateDataSlot, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&privateDataSlot);
                *readStreamPtrPtr += 8;
                // Begin manual dispatchable handle unboxing for pData;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pData, sizeof(uint64_t));
                memcpy((uint64_t*)pData, *readStreamPtrPtr, sizeof(uint64_t));
                *readStreamPtrPtr += sizeof(uint64_t);
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkGetPrivateData 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)objectType,
                        (unsigned long long)objectHandle, (unsigned long long)privateDataSlot,
                        (unsigned long long)pData);
                }
                vk->vkGetPrivateData(unboxed_device, objectType, objectHandle, privateDataSlot,
                                     pData);
                vkStream->unsetHandleMapping();
                vkStream->write((uint64_t*)pData, sizeof(uint64_t));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPrivateData(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, device, objectType, objectHandle,
                                                          privateDataSlot, pData);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetEvent2: {
                android::base::beginTrace("vkCmdSetEvent2 decode");
                VkCommandBuffer commandBuffer;
                VkEvent event;
                const VkDependencyInfo* pDependencyInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkEvent*)&event = (VkEvent)unbox_VkEvent((VkEvent)(*&cgen_var_1));
                vkReadStream->alloc((void**)&pDependencyInfo, sizeof(const VkDependencyInfo));
                reservedunmarshal_VkDependencyInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkDependencyInfo*)(pDependencyInfo),
                                                   readStreamPtrPtr);
                if (pDependencyInfo) {
                    transform_tohost_VkDependencyInfo(m_state,
                                                      (VkDependencyInfo*)(pDependencyInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetEvent2 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)event,
                            (unsigned long long)pDependencyInfo);
                }
                vk->vkCmdSetEvent2(unboxed_commandBuffer, event, pDependencyInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetEvent2(snapshotTraceBegin, snapshotTraceBytes,
                                                        &m_pool, commandBuffer, event,
                                                        pDependencyInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdResetEvent2: {
                android::base::beginTrace("vkCmdResetEvent2 decode");
                VkCommandBuffer commandBuffer;
                VkEvent event;
                VkPipelineStageFlags2 stageMask;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkEvent*)&event = (VkEvent)unbox_VkEvent((VkEvent)(*&cgen_var_1));
                memcpy((VkPipelineStageFlags2*)&stageMask, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags2));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags2);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdResetEvent2 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)event,
                            (unsigned long long)stageMask);
                }
                vk->vkCmdResetEvent2(unboxed_commandBuffer, event, stageMask);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdResetEvent2(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, commandBuffer, event, stageMask);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdWaitEvents2: {
                android::base::beginTrace("vkCmdWaitEvents2 decode");
                VkCommandBuffer commandBuffer;
                uint32_t eventCount;
                const VkEvent* pEvents;
                const VkDependencyInfo* pDependencyInfos;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&eventCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pEvents, ((eventCount)) * sizeof(const VkEvent));
                if (((eventCount))) {
                    uint8_t* cgen_var_1_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((eventCount));
                    for (uint32_t k = 0; k < ((eventCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_1_ptr + k * 8, sizeof(uint64_t));
                        *(((VkEvent*)pEvents) + k) = (VkEvent)unbox_VkEvent((VkEvent)tmpval);
                    }
                }
                vkReadStream->alloc((void**)&pDependencyInfos,
                                    ((eventCount)) * sizeof(const VkDependencyInfo));
                for (uint32_t i = 0; i < (uint32_t)((eventCount)); ++i) {
                    reservedunmarshal_VkDependencyInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                       (VkDependencyInfo*)(pDependencyInfos + i),
                                                       readStreamPtrPtr);
                }
                if (pDependencyInfos) {
                    for (uint32_t i = 0; i < (uint32_t)((eventCount)); ++i) {
                        transform_tohost_VkDependencyInfo(
                            m_state, (VkDependencyInfo*)(pDependencyInfos + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(
                        stderr, "stream %p: call vkCmdWaitEvents2 0x%llx 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)commandBuffer, (unsigned long long)eventCount,
                        (unsigned long long)pEvents, (unsigned long long)pDependencyInfos);
                }
                vk->vkCmdWaitEvents2(unboxed_commandBuffer, eventCount, pEvents, pDependencyInfos);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdWaitEvents2(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, commandBuffer, eventCount,
                                                          pEvents, pDependencyInfos);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdPipelineBarrier2: {
                android::base::beginTrace("vkCmdPipelineBarrier2 decode");
                VkCommandBuffer commandBuffer;
                const VkDependencyInfo* pDependencyInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pDependencyInfo, sizeof(const VkDependencyInfo));
                reservedunmarshal_VkDependencyInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkDependencyInfo*)(pDependencyInfo),
                                                   readStreamPtrPtr);
                if (pDependencyInfo) {
                    transform_tohost_VkDependencyInfo(m_state,
                                                      (VkDependencyInfo*)(pDependencyInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdPipelineBarrier2 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pDependencyInfo);
                }
                vk->vkCmdPipelineBarrier2(unboxed_commandBuffer, pDependencyInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdPipelineBarrier2(snapshotTraceBegin,
                                                               snapshotTraceBytes, &m_pool,
                                                               commandBuffer, pDependencyInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdWriteTimestamp2: {
                android::base::beginTrace("vkCmdWriteTimestamp2 decode");
                VkCommandBuffer commandBuffer;
                VkPipelineStageFlags2 stage;
                VkQueryPool queryPool;
                uint32_t query;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkPipelineStageFlags2*)&stage, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags2));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags2);
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueryPool*)&queryPool =
                    (VkQueryPool)unbox_VkQueryPool((VkQueryPool)(*&cgen_var_1));
                memcpy((uint32_t*)&query, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdWriteTimestamp2 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)stage,
                            (unsigned long long)queryPool, (unsigned long long)query);
                }
                vk->vkCmdWriteTimestamp2(unboxed_commandBuffer, stage, queryPool, query);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdWriteTimestamp2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, stage,
                        queryPool, query);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkQueueSubmit2: {
                android::base::beginTrace("vkQueueSubmit2 decode");
                VkQueue queue;
                uint32_t submitCount;
                const VkSubmitInfo2* pSubmits;
                VkFence fence;
                // Begin global wrapped dispatchable handle unboxing for queue;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueue*)&queue = (VkQueue)(VkQueue)((VkQueue)(*&cgen_var_0));
                memcpy((uint32_t*)&submitCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pSubmits,
                                    ((submitCount)) * sizeof(const VkSubmitInfo2));
                for (uint32_t i = 0; i < (uint32_t)((submitCount)); ++i) {
                    reservedunmarshal_VkSubmitInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkSubmitInfo2*)(pSubmits + i),
                                                    readStreamPtrPtr);
                }
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkFence*)&fence = (VkFence)unbox_VkFence((VkFence)(*&cgen_var_1));
                if (pSubmits) {
                    for (uint32_t i = 0; i < (uint32_t)((submitCount)); ++i) {
                        transform_tohost_VkSubmitInfo2(m_state, (VkSubmitInfo2*)(pSubmits + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkQueueSubmit2 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)queue, (unsigned long long)submitCount,
                            (unsigned long long)pSubmits, (unsigned long long)fence);
                }
                VkResult vkQueueSubmit2_VkResult_return = (VkResult)0;
                vkQueueSubmit2_VkResult_return =
                    m_state->on_vkQueueSubmit2(&m_pool, queue, submitCount, pSubmits, fence);
                if ((vkQueueSubmit2_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkQueueSubmit2_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkQueueSubmit2_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkQueueSubmit2(snapshotTraceBegin, snapshotTraceBytes,
                                                        &m_pool, vkQueueSubmit2_VkResult_return,
                                                        queue, submitCount, pSubmits, fence);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyBuffer2: {
                android::base::beginTrace("vkCmdCopyBuffer2 decode");
                VkCommandBuffer commandBuffer;
                const VkCopyBufferInfo2* pCopyBufferInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pCopyBufferInfo, sizeof(const VkCopyBufferInfo2));
                reservedunmarshal_VkCopyBufferInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkCopyBufferInfo2*)(pCopyBufferInfo),
                                                    readStreamPtrPtr);
                if (pCopyBufferInfo) {
                    transform_tohost_VkCopyBufferInfo2(m_state,
                                                       (VkCopyBufferInfo2*)(pCopyBufferInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdCopyBuffer2 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer, (unsigned long long)pCopyBufferInfo);
                }
                vk->vkCmdCopyBuffer2(unboxed_commandBuffer, pCopyBufferInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyBuffer2(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, commandBuffer, pCopyBufferInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyImage2: {
                android::base::beginTrace("vkCmdCopyImage2 decode");
                VkCommandBuffer commandBuffer;
                const VkCopyImageInfo2* pCopyImageInfo;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCopyImageInfo, sizeof(const VkCopyImageInfo2));
                reservedunmarshal_VkCopyImageInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkCopyImageInfo2*)(pCopyImageInfo),
                                                   readStreamPtrPtr);
                if (pCopyImageInfo) {
                    transform_tohost_VkCopyImageInfo2(m_state, (VkCopyImageInfo2*)(pCopyImageInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdCopyImage2 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer, (unsigned long long)pCopyImageInfo);
                }
                m_state->on_vkCmdCopyImage2(&m_pool, commandBuffer, pCopyImageInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyImage2(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, commandBuffer, pCopyImageInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyBufferToImage2: {
                android::base::beginTrace("vkCmdCopyBufferToImage2 decode");
                VkCommandBuffer commandBuffer;
                const VkCopyBufferToImageInfo2* pCopyBufferToImageInfo;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCopyBufferToImageInfo,
                                    sizeof(const VkCopyBufferToImageInfo2));
                reservedunmarshal_VkCopyBufferToImageInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkCopyBufferToImageInfo2*)(pCopyBufferToImageInfo), readStreamPtrPtr);
                if (pCopyBufferToImageInfo) {
                    transform_tohost_VkCopyBufferToImageInfo2(
                        m_state, (VkCopyBufferToImageInfo2*)(pCopyBufferToImageInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdCopyBufferToImage2 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pCopyBufferToImageInfo);
                }
                m_state->on_vkCmdCopyBufferToImage2(&m_pool, commandBuffer, pCopyBufferToImageInfo,
                                                    context);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyBufferToImage2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        pCopyBufferToImageInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyImageToBuffer2: {
                android::base::beginTrace("vkCmdCopyImageToBuffer2 decode");
                VkCommandBuffer commandBuffer;
                const VkCopyImageToBufferInfo2* pCopyImageToBufferInfo;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCopyImageToBufferInfo,
                                    sizeof(const VkCopyImageToBufferInfo2));
                reservedunmarshal_VkCopyImageToBufferInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkCopyImageToBufferInfo2*)(pCopyImageToBufferInfo), readStreamPtrPtr);
                if (pCopyImageToBufferInfo) {
                    transform_tohost_VkCopyImageToBufferInfo2(
                        m_state, (VkCopyImageToBufferInfo2*)(pCopyImageToBufferInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdCopyImageToBuffer2 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pCopyImageToBufferInfo);
                }
                m_state->on_vkCmdCopyImageToBuffer2(&m_pool, commandBuffer, pCopyImageToBufferInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyImageToBuffer2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        pCopyImageToBufferInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBlitImage2: {
                android::base::beginTrace("vkCmdBlitImage2 decode");
                VkCommandBuffer commandBuffer;
                const VkBlitImageInfo2* pBlitImageInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pBlitImageInfo, sizeof(const VkBlitImageInfo2));
                reservedunmarshal_VkBlitImageInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkBlitImageInfo2*)(pBlitImageInfo),
                                                   readStreamPtrPtr);
                if (pBlitImageInfo) {
                    transform_tohost_VkBlitImageInfo2(m_state, (VkBlitImageInfo2*)(pBlitImageInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdBlitImage2 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer, (unsigned long long)pBlitImageInfo);
                }
                vk->vkCmdBlitImage2(unboxed_commandBuffer, pBlitImageInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBlitImage2(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, commandBuffer, pBlitImageInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdResolveImage2: {
                android::base::beginTrace("vkCmdResolveImage2 decode");
                VkCommandBuffer commandBuffer;
                const VkResolveImageInfo2* pResolveImageInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pResolveImageInfo, sizeof(const VkResolveImageInfo2));
                reservedunmarshal_VkResolveImageInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkResolveImageInfo2*)(pResolveImageInfo),
                                                      readStreamPtrPtr);
                if (pResolveImageInfo) {
                    transform_tohost_VkResolveImageInfo2(m_state,
                                                         (VkResolveImageInfo2*)(pResolveImageInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdResolveImage2 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer,
                            (unsigned long long)pResolveImageInfo);
                }
                vk->vkCmdResolveImage2(unboxed_commandBuffer, pResolveImageInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdResolveImage2(snapshotTraceBegin, snapshotTraceBytes,
                                                            &m_pool, commandBuffer,
                                                            pResolveImageInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBeginRendering: {
                android::base::beginTrace("vkCmdBeginRendering decode");
                VkCommandBuffer commandBuffer;
                const VkRenderingInfo* pRenderingInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pRenderingInfo, sizeof(const VkRenderingInfo));
                reservedunmarshal_VkRenderingInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                  (VkRenderingInfo*)(pRenderingInfo),
                                                  readStreamPtrPtr);
                if (pRenderingInfo) {
                    transform_tohost_VkRenderingInfo(m_state, (VkRenderingInfo*)(pRenderingInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdBeginRendering 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pRenderingInfo);
                }
                vk->vkCmdBeginRendering(unboxed_commandBuffer, pRenderingInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBeginRendering(snapshotTraceBegin, snapshotTraceBytes,
                                                             &m_pool, commandBuffer,
                                                             pRenderingInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdEndRendering: {
                android::base::beginTrace("vkCmdEndRendering decode");
                VkCommandBuffer commandBuffer;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdEndRendering 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer);
                }
                vk->vkCmdEndRendering(unboxed_commandBuffer);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdEndRendering(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, commandBuffer);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetCullMode: {
                android::base::beginTrace("vkCmdSetCullMode decode");
                VkCommandBuffer commandBuffer;
                VkCullModeFlags cullMode;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkCullModeFlags*)&cullMode, *readStreamPtrPtr, sizeof(VkCullModeFlags));
                *readStreamPtrPtr += sizeof(VkCullModeFlags);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetCullMode 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer, (unsigned long long)cullMode);
                }
                vk->vkCmdSetCullMode(unboxed_commandBuffer, cullMode);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetCullMode(snapshotTraceBegin, snapshotTraceBytes,
                                                          &m_pool, commandBuffer, cullMode);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetFrontFace: {
                android::base::beginTrace("vkCmdSetFrontFace decode");
                VkCommandBuffer commandBuffer;
                VkFrontFace frontFace;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkFrontFace*)&frontFace, *readStreamPtrPtr, sizeof(VkFrontFace));
                *readStreamPtrPtr += sizeof(VkFrontFace);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetFrontFace 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer, (unsigned long long)frontFace);
                }
                vk->vkCmdSetFrontFace(unboxed_commandBuffer, frontFace);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetFrontFace(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, commandBuffer, frontFace);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetPrimitiveTopology: {
                android::base::beginTrace("vkCmdSetPrimitiveTopology decode");
                VkCommandBuffer commandBuffer;
                VkPrimitiveTopology primitiveTopology;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkPrimitiveTopology*)&primitiveTopology, *readStreamPtrPtr,
                       sizeof(VkPrimitiveTopology));
                *readStreamPtrPtr += sizeof(VkPrimitiveTopology);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetPrimitiveTopology 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)primitiveTopology);
                }
                vk->vkCmdSetPrimitiveTopology(unboxed_commandBuffer, primitiveTopology);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetPrimitiveTopology(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        primitiveTopology);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetViewportWithCount: {
                android::base::beginTrace("vkCmdSetViewportWithCount decode");
                VkCommandBuffer commandBuffer;
                uint32_t viewportCount;
                const VkViewport* pViewports;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&viewportCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pViewports,
                                    ((viewportCount)) * sizeof(const VkViewport));
                for (uint32_t i = 0; i < (uint32_t)((viewportCount)); ++i) {
                    reservedunmarshal_VkViewport(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                 (VkViewport*)(pViewports + i), readStreamPtrPtr);
                }
                if (pViewports) {
                    for (uint32_t i = 0; i < (uint32_t)((viewportCount)); ++i) {
                        transform_tohost_VkViewport(m_state, (VkViewport*)(pViewports + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetViewportWithCount 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)viewportCount, (unsigned long long)pViewports);
                }
                vk->vkCmdSetViewportWithCount(unboxed_commandBuffer, viewportCount, pViewports);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetViewportWithCount(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        viewportCount, pViewports);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetScissorWithCount: {
                android::base::beginTrace("vkCmdSetScissorWithCount decode");
                VkCommandBuffer commandBuffer;
                uint32_t scissorCount;
                const VkRect2D* pScissors;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&scissorCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pScissors, ((scissorCount)) * sizeof(const VkRect2D));
                for (uint32_t i = 0; i < (uint32_t)((scissorCount)); ++i) {
                    reservedunmarshal_VkRect2D(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                               (VkRect2D*)(pScissors + i), readStreamPtrPtr);
                }
                if (pScissors) {
                    for (uint32_t i = 0; i < (uint32_t)((scissorCount)); ++i) {
                        transform_tohost_VkRect2D(m_state, (VkRect2D*)(pScissors + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetScissorWithCount 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)scissorCount, (unsigned long long)pScissors);
                }
                vk->vkCmdSetScissorWithCount(unboxed_commandBuffer, scissorCount, pScissors);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetScissorWithCount(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        scissorCount, pScissors);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBindVertexBuffers2: {
                android::base::beginTrace("vkCmdBindVertexBuffers2 decode");
                VkCommandBuffer commandBuffer;
                uint32_t firstBinding;
                uint32_t bindingCount;
                const VkBuffer* pBuffers;
                const VkDeviceSize* pOffsets;
                const VkDeviceSize* pSizes;
                const VkDeviceSize* pStrides;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&firstBinding, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                memcpy((uint32_t*)&bindingCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                // WARNING PTR CHECK
                memcpy((VkBuffer**)&pBuffers, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pBuffers);
                *readStreamPtrPtr += 8;
                if (pBuffers) {
                    vkReadStream->alloc((void**)&pBuffers,
                                        ((bindingCount)) * sizeof(const VkBuffer));
                    if (((bindingCount))) {
                        uint8_t* cgen_var_1_0_ptr = (uint8_t*)(*readStreamPtrPtr);
                        *readStreamPtrPtr += 8 * ((bindingCount));
                        for (uint32_t k = 0; k < ((bindingCount)); ++k) {
                            uint64_t tmpval;
                            memcpy(&tmpval, cgen_var_1_0_ptr + k * 8, sizeof(uint64_t));
                            *(((VkBuffer*)pBuffers) + k) =
                                (VkBuffer)unbox_VkBuffer((VkBuffer)tmpval);
                        }
                    }
                }
                vkReadStream->alloc((void**)&pOffsets,
                                    ((bindingCount)) * sizeof(const VkDeviceSize));
                memcpy((VkDeviceSize*)pOffsets, *readStreamPtrPtr,
                       ((bindingCount)) * sizeof(const VkDeviceSize));
                *readStreamPtrPtr += ((bindingCount)) * sizeof(const VkDeviceSize);
                // WARNING PTR CHECK
                memcpy((VkDeviceSize**)&pSizes, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSizes);
                *readStreamPtrPtr += 8;
                if (pSizes) {
                    vkReadStream->alloc((void**)&pSizes,
                                        ((bindingCount)) * sizeof(const VkDeviceSize));
                    memcpy((VkDeviceSize*)pSizes, *readStreamPtrPtr,
                           ((bindingCount)) * sizeof(const VkDeviceSize));
                    *readStreamPtrPtr += ((bindingCount)) * sizeof(const VkDeviceSize);
                }
                // WARNING PTR CHECK
                memcpy((VkDeviceSize**)&pStrides, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pStrides);
                *readStreamPtrPtr += 8;
                if (pStrides) {
                    vkReadStream->alloc((void**)&pStrides,
                                        ((bindingCount)) * sizeof(const VkDeviceSize));
                    memcpy((VkDeviceSize*)pStrides, *readStreamPtrPtr,
                           ((bindingCount)) * sizeof(const VkDeviceSize));
                    *readStreamPtrPtr += ((bindingCount)) * sizeof(const VkDeviceSize);
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdBindVertexBuffers2 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)firstBinding, (unsigned long long)bindingCount,
                            (unsigned long long)pBuffers, (unsigned long long)pOffsets,
                            (unsigned long long)pSizes, (unsigned long long)pStrides);
                }
                vk->vkCmdBindVertexBuffers2(unboxed_commandBuffer, firstBinding, bindingCount,
                                            pBuffers, pOffsets, pSizes, pStrides);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBindVertexBuffers2(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        firstBinding, bindingCount, pBuffers, pOffsets, pSizes, pStrides);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetDepthTestEnable: {
                android::base::beginTrace("vkCmdSetDepthTestEnable decode");
                VkCommandBuffer commandBuffer;
                VkBool32 depthTestEnable;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkBool32*)&depthTestEnable, *readStreamPtrPtr, sizeof(VkBool32));
                *readStreamPtrPtr += sizeof(VkBool32);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetDepthTestEnable 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)depthTestEnable);
                }
                vk->vkCmdSetDepthTestEnable(unboxed_commandBuffer, depthTestEnable);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetDepthTestEnable(snapshotTraceBegin,
                                                                 snapshotTraceBytes, &m_pool,
                                                                 commandBuffer, depthTestEnable);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetDepthWriteEnable: {
                android::base::beginTrace("vkCmdSetDepthWriteEnable decode");
                VkCommandBuffer commandBuffer;
                VkBool32 depthWriteEnable;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkBool32*)&depthWriteEnable, *readStreamPtrPtr, sizeof(VkBool32));
                *readStreamPtrPtr += sizeof(VkBool32);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetDepthWriteEnable 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)depthWriteEnable);
                }
                vk->vkCmdSetDepthWriteEnable(unboxed_commandBuffer, depthWriteEnable);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetDepthWriteEnable(snapshotTraceBegin,
                                                                  snapshotTraceBytes, &m_pool,
                                                                  commandBuffer, depthWriteEnable);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetDepthCompareOp: {
                android::base::beginTrace("vkCmdSetDepthCompareOp decode");
                VkCommandBuffer commandBuffer;
                VkCompareOp depthCompareOp;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkCompareOp*)&depthCompareOp, *readStreamPtrPtr, sizeof(VkCompareOp));
                *readStreamPtrPtr += sizeof(VkCompareOp);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetDepthCompareOp 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)depthCompareOp);
                }
                vk->vkCmdSetDepthCompareOp(unboxed_commandBuffer, depthCompareOp);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetDepthCompareOp(snapshotTraceBegin,
                                                                snapshotTraceBytes, &m_pool,
                                                                commandBuffer, depthCompareOp);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetDepthBoundsTestEnable: {
                android::base::beginTrace("vkCmdSetDepthBoundsTestEnable decode");
                VkCommandBuffer commandBuffer;
                VkBool32 depthBoundsTestEnable;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkBool32*)&depthBoundsTestEnable, *readStreamPtrPtr, sizeof(VkBool32));
                *readStreamPtrPtr += sizeof(VkBool32);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetDepthBoundsTestEnable 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)depthBoundsTestEnable);
                }
                vk->vkCmdSetDepthBoundsTestEnable(unboxed_commandBuffer, depthBoundsTestEnable);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetDepthBoundsTestEnable(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        depthBoundsTestEnable);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetStencilTestEnable: {
                android::base::beginTrace("vkCmdSetStencilTestEnable decode");
                VkCommandBuffer commandBuffer;
                VkBool32 stencilTestEnable;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkBool32*)&stencilTestEnable, *readStreamPtrPtr, sizeof(VkBool32));
                *readStreamPtrPtr += sizeof(VkBool32);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetStencilTestEnable 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)stencilTestEnable);
                }
                vk->vkCmdSetStencilTestEnable(unboxed_commandBuffer, stencilTestEnable);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetStencilTestEnable(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        stencilTestEnable);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetStencilOp: {
                android::base::beginTrace("vkCmdSetStencilOp decode");
                VkCommandBuffer commandBuffer;
                VkStencilFaceFlags faceMask;
                VkStencilOp failOp;
                VkStencilOp passOp;
                VkStencilOp depthFailOp;
                VkCompareOp compareOp;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkStencilFaceFlags*)&faceMask, *readStreamPtrPtr,
                       sizeof(VkStencilFaceFlags));
                *readStreamPtrPtr += sizeof(VkStencilFaceFlags);
                memcpy((VkStencilOp*)&failOp, *readStreamPtrPtr, sizeof(VkStencilOp));
                *readStreamPtrPtr += sizeof(VkStencilOp);
                memcpy((VkStencilOp*)&passOp, *readStreamPtrPtr, sizeof(VkStencilOp));
                *readStreamPtrPtr += sizeof(VkStencilOp);
                memcpy((VkStencilOp*)&depthFailOp, *readStreamPtrPtr, sizeof(VkStencilOp));
                *readStreamPtrPtr += sizeof(VkStencilOp);
                memcpy((VkCompareOp*)&compareOp, *readStreamPtrPtr, sizeof(VkCompareOp));
                *readStreamPtrPtr += sizeof(VkCompareOp);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetStencilOp 0x%llx 0x%llx 0x%llx 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)faceMask, (unsigned long long)failOp,
                            (unsigned long long)passOp, (unsigned long long)depthFailOp,
                            (unsigned long long)compareOp);
                }
                vk->vkCmdSetStencilOp(unboxed_commandBuffer, faceMask, failOp, passOp, depthFailOp,
                                      compareOp);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetStencilOp(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, commandBuffer, faceMask, failOp,
                                                           passOp, depthFailOp, compareOp);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetRasterizerDiscardEnable: {
                android::base::beginTrace("vkCmdSetRasterizerDiscardEnable decode");
                VkCommandBuffer commandBuffer;
                VkBool32 rasterizerDiscardEnable;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkBool32*)&rasterizerDiscardEnable, *readStreamPtrPtr, sizeof(VkBool32));
                *readStreamPtrPtr += sizeof(VkBool32);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetRasterizerDiscardEnable 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)rasterizerDiscardEnable);
                }
                vk->vkCmdSetRasterizerDiscardEnable(unboxed_commandBuffer, rasterizerDiscardEnable);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetRasterizerDiscardEnable(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        rasterizerDiscardEnable);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetDepthBiasEnable: {
                android::base::beginTrace("vkCmdSetDepthBiasEnable decode");
                VkCommandBuffer commandBuffer;
                VkBool32 depthBiasEnable;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkBool32*)&depthBiasEnable, *readStreamPtrPtr, sizeof(VkBool32));
                *readStreamPtrPtr += sizeof(VkBool32);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetDepthBiasEnable 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)depthBiasEnable);
                }
                vk->vkCmdSetDepthBiasEnable(unboxed_commandBuffer, depthBiasEnable);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetDepthBiasEnable(snapshotTraceBegin,
                                                                 snapshotTraceBytes, &m_pool,
                                                                 commandBuffer, depthBiasEnable);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdSetPrimitiveRestartEnable: {
                android::base::beginTrace("vkCmdSetPrimitiveRestartEnable decode");
                VkCommandBuffer commandBuffer;
                VkBool32 primitiveRestartEnable;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkBool32*)&primitiveRestartEnable, *readStreamPtrPtr, sizeof(VkBool32));
                *readStreamPtrPtr += sizeof(VkBool32);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdSetPrimitiveRestartEnable 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)primitiveRestartEnable);
                }
                vk->vkCmdSetPrimitiveRestartEnable(unboxed_commandBuffer, primitiveRestartEnable);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetPrimitiveRestartEnable(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        primitiveRestartEnable);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceBufferMemoryRequirements: {
                android::base::beginTrace("vkGetDeviceBufferMemoryRequirements decode");
                VkDevice device;
                const VkDeviceBufferMemoryRequirements* pInfo;
                VkMemoryRequirements2* pMemoryRequirements;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkDeviceBufferMemoryRequirements));
                reservedunmarshal_VkDeviceBufferMemoryRequirements(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDeviceBufferMemoryRequirements*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryRequirements, sizeof(VkMemoryRequirements2));
                reservedunmarshal_VkMemoryRequirements2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkMemoryRequirements2*)(pMemoryRequirements), readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkDeviceBufferMemoryRequirements(
                        m_state, (VkDeviceBufferMemoryRequirements*)(pInfo));
                }
                if (pMemoryRequirements) {
                    transform_tohost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetDeviceBufferMemoryRequirements 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                            (unsigned long long)pMemoryRequirements);
                }
                vk->vkGetDeviceBufferMemoryRequirements(unboxed_device, pInfo, pMemoryRequirements);
                vkStream->unsetHandleMapping();
                if (pMemoryRequirements) {
                    transform_fromhost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                marshal_VkMemoryRequirements2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                              (VkMemoryRequirements2*)(pMemoryRequirements));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceBufferMemoryRequirements(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceImageMemoryRequirements: {
                android::base::beginTrace("vkGetDeviceImageMemoryRequirements decode");
                VkDevice device;
                const VkDeviceImageMemoryRequirements* pInfo;
                VkMemoryRequirements2* pMemoryRequirements;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkDeviceImageMemoryRequirements));
                reservedunmarshal_VkDeviceImageMemoryRequirements(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDeviceImageMemoryRequirements*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryRequirements, sizeof(VkMemoryRequirements2));
                reservedunmarshal_VkMemoryRequirements2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkMemoryRequirements2*)(pMemoryRequirements), readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkDeviceImageMemoryRequirements(
                        m_state, (VkDeviceImageMemoryRequirements*)(pInfo));
                }
                if (pMemoryRequirements) {
                    transform_tohost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetDeviceImageMemoryRequirements 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                            (unsigned long long)pMemoryRequirements);
                }
                vk->vkGetDeviceImageMemoryRequirements(unboxed_device, pInfo, pMemoryRequirements);
                vkStream->unsetHandleMapping();
                if (pMemoryRequirements) {
                    transform_fromhost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                marshal_VkMemoryRequirements2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                              (VkMemoryRequirements2*)(pMemoryRequirements));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceImageMemoryRequirements(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceImageSparseMemoryRequirements: {
                android::base::beginTrace("vkGetDeviceImageSparseMemoryRequirements decode");
                VkDevice device;
                const VkDeviceImageMemoryRequirements* pInfo;
                uint32_t* pSparseMemoryRequirementCount;
                VkSparseImageMemoryRequirements2* pSparseMemoryRequirements;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkDeviceImageMemoryRequirements));
                reservedunmarshal_VkDeviceImageMemoryRequirements(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDeviceImageMemoryRequirements*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pSparseMemoryRequirementCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pSparseMemoryRequirementCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSparseMemoryRequirementCount);
                *readStreamPtrPtr += 8;
                if (pSparseMemoryRequirementCount) {
                    vkReadStream->alloc((void**)&pSparseMemoryRequirementCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pSparseMemoryRequirementCount, *readStreamPtrPtr,
                           sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pSparseMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkSparseImageMemoryRequirements2**)&pSparseMemoryRequirements,
                       (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSparseMemoryRequirements);
                *readStreamPtrPtr += 8;
                if (pSparseMemoryRequirements) {
                    vkReadStream->alloc((void**)&pSparseMemoryRequirements,
                                        (*(pSparseMemoryRequirementCount)) *
                                            sizeof(VkSparseImageMemoryRequirements2));
                    for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount)); ++i) {
                        reservedunmarshal_VkSparseImageMemoryRequirements2(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i),
                            readStreamPtrPtr);
                    }
                }
                if (pInfo) {
                    transform_tohost_VkDeviceImageMemoryRequirements(
                        m_state, (VkDeviceImageMemoryRequirements*)(pInfo));
                }
                if (pSparseMemoryRequirementCount) {
                    if (pSparseMemoryRequirements) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            transform_tohost_VkSparseImageMemoryRequirements2(
                                m_state,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetDeviceImageSparseMemoryRequirements 0x%llx "
                            "0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                            (unsigned long long)pSparseMemoryRequirementCount,
                            (unsigned long long)pSparseMemoryRequirements);
                }
                vk->vkGetDeviceImageSparseMemoryRequirements(unboxed_device, pInfo,
                                                             pSparseMemoryRequirementCount,
                                                             pSparseMemoryRequirements);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pSparseMemoryRequirementCount;
                vkStream->putBe64(cgen_var_3);
                if (pSparseMemoryRequirementCount) {
                    vkStream->write((uint32_t*)pSparseMemoryRequirementCount, sizeof(uint32_t));
                }
                if (pSparseMemoryRequirementCount) {
                    if (pSparseMemoryRequirements) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            transform_fromhost_VkSparseImageMemoryRequirements2(
                                m_state,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pSparseMemoryRequirements;
                vkStream->putBe64(cgen_var_4);
                if (pSparseMemoryRequirements) {
                    if (pSparseMemoryRequirementCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            marshal_VkSparseImageMemoryRequirements2(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceImageSparseMemoryRequirements(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pSparseMemoryRequirementCount, pSparseMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_surface
#endif
#ifdef VK_KHR_swapchain
#endif
#ifdef VK_KHR_xcb_surface
#endif
#ifdef VK_KHR_android_surface
#endif
#ifdef VK_KHR_win32_surface
#endif
#ifdef VK_KHR_dynamic_rendering
            case OP_vkCmdBeginRenderingKHR: {
                android::base::beginTrace("vkCmdBeginRenderingKHR decode");
                VkCommandBuffer commandBuffer;
                const VkRenderingInfo* pRenderingInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pRenderingInfo, sizeof(const VkRenderingInfo));
                reservedunmarshal_VkRenderingInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                  (VkRenderingInfo*)(pRenderingInfo),
                                                  readStreamPtrPtr);
                if (pRenderingInfo) {
                    transform_tohost_VkRenderingInfo(m_state, (VkRenderingInfo*)(pRenderingInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdBeginRenderingKHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pRenderingInfo);
                }
                vk->vkCmdBeginRenderingKHR(unboxed_commandBuffer, pRenderingInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBeginRenderingKHR(snapshotTraceBegin,
                                                                snapshotTraceBytes, &m_pool,
                                                                commandBuffer, pRenderingInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdEndRenderingKHR: {
                android::base::beginTrace("vkCmdEndRenderingKHR decode");
                VkCommandBuffer commandBuffer;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdEndRenderingKHR 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer);
                }
                vk->vkCmdEndRenderingKHR(unboxed_commandBuffer);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdEndRenderingKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_get_physical_device_properties2
            case OP_vkGetPhysicalDeviceFeatures2KHR: {
                android::base::beginTrace("vkGetPhysicalDeviceFeatures2KHR decode");
                VkPhysicalDevice physicalDevice;
                VkPhysicalDeviceFeatures2* pFeatures;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                // Begin manual dispatchable handle unboxing for pFeatures;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pFeatures, sizeof(VkPhysicalDeviceFeatures2));
                reservedunmarshal_VkPhysicalDeviceFeatures2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceFeatures2*)(pFeatures), readStreamPtrPtr);
                if (pFeatures) {
                    transform_tohost_VkPhysicalDeviceFeatures2(
                        m_state, (VkPhysicalDeviceFeatures2*)(pFeatures));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceFeatures2KHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pFeatures);
                }
                m_state->on_vkGetPhysicalDeviceFeatures2KHR(&m_pool, physicalDevice, pFeatures);
                vkStream->unsetHandleMapping();
                if (pFeatures) {
                    transform_fromhost_VkPhysicalDeviceFeatures2(
                        m_state, (VkPhysicalDeviceFeatures2*)(pFeatures));
                }
                marshal_VkPhysicalDeviceFeatures2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                  (VkPhysicalDeviceFeatures2*)(pFeatures));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceFeatures2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice, pFeatures);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceProperties2KHR: {
                android::base::beginTrace("vkGetPhysicalDeviceProperties2KHR decode");
                VkPhysicalDevice physicalDevice;
                VkPhysicalDeviceProperties2* pProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pProperties, sizeof(VkPhysicalDeviceProperties2));
                reservedunmarshal_VkPhysicalDeviceProperties2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceProperties2*)(pProperties), readStreamPtrPtr);
                if (pProperties) {
                    transform_tohost_VkPhysicalDeviceProperties2(
                        m_state, (VkPhysicalDeviceProperties2*)(pProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceProperties2KHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pProperties);
                }
                m_state->on_vkGetPhysicalDeviceProperties2KHR(&m_pool, physicalDevice, pProperties);
                vkStream->unsetHandleMapping();
                if (pProperties) {
                    transform_fromhost_VkPhysicalDeviceProperties2(
                        m_state, (VkPhysicalDeviceProperties2*)(pProperties));
                }
                marshal_VkPhysicalDeviceProperties2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkPhysicalDeviceProperties2*)(pProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceProperties2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceFormatProperties2KHR: {
                android::base::beginTrace("vkGetPhysicalDeviceFormatProperties2KHR decode");
                VkPhysicalDevice physicalDevice;
                VkFormat format;
                VkFormatProperties2* pFormatProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                memcpy((VkFormat*)&format, *readStreamPtrPtr, sizeof(VkFormat));
                *readStreamPtrPtr += sizeof(VkFormat);
                // Begin manual dispatchable handle unboxing for pFormatProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pFormatProperties, sizeof(VkFormatProperties2));
                reservedunmarshal_VkFormatProperties2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkFormatProperties2*)(pFormatProperties),
                                                      readStreamPtrPtr);
                if (pFormatProperties) {
                    transform_tohost_VkFormatProperties2(m_state,
                                                         (VkFormatProperties2*)(pFormatProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceFormatProperties2KHR 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)format, (unsigned long long)pFormatProperties);
                }
                m_state->on_vkGetPhysicalDeviceFormatProperties2KHR(&m_pool, physicalDevice, format,
                                                                    pFormatProperties);
                vkStream->unsetHandleMapping();
                if (pFormatProperties) {
                    transform_fromhost_VkFormatProperties2(
                        m_state, (VkFormatProperties2*)(pFormatProperties));
                }
                marshal_VkFormatProperties2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                            (VkFormatProperties2*)(pFormatProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceFormatProperties2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice, format,
                        pFormatProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceImageFormatProperties2KHR: {
                android::base::beginTrace("vkGetPhysicalDeviceImageFormatProperties2KHR decode");
                VkPhysicalDevice physicalDevice;
                const VkPhysicalDeviceImageFormatInfo2* pImageFormatInfo;
                VkImageFormatProperties2* pImageFormatProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pImageFormatInfo,
                                    sizeof(const VkPhysicalDeviceImageFormatInfo2));
                reservedunmarshal_VkPhysicalDeviceImageFormatInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceImageFormatInfo2*)(pImageFormatInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pImageFormatProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pImageFormatProperties,
                                    sizeof(VkImageFormatProperties2));
                reservedunmarshal_VkImageFormatProperties2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImageFormatProperties2*)(pImageFormatProperties), readStreamPtrPtr);
                if (pImageFormatInfo) {
                    transform_tohost_VkPhysicalDeviceImageFormatInfo2(
                        m_state, (VkPhysicalDeviceImageFormatInfo2*)(pImageFormatInfo));
                }
                if (pImageFormatProperties) {
                    transform_tohost_VkImageFormatProperties2(
                        m_state, (VkImageFormatProperties2*)(pImageFormatProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceImageFormatProperties2KHR 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pImageFormatInfo,
                            (unsigned long long)pImageFormatProperties);
                }
                VkResult vkGetPhysicalDeviceImageFormatProperties2KHR_VkResult_return = (VkResult)0;
                vkGetPhysicalDeviceImageFormatProperties2KHR_VkResult_return =
                    m_state->on_vkGetPhysicalDeviceImageFormatProperties2KHR(
                        &m_pool, physicalDevice, pImageFormatInfo, pImageFormatProperties);
                if ((vkGetPhysicalDeviceImageFormatProperties2KHR_VkResult_return) ==
                    VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(
                    vkGetPhysicalDeviceImageFormatProperties2KHR_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                if (pImageFormatProperties) {
                    transform_fromhost_VkImageFormatProperties2(
                        m_state, (VkImageFormatProperties2*)(pImageFormatProperties));
                }
                marshal_VkImageFormatProperties2(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImageFormatProperties2*)(pImageFormatProperties));
                vkStream->write(&vkGetPhysicalDeviceImageFormatProperties2KHR_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceImageFormatProperties2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetPhysicalDeviceImageFormatProperties2KHR_VkResult_return,
                        physicalDevice, pImageFormatInfo, pImageFormatProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceQueueFamilyProperties2KHR: {
                android::base::beginTrace("vkGetPhysicalDeviceQueueFamilyProperties2KHR decode");
                VkPhysicalDevice physicalDevice;
                uint32_t* pQueueFamilyPropertyCount;
                VkQueueFamilyProperties2* pQueueFamilyProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                // Begin manual dispatchable handle unboxing for pQueueFamilyPropertyCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pQueueFamilyPropertyCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pQueueFamilyPropertyCount);
                *readStreamPtrPtr += 8;
                if (pQueueFamilyPropertyCount) {
                    vkReadStream->alloc((void**)&pQueueFamilyPropertyCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pQueueFamilyPropertyCount, *readStreamPtrPtr,
                           sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pQueueFamilyProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkQueueFamilyProperties2**)&pQueueFamilyProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pQueueFamilyProperties);
                *readStreamPtrPtr += 8;
                if (pQueueFamilyProperties) {
                    vkReadStream->alloc(
                        (void**)&pQueueFamilyProperties,
                        (*(pQueueFamilyPropertyCount)) * sizeof(VkQueueFamilyProperties2));
                    for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                        reservedunmarshal_VkQueueFamilyProperties2(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkQueueFamilyProperties2*)(pQueueFamilyProperties + i),
                            readStreamPtrPtr);
                    }
                }
                if (pQueueFamilyPropertyCount) {
                    if (pQueueFamilyProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                            transform_tohost_VkQueueFamilyProperties2(
                                m_state, (VkQueueFamilyProperties2*)(pQueueFamilyProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceQueueFamilyProperties2KHR 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pQueueFamilyPropertyCount,
                            (unsigned long long)pQueueFamilyProperties);
                }
                vk->vkGetPhysicalDeviceQueueFamilyProperties2KHR(
                    unboxed_physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pQueueFamilyPropertyCount;
                vkStream->putBe64(cgen_var_3);
                if (pQueueFamilyPropertyCount) {
                    vkStream->write((uint32_t*)pQueueFamilyPropertyCount, sizeof(uint32_t));
                }
                if (pQueueFamilyPropertyCount) {
                    if (pQueueFamilyProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                            transform_fromhost_VkQueueFamilyProperties2(
                                m_state, (VkQueueFamilyProperties2*)(pQueueFamilyProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pQueueFamilyProperties;
                vkStream->putBe64(cgen_var_4);
                if (pQueueFamilyProperties) {
                    if (pQueueFamilyPropertyCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pQueueFamilyPropertyCount)); ++i) {
                            marshal_VkQueueFamilyProperties2(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkQueueFamilyProperties2*)(pQueueFamilyProperties + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceQueueFamilyProperties2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pQueueFamilyPropertyCount, pQueueFamilyProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceMemoryProperties2KHR: {
                android::base::beginTrace("vkGetPhysicalDeviceMemoryProperties2KHR decode");
                VkPhysicalDevice physicalDevice;
                VkPhysicalDeviceMemoryProperties2* pMemoryProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                // Begin manual dispatchable handle unboxing for pMemoryProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryProperties,
                                    sizeof(VkPhysicalDeviceMemoryProperties2));
                reservedunmarshal_VkPhysicalDeviceMemoryProperties2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceMemoryProperties2*)(pMemoryProperties), readStreamPtrPtr);
                if (pMemoryProperties) {
                    transform_tohost_VkPhysicalDeviceMemoryProperties2(
                        m_state, (VkPhysicalDeviceMemoryProperties2*)(pMemoryProperties));
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkGetPhysicalDeviceMemoryProperties2KHR 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)physicalDevice,
                        (unsigned long long)pMemoryProperties);
                }
                m_state->on_vkGetPhysicalDeviceMemoryProperties2KHR(&m_pool, physicalDevice,
                                                                    pMemoryProperties);
                vkStream->unsetHandleMapping();
                if (pMemoryProperties) {
                    transform_fromhost_VkPhysicalDeviceMemoryProperties2(
                        m_state, (VkPhysicalDeviceMemoryProperties2*)(pMemoryProperties));
                }
                marshal_VkPhysicalDeviceMemoryProperties2(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceMemoryProperties2*)(pMemoryProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceMemoryProperties2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pMemoryProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPhysicalDeviceSparseImageFormatProperties2KHR: {
                android::base::beginTrace(
                    "vkGetPhysicalDeviceSparseImageFormatProperties2KHR decode");
                VkPhysicalDevice physicalDevice;
                const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo;
                uint32_t* pPropertyCount;
                VkSparseImageFormatProperties2* pProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                vkReadStream->alloc((void**)&pFormatInfo,
                                    sizeof(const VkPhysicalDeviceSparseImageFormatInfo2));
                reservedunmarshal_VkPhysicalDeviceSparseImageFormatInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceSparseImageFormatInfo2*)(pFormatInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pPropertyCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pPropertyCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pPropertyCount);
                *readStreamPtrPtr += 8;
                if (pPropertyCount) {
                    vkReadStream->alloc((void**)&pPropertyCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pPropertyCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkSparseImageFormatProperties2**)&pProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pProperties);
                *readStreamPtrPtr += 8;
                if (pProperties) {
                    vkReadStream->alloc(
                        (void**)&pProperties,
                        (*(pPropertyCount)) * sizeof(VkSparseImageFormatProperties2));
                    for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                        reservedunmarshal_VkSparseImageFormatProperties2(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkSparseImageFormatProperties2*)(pProperties + i), readStreamPtrPtr);
                    }
                }
                if (pFormatInfo) {
                    transform_tohost_VkPhysicalDeviceSparseImageFormatInfo2(
                        m_state, (VkPhysicalDeviceSparseImageFormatInfo2*)(pFormatInfo));
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_tohost_VkSparseImageFormatProperties2(
                                m_state, (VkSparseImageFormatProperties2*)(pProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceSparseImageFormatProperties2KHR "
                            "0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pFormatInfo, (unsigned long long)pPropertyCount,
                            (unsigned long long)pProperties);
                }
                vk->vkGetPhysicalDeviceSparseImageFormatProperties2KHR(
                    unboxed_physicalDevice, pFormatInfo, pPropertyCount, pProperties);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pPropertyCount;
                vkStream->putBe64(cgen_var_3);
                if (pPropertyCount) {
                    vkStream->write((uint32_t*)pPropertyCount, sizeof(uint32_t));
                }
                if (pPropertyCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            transform_fromhost_VkSparseImageFormatProperties2(
                                m_state, (VkSparseImageFormatProperties2*)(pProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pProperties;
                vkStream->putBe64(cgen_var_4);
                if (pProperties) {
                    if (pPropertyCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pPropertyCount)); ++i) {
                            marshal_VkSparseImageFormatProperties2(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkSparseImageFormatProperties2*)(pProperties + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceSparseImageFormatProperties2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pFormatInfo, pPropertyCount, pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_maintenance1
            case OP_vkTrimCommandPoolKHR: {
                android::base::beginTrace("vkTrimCommandPoolKHR decode");
                VkDevice device;
                VkCommandPool commandPool;
                VkCommandPoolTrimFlags flags;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandPool*)&commandPool =
                    (VkCommandPool)unbox_VkCommandPool((VkCommandPool)(*&cgen_var_1));
                memcpy((VkCommandPoolTrimFlags*)&flags, *readStreamPtrPtr,
                       sizeof(VkCommandPoolTrimFlags));
                *readStreamPtrPtr += sizeof(VkCommandPoolTrimFlags);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkTrimCommandPoolKHR 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)commandPool,
                            (unsigned long long)flags);
                }
                vk->vkTrimCommandPoolKHR(unboxed_device, commandPool, flags);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkTrimCommandPoolKHR(snapshotTraceBegin,
                                                              snapshotTraceBytes, &m_pool, device,
                                                              commandPool, flags);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_external_memory_capabilities
            case OP_vkGetPhysicalDeviceExternalBufferPropertiesKHR: {
                android::base::beginTrace("vkGetPhysicalDeviceExternalBufferPropertiesKHR decode");
                VkPhysicalDevice physicalDevice;
                const VkPhysicalDeviceExternalBufferInfo* pExternalBufferInfo;
                VkExternalBufferProperties* pExternalBufferProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                vkReadStream->alloc((void**)&pExternalBufferInfo,
                                    sizeof(const VkPhysicalDeviceExternalBufferInfo));
                reservedunmarshal_VkPhysicalDeviceExternalBufferInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceExternalBufferInfo*)(pExternalBufferInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pExternalBufferProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pExternalBufferProperties,
                                    sizeof(VkExternalBufferProperties));
                reservedunmarshal_VkExternalBufferProperties(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalBufferProperties*)(pExternalBufferProperties), readStreamPtrPtr);
                if (pExternalBufferInfo) {
                    m_state->transformImpl_VkPhysicalDeviceExternalBufferInfo_tohost(
                        pExternalBufferInfo, 1);
                    transform_tohost_VkPhysicalDeviceExternalBufferInfo(
                        m_state, (VkPhysicalDeviceExternalBufferInfo*)(pExternalBufferInfo));
                }
                if (pExternalBufferProperties) {
                    m_state->transformImpl_VkExternalBufferProperties_tohost(
                        pExternalBufferProperties, 1);
                    transform_tohost_VkExternalBufferProperties(
                        m_state, (VkExternalBufferProperties*)(pExternalBufferProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceExternalBufferPropertiesKHR 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pExternalBufferInfo,
                            (unsigned long long)pExternalBufferProperties);
                }
                vk->vkGetPhysicalDeviceExternalBufferPropertiesKHR(
                    unboxed_physicalDevice, pExternalBufferInfo, pExternalBufferProperties);
                vkStream->unsetHandleMapping();
                if (pExternalBufferProperties) {
                    m_state->transformImpl_VkExternalBufferProperties_fromhost(
                        pExternalBufferProperties, 1);
                    transform_fromhost_VkExternalBufferProperties(
                        m_state, (VkExternalBufferProperties*)(pExternalBufferProperties));
                }
                marshal_VkExternalBufferProperties(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalBufferProperties*)(pExternalBufferProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceExternalBufferPropertiesKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pExternalBufferInfo, pExternalBufferProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_external_memory
#endif
#ifdef VK_KHR_external_memory_win32
#endif
#ifdef VK_KHR_external_memory_fd
#endif
#ifdef VK_KHR_external_semaphore_capabilities
            case OP_vkGetPhysicalDeviceExternalSemaphorePropertiesKHR: {
                android::base::beginTrace(
                    "vkGetPhysicalDeviceExternalSemaphorePropertiesKHR decode");
                VkPhysicalDevice physicalDevice;
                const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo;
                VkExternalSemaphoreProperties* pExternalSemaphoreProperties;
                // Begin global wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pExternalSemaphoreInfo,
                                    sizeof(const VkPhysicalDeviceExternalSemaphoreInfo));
                reservedunmarshal_VkPhysicalDeviceExternalSemaphoreInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceExternalSemaphoreInfo*)(pExternalSemaphoreInfo),
                    readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pExternalSemaphoreProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pExternalSemaphoreProperties,
                                    sizeof(VkExternalSemaphoreProperties));
                reservedunmarshal_VkExternalSemaphoreProperties(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalSemaphoreProperties*)(pExternalSemaphoreProperties),
                    readStreamPtrPtr);
                if (pExternalSemaphoreInfo) {
                    transform_tohost_VkPhysicalDeviceExternalSemaphoreInfo(
                        m_state, (VkPhysicalDeviceExternalSemaphoreInfo*)(pExternalSemaphoreInfo));
                }
                if (pExternalSemaphoreProperties) {
                    transform_tohost_VkExternalSemaphoreProperties(
                        m_state, (VkExternalSemaphoreProperties*)(pExternalSemaphoreProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceExternalSemaphorePropertiesKHR "
                            "0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pExternalSemaphoreInfo,
                            (unsigned long long)pExternalSemaphoreProperties);
                }
                m_state->on_vkGetPhysicalDeviceExternalSemaphorePropertiesKHR(
                    &m_pool, physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);
                vkStream->unsetHandleMapping();
                if (pExternalSemaphoreProperties) {
                    transform_fromhost_VkExternalSemaphoreProperties(
                        m_state, (VkExternalSemaphoreProperties*)(pExternalSemaphoreProperties));
                }
                marshal_VkExternalSemaphoreProperties(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalSemaphoreProperties*)(pExternalSemaphoreProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceExternalSemaphorePropertiesKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pExternalSemaphoreInfo, pExternalSemaphoreProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_external_semaphore
#endif
#ifdef VK_KHR_external_semaphore_win32
#endif
#ifdef VK_KHR_external_semaphore_fd
            case OP_vkImportSemaphoreFdKHR: {
                android::base::beginTrace("vkImportSemaphoreFdKHR decode");
                VkDevice device;
                const VkImportSemaphoreFdInfoKHR* pImportSemaphoreFdInfo;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pImportSemaphoreFdInfo,
                                    sizeof(const VkImportSemaphoreFdInfoKHR));
                reservedunmarshal_VkImportSemaphoreFdInfoKHR(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImportSemaphoreFdInfoKHR*)(pImportSemaphoreFdInfo), readStreamPtrPtr);
                if (pImportSemaphoreFdInfo) {
                    transform_tohost_VkImportSemaphoreFdInfoKHR(
                        m_state, (VkImportSemaphoreFdInfoKHR*)(pImportSemaphoreFdInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkImportSemaphoreFdKHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)pImportSemaphoreFdInfo);
                }
                VkResult vkImportSemaphoreFdKHR_VkResult_return = (VkResult)0;
                vkImportSemaphoreFdKHR_VkResult_return =
                    m_state->on_vkImportSemaphoreFdKHR(&m_pool, device, pImportSemaphoreFdInfo);
                if ((vkImportSemaphoreFdKHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkImportSemaphoreFdKHR_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkImportSemaphoreFdKHR_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkImportSemaphoreFdKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkImportSemaphoreFdKHR_VkResult_return, device, pImportSemaphoreFdInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetSemaphoreFdKHR: {
                android::base::beginTrace("vkGetSemaphoreFdKHR decode");
                VkDevice device;
                const VkSemaphoreGetFdInfoKHR* pGetFdInfo;
                int* pFd;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pGetFdInfo, sizeof(const VkSemaphoreGetFdInfoKHR));
                reservedunmarshal_VkSemaphoreGetFdInfoKHR(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                          (VkSemaphoreGetFdInfoKHR*)(pGetFdInfo),
                                                          readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pFd;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pFd, sizeof(int));
                memcpy((int*)pFd, *readStreamPtrPtr, sizeof(int));
                *readStreamPtrPtr += sizeof(int);
                if (pGetFdInfo) {
                    transform_tohost_VkSemaphoreGetFdInfoKHR(
                        m_state, (VkSemaphoreGetFdInfoKHR*)(pGetFdInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetSemaphoreFdKHR 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pGetFdInfo,
                            (unsigned long long)pFd);
                }
                VkResult vkGetSemaphoreFdKHR_VkResult_return = (VkResult)0;
                vkGetSemaphoreFdKHR_VkResult_return =
                    m_state->on_vkGetSemaphoreFdKHR(&m_pool, device, pGetFdInfo, pFd);
                if ((vkGetSemaphoreFdKHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkGetSemaphoreFdKHR_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write((int*)pFd, sizeof(int));
                vkStream->write(&vkGetSemaphoreFdKHR_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetSemaphoreFdKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetSemaphoreFdKHR_VkResult_return, device, pGetFdInfo, pFd);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_shader_float16_int8
#endif
#ifdef VK_KHR_incremental_present
#endif
#ifdef VK_KHR_descriptor_update_template
            case OP_vkCreateDescriptorUpdateTemplateKHR: {
                android::base::beginTrace("vkCreateDescriptorUpdateTemplateKHR decode");
                VkDevice device;
                const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkDescriptorUpdateTemplate* pDescriptorUpdateTemplate;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo,
                                    sizeof(const VkDescriptorUpdateTemplateCreateInfo));
                reservedunmarshal_VkDescriptorUpdateTemplateCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDescriptorUpdateTemplateCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pDescriptorUpdateTemplate;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pDescriptorUpdateTemplate,
                                    sizeof(VkDescriptorUpdateTemplate));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkDescriptorUpdateTemplate*)pDescriptorUpdateTemplate =
                    (VkDescriptorUpdateTemplate)(VkDescriptorUpdateTemplate)((
                        VkDescriptorUpdateTemplate)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkDescriptorUpdateTemplateCreateInfo(
                        m_state, (VkDescriptorUpdateTemplateCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateDescriptorUpdateTemplateKHR 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator,
                            (unsigned long long)pDescriptorUpdateTemplate);
                }
                VkResult vkCreateDescriptorUpdateTemplateKHR_VkResult_return = (VkResult)0;
                vkCreateDescriptorUpdateTemplateKHR_VkResult_return =
                    m_state->on_vkCreateDescriptorUpdateTemplateKHR(
                        &m_pool, device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate);
                if ((vkCreateDescriptorUpdateTemplateKHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateDescriptorUpdateTemplateKHR_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pDescriptorUpdateTemplate;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(
                    8 == sizeof(VkDescriptorUpdateTemplate),
                    "handle map overwrite requires VkDescriptorUpdateTemplate to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkDescriptorUpdateTemplate(
                    (VkDescriptorUpdateTemplate*)pDescriptorUpdateTemplate, 1);
                vkStream->write((VkDescriptorUpdateTemplate*)pDescriptorUpdateTemplate, 8 * 1);
                // Begin manual non dispatchable handle create for pDescriptorUpdateTemplate;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateDescriptorUpdateTemplateKHR_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateDescriptorUpdateTemplateKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateDescriptorUpdateTemplateKHR_VkResult_return, device, pCreateInfo,
                        pAllocator, pDescriptorUpdateTemplate);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroyDescriptorUpdateTemplateKHR: {
                android::base::beginTrace("vkDestroyDescriptorUpdateTemplateKHR decode");
                VkDevice device;
                VkDescriptorUpdateTemplate descriptorUpdateTemplate;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for
                // descriptorUpdateTemplate;
                VkDescriptorUpdateTemplate boxed_descriptorUpdateTemplate_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDescriptorUpdateTemplate*)&descriptorUpdateTemplate =
                    (VkDescriptorUpdateTemplate)(VkDescriptorUpdateTemplate)((
                        VkDescriptorUpdateTemplate)(*&cgen_var_1));
                boxed_descriptorUpdateTemplate_preserve = descriptorUpdateTemplate;
                descriptorUpdateTemplate =
                    unbox_VkDescriptorUpdateTemplate(descriptorUpdateTemplate);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkDestroyDescriptorUpdateTemplateKHR 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)descriptorUpdateTemplate,
                            (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroyDescriptorUpdateTemplateKHR(
                    &m_pool, device, descriptorUpdateTemplate, pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroyDescriptorUpdateTemplateKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_descriptorUpdateTemplate_preserve, pAllocator);
                }
                delete_VkDescriptorUpdateTemplate(boxed_descriptorUpdateTemplate_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkUpdateDescriptorSetWithTemplateKHR: {
                android::base::beginTrace("vkUpdateDescriptorSetWithTemplateKHR decode");
                VkDevice device;
                VkDescriptorSet descriptorSet;
                VkDescriptorUpdateTemplate descriptorUpdateTemplate;
                const void* pData;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDescriptorSet*)&descriptorSet =
                    (VkDescriptorSet)unbox_VkDescriptorSet((VkDescriptorSet)(*&cgen_var_1));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDescriptorUpdateTemplate*)&descriptorUpdateTemplate =
                    (VkDescriptorUpdateTemplate)unbox_VkDescriptorUpdateTemplate(
                        (VkDescriptorUpdateTemplate)(*&cgen_var_2));
                // WARNING PTR CHECK
                memcpy((void**)&pData, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pData);
                *readStreamPtrPtr += 8;
                if (pData) {
                    vkReadStream->alloc((void**)&pData, sizeof(const uint8_t));
                    memcpy((void*)pData, *readStreamPtrPtr, sizeof(const uint8_t));
                    *readStreamPtrPtr += sizeof(const uint8_t);
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkUpdateDescriptorSetWithTemplateKHR 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)descriptorSet,
                            (unsigned long long)descriptorUpdateTemplate,
                            (unsigned long long)pData);
                }
                vk->vkUpdateDescriptorSetWithTemplateKHR(unboxed_device, descriptorSet,
                                                         descriptorUpdateTemplate, pData);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkUpdateDescriptorSetWithTemplateKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, descriptorSet,
                        descriptorUpdateTemplate, pData);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_imageless_framebuffer
#endif
#ifdef VK_KHR_create_renderpass2
            case OP_vkCreateRenderPass2KHR: {
                android::base::beginTrace("vkCreateRenderPass2KHR decode");
                VkDevice device;
                const VkRenderPassCreateInfo2* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkRenderPass* pRenderPass;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo, sizeof(const VkRenderPassCreateInfo2));
                reservedunmarshal_VkRenderPassCreateInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                          (VkRenderPassCreateInfo2*)(pCreateInfo),
                                                          readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pRenderPass;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pRenderPass, sizeof(VkRenderPass));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkRenderPass*)pRenderPass =
                    (VkRenderPass)(VkRenderPass)((VkRenderPass)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkRenderPassCreateInfo2(
                        m_state, (VkRenderPassCreateInfo2*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateRenderPass2KHR 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pRenderPass);
                }
                VkResult vkCreateRenderPass2KHR_VkResult_return = (VkResult)0;
                vkCreateRenderPass2KHR_VkResult_return = m_state->on_vkCreateRenderPass2KHR(
                    &m_pool, device, pCreateInfo, pAllocator, pRenderPass);
                if ((vkCreateRenderPass2KHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateRenderPass2KHR_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pRenderPass;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(8 == sizeof(VkRenderPass),
                              "handle map overwrite requires VkRenderPass to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkRenderPass((VkRenderPass*)pRenderPass, 1);
                vkStream->write((VkRenderPass*)pRenderPass, 8 * 1);
                // Begin manual non dispatchable handle create for pRenderPass;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateRenderPass2KHR_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateRenderPass2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateRenderPass2KHR_VkResult_return, device, pCreateInfo, pAllocator,
                        pRenderPass);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBeginRenderPass2KHR: {
                android::base::beginTrace("vkCmdBeginRenderPass2KHR decode");
                VkCommandBuffer commandBuffer;
                const VkRenderPassBeginInfo* pRenderPassBegin;
                const VkSubpassBeginInfo* pSubpassBeginInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pRenderPassBegin, sizeof(const VkRenderPassBeginInfo));
                reservedunmarshal_VkRenderPassBeginInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                        (VkRenderPassBeginInfo*)(pRenderPassBegin),
                                                        readStreamPtrPtr);
                vkReadStream->alloc((void**)&pSubpassBeginInfo, sizeof(const VkSubpassBeginInfo));
                reservedunmarshal_VkSubpassBeginInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkSubpassBeginInfo*)(pSubpassBeginInfo),
                                                     readStreamPtrPtr);
                if (pRenderPassBegin) {
                    transform_tohost_VkRenderPassBeginInfo(
                        m_state, (VkRenderPassBeginInfo*)(pRenderPassBegin));
                }
                if (pSubpassBeginInfo) {
                    transform_tohost_VkSubpassBeginInfo(m_state,
                                                        (VkSubpassBeginInfo*)(pSubpassBeginInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdBeginRenderPass2KHR 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pRenderPassBegin,
                            (unsigned long long)pSubpassBeginInfo);
                }
                vk->vkCmdBeginRenderPass2KHR(unboxed_commandBuffer, pRenderPassBegin,
                                             pSubpassBeginInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBeginRenderPass2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        pRenderPassBegin, pSubpassBeginInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdNextSubpass2KHR: {
                android::base::beginTrace("vkCmdNextSubpass2KHR decode");
                VkCommandBuffer commandBuffer;
                const VkSubpassBeginInfo* pSubpassBeginInfo;
                const VkSubpassEndInfo* pSubpassEndInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pSubpassBeginInfo, sizeof(const VkSubpassBeginInfo));
                reservedunmarshal_VkSubpassBeginInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkSubpassBeginInfo*)(pSubpassBeginInfo),
                                                     readStreamPtrPtr);
                vkReadStream->alloc((void**)&pSubpassEndInfo, sizeof(const VkSubpassEndInfo));
                reservedunmarshal_VkSubpassEndInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkSubpassEndInfo*)(pSubpassEndInfo),
                                                   readStreamPtrPtr);
                if (pSubpassBeginInfo) {
                    transform_tohost_VkSubpassBeginInfo(m_state,
                                                        (VkSubpassBeginInfo*)(pSubpassBeginInfo));
                }
                if (pSubpassEndInfo) {
                    transform_tohost_VkSubpassEndInfo(m_state,
                                                      (VkSubpassEndInfo*)(pSubpassEndInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdNextSubpass2KHR 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pSubpassBeginInfo,
                            (unsigned long long)pSubpassEndInfo);
                }
                vk->vkCmdNextSubpass2KHR(unboxed_commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdNextSubpass2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        pSubpassBeginInfo, pSubpassEndInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdEndRenderPass2KHR: {
                android::base::beginTrace("vkCmdEndRenderPass2KHR decode");
                VkCommandBuffer commandBuffer;
                const VkSubpassEndInfo* pSubpassEndInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pSubpassEndInfo, sizeof(const VkSubpassEndInfo));
                reservedunmarshal_VkSubpassEndInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkSubpassEndInfo*)(pSubpassEndInfo),
                                                   readStreamPtrPtr);
                if (pSubpassEndInfo) {
                    transform_tohost_VkSubpassEndInfo(m_state,
                                                      (VkSubpassEndInfo*)(pSubpassEndInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdEndRenderPass2KHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pSubpassEndInfo);
                }
                vk->vkCmdEndRenderPass2KHR(unboxed_commandBuffer, pSubpassEndInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdEndRenderPass2KHR(snapshotTraceBegin,
                                                                snapshotTraceBytes, &m_pool,
                                                                commandBuffer, pSubpassEndInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_external_fence_capabilities
            case OP_vkGetPhysicalDeviceExternalFencePropertiesKHR: {
                android::base::beginTrace("vkGetPhysicalDeviceExternalFencePropertiesKHR decode");
                VkPhysicalDevice physicalDevice;
                const VkPhysicalDeviceExternalFenceInfo* pExternalFenceInfo;
                VkExternalFenceProperties* pExternalFenceProperties;
                // Begin non wrapped dispatchable handle unboxing for physicalDevice;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkPhysicalDevice*)&physicalDevice =
                    (VkPhysicalDevice)(VkPhysicalDevice)((VkPhysicalDevice)(*&cgen_var_0));
                auto unboxed_physicalDevice = unbox_VkPhysicalDevice(physicalDevice);
                auto vk = dispatch_VkPhysicalDevice(physicalDevice);
                // End manual dispatchable handle unboxing for physicalDevice;
                vkReadStream->alloc((void**)&pExternalFenceInfo,
                                    sizeof(const VkPhysicalDeviceExternalFenceInfo));
                reservedunmarshal_VkPhysicalDeviceExternalFenceInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPhysicalDeviceExternalFenceInfo*)(pExternalFenceInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pExternalFenceProperties;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pExternalFenceProperties,
                                    sizeof(VkExternalFenceProperties));
                reservedunmarshal_VkExternalFenceProperties(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalFenceProperties*)(pExternalFenceProperties), readStreamPtrPtr);
                if (pExternalFenceInfo) {
                    transform_tohost_VkPhysicalDeviceExternalFenceInfo(
                        m_state, (VkPhysicalDeviceExternalFenceInfo*)(pExternalFenceInfo));
                }
                if (pExternalFenceProperties) {
                    transform_tohost_VkExternalFenceProperties(
                        m_state, (VkExternalFenceProperties*)(pExternalFenceProperties));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPhysicalDeviceExternalFencePropertiesKHR 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)physicalDevice,
                            (unsigned long long)pExternalFenceInfo,
                            (unsigned long long)pExternalFenceProperties);
                }
                vk->vkGetPhysicalDeviceExternalFencePropertiesKHR(
                    unboxed_physicalDevice, pExternalFenceInfo, pExternalFenceProperties);
                vkStream->unsetHandleMapping();
                if (pExternalFenceProperties) {
                    transform_fromhost_VkExternalFenceProperties(
                        m_state, (VkExternalFenceProperties*)(pExternalFenceProperties));
                }
                marshal_VkExternalFenceProperties(
                    vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkExternalFenceProperties*)(pExternalFenceProperties));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPhysicalDeviceExternalFencePropertiesKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, physicalDevice,
                        pExternalFenceInfo, pExternalFenceProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_external_fence
#endif
#ifdef VK_KHR_external_fence_fd
            case OP_vkImportFenceFdKHR: {
                android::base::beginTrace("vkImportFenceFdKHR decode");
                VkDevice device;
                const VkImportFenceFdInfoKHR* pImportFenceFdInfo;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pImportFenceFdInfo,
                                    sizeof(const VkImportFenceFdInfoKHR));
                reservedunmarshal_VkImportFenceFdInfoKHR(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImportFenceFdInfoKHR*)(pImportFenceFdInfo), readStreamPtrPtr);
                if (pImportFenceFdInfo) {
                    transform_tohost_VkImportFenceFdInfoKHR(
                        m_state, (VkImportFenceFdInfoKHR*)(pImportFenceFdInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkImportFenceFdKHR 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)device, (unsigned long long)pImportFenceFdInfo);
                }
                VkResult vkImportFenceFdKHR_VkResult_return = (VkResult)0;
                vkImportFenceFdKHR_VkResult_return =
                    vk->vkImportFenceFdKHR(unboxed_device, pImportFenceFdInfo);
                if ((vkImportFenceFdKHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkImportFenceFdKHR_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkImportFenceFdKHR_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkImportFenceFdKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkImportFenceFdKHR_VkResult_return, device, pImportFenceFdInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetFenceFdKHR: {
                android::base::beginTrace("vkGetFenceFdKHR decode");
                VkDevice device;
                const VkFenceGetFdInfoKHR* pGetFdInfo;
                int* pFd;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pGetFdInfo, sizeof(const VkFenceGetFdInfoKHR));
                reservedunmarshal_VkFenceGetFdInfoKHR(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkFenceGetFdInfoKHR*)(pGetFdInfo),
                                                      readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pFd;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pFd, sizeof(int));
                memcpy((int*)pFd, *readStreamPtrPtr, sizeof(int));
                *readStreamPtrPtr += sizeof(int);
                if (pGetFdInfo) {
                    transform_tohost_VkFenceGetFdInfoKHR(m_state,
                                                         (VkFenceGetFdInfoKHR*)(pGetFdInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetFenceFdKHR 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pGetFdInfo,
                            (unsigned long long)pFd);
                }
                VkResult vkGetFenceFdKHR_VkResult_return = (VkResult)0;
                vkGetFenceFdKHR_VkResult_return =
                    vk->vkGetFenceFdKHR(unboxed_device, pGetFdInfo, pFd);
                if ((vkGetFenceFdKHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkGetFenceFdKHR_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write((int*)pFd, sizeof(int));
                vkStream->write(&vkGetFenceFdKHR_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetFenceFdKHR(snapshotTraceBegin, snapshotTraceBytes,
                                                         &m_pool, vkGetFenceFdKHR_VkResult_return,
                                                         device, pGetFdInfo, pFd);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_maintenance2
#endif
#ifdef VK_KHR_dedicated_allocation
#endif
#ifdef VK_KHR_storage_buffer_storage_class
#endif
#ifdef VK_KHR_get_memory_requirements2
            case OP_vkGetImageMemoryRequirements2KHR: {
                android::base::beginTrace("vkGetImageMemoryRequirements2KHR decode");
                VkDevice device;
                const VkImageMemoryRequirementsInfo2* pInfo;
                VkMemoryRequirements2* pMemoryRequirements;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkImageMemoryRequirementsInfo2));
                reservedunmarshal_VkImageMemoryRequirementsInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImageMemoryRequirementsInfo2*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryRequirements, sizeof(VkMemoryRequirements2));
                reservedunmarshal_VkMemoryRequirements2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkMemoryRequirements2*)(pMemoryRequirements), readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkImageMemoryRequirementsInfo2(
                        m_state, (VkImageMemoryRequirementsInfo2*)(pInfo));
                }
                if (pMemoryRequirements) {
                    transform_tohost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkGetImageMemoryRequirements2KHR 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                        (unsigned long long)pMemoryRequirements);
                }
                m_state->on_vkGetImageMemoryRequirements2KHR(&m_pool, device, pInfo,
                                                             pMemoryRequirements);
                vkStream->unsetHandleMapping();
                if (pMemoryRequirements) {
                    transform_fromhost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                marshal_VkMemoryRequirements2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                              (VkMemoryRequirements2*)(pMemoryRequirements));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetImageMemoryRequirements2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetBufferMemoryRequirements2KHR: {
                android::base::beginTrace("vkGetBufferMemoryRequirements2KHR decode");
                VkDevice device;
                const VkBufferMemoryRequirementsInfo2* pInfo;
                VkMemoryRequirements2* pMemoryRequirements;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkBufferMemoryRequirementsInfo2));
                reservedunmarshal_VkBufferMemoryRequirementsInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkBufferMemoryRequirementsInfo2*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryRequirements, sizeof(VkMemoryRequirements2));
                reservedunmarshal_VkMemoryRequirements2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkMemoryRequirements2*)(pMemoryRequirements), readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkBufferMemoryRequirementsInfo2(
                        m_state, (VkBufferMemoryRequirementsInfo2*)(pInfo));
                }
                if (pMemoryRequirements) {
                    transform_tohost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkGetBufferMemoryRequirements2KHR 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                        (unsigned long long)pMemoryRequirements);
                }
                m_state->on_vkGetBufferMemoryRequirements2KHR(&m_pool, device, pInfo,
                                                              pMemoryRequirements);
                vkStream->unsetHandleMapping();
                if (pMemoryRequirements) {
                    transform_fromhost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                marshal_VkMemoryRequirements2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                              (VkMemoryRequirements2*)(pMemoryRequirements));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetBufferMemoryRequirements2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetImageSparseMemoryRequirements2KHR: {
                android::base::beginTrace("vkGetImageSparseMemoryRequirements2KHR decode");
                VkDevice device;
                const VkImageSparseMemoryRequirementsInfo2* pInfo;
                uint32_t* pSparseMemoryRequirementCount;
                VkSparseImageMemoryRequirements2* pSparseMemoryRequirements;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo,
                                    sizeof(const VkImageSparseMemoryRequirementsInfo2));
                reservedunmarshal_VkImageSparseMemoryRequirementsInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkImageSparseMemoryRequirementsInfo2*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pSparseMemoryRequirementCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pSparseMemoryRequirementCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSparseMemoryRequirementCount);
                *readStreamPtrPtr += 8;
                if (pSparseMemoryRequirementCount) {
                    vkReadStream->alloc((void**)&pSparseMemoryRequirementCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pSparseMemoryRequirementCount, *readStreamPtrPtr,
                           sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pSparseMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkSparseImageMemoryRequirements2**)&pSparseMemoryRequirements,
                       (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSparseMemoryRequirements);
                *readStreamPtrPtr += 8;
                if (pSparseMemoryRequirements) {
                    vkReadStream->alloc((void**)&pSparseMemoryRequirements,
                                        (*(pSparseMemoryRequirementCount)) *
                                            sizeof(VkSparseImageMemoryRequirements2));
                    for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount)); ++i) {
                        reservedunmarshal_VkSparseImageMemoryRequirements2(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i),
                            readStreamPtrPtr);
                    }
                }
                if (pInfo) {
                    transform_tohost_VkImageSparseMemoryRequirementsInfo2(
                        m_state, (VkImageSparseMemoryRequirementsInfo2*)(pInfo));
                }
                if (pSparseMemoryRequirementCount) {
                    if (pSparseMemoryRequirements) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            transform_tohost_VkSparseImageMemoryRequirements2(
                                m_state,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetImageSparseMemoryRequirements2KHR 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                            (unsigned long long)pSparseMemoryRequirementCount,
                            (unsigned long long)pSparseMemoryRequirements);
                }
                vk->vkGetImageSparseMemoryRequirements2KHR(unboxed_device, pInfo,
                                                           pSparseMemoryRequirementCount,
                                                           pSparseMemoryRequirements);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pSparseMemoryRequirementCount;
                vkStream->putBe64(cgen_var_3);
                if (pSparseMemoryRequirementCount) {
                    vkStream->write((uint32_t*)pSparseMemoryRequirementCount, sizeof(uint32_t));
                }
                if (pSparseMemoryRequirementCount) {
                    if (pSparseMemoryRequirements) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            transform_fromhost_VkSparseImageMemoryRequirements2(
                                m_state,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pSparseMemoryRequirements;
                vkStream->putBe64(cgen_var_4);
                if (pSparseMemoryRequirements) {
                    if (pSparseMemoryRequirementCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            marshal_VkSparseImageMemoryRequirements2(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetImageSparseMemoryRequirements2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pSparseMemoryRequirementCount, pSparseMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_image_format_list
#endif
#ifdef VK_KHR_sampler_ycbcr_conversion
            case OP_vkCreateSamplerYcbcrConversionKHR: {
                android::base::beginTrace("vkCreateSamplerYcbcrConversionKHR decode");
                VkDevice device;
                const VkSamplerYcbcrConversionCreateInfo* pCreateInfo;
                const VkAllocationCallbacks* pAllocator;
                VkSamplerYcbcrConversion* pYcbcrConversion;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCreateInfo,
                                    sizeof(const VkSamplerYcbcrConversionCreateInfo));
                reservedunmarshal_VkSamplerYcbcrConversionCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkSamplerYcbcrConversionCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                // Begin manual dispatchable handle unboxing for pYcbcrConversion;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pYcbcrConversion, sizeof(VkSamplerYcbcrConversion));
                uint64_t cgen_var_2;
                memcpy((uint64_t*)&cgen_var_2, *readStreamPtrPtr, 8);
                *readStreamPtrPtr += 8;
                *(VkSamplerYcbcrConversion*)pYcbcrConversion =
                    (VkSamplerYcbcrConversion)(VkSamplerYcbcrConversion)((
                        VkSamplerYcbcrConversion)(*&cgen_var_2));
                if (pCreateInfo) {
                    transform_tohost_VkSamplerYcbcrConversionCreateInfo(
                        m_state, (VkSamplerYcbcrConversionCreateInfo*)(pCreateInfo));
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCreateSamplerYcbcrConversionKHR 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pAllocator, (unsigned long long)pYcbcrConversion);
                }
                VkResult vkCreateSamplerYcbcrConversionKHR_VkResult_return = (VkResult)0;
                vkCreateSamplerYcbcrConversionKHR_VkResult_return =
                    m_state->on_vkCreateSamplerYcbcrConversionKHR(&m_pool, device, pCreateInfo,
                                                                  pAllocator, pYcbcrConversion);
                if ((vkCreateSamplerYcbcrConversionKHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkCreateSamplerYcbcrConversionKHR_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // Begin manual non dispatchable handle create for pYcbcrConversion;
                vkStream->unsetHandleMapping();
                uint64_t cgen_var_3;
                static_assert(
                    8 == sizeof(VkSamplerYcbcrConversion),
                    "handle map overwrite requires VkSamplerYcbcrConversion to be 8 bytes long");
                vkStream->handleMapping()->mapHandles_VkSamplerYcbcrConversion(
                    (VkSamplerYcbcrConversion*)pYcbcrConversion, 1);
                vkStream->write((VkSamplerYcbcrConversion*)pYcbcrConversion, 8 * 1);
                // Begin manual non dispatchable handle create for pYcbcrConversion;
                vkStream->setHandleMapping(&m_boxedHandleUnwrapMapping);
                vkStream->write(&vkCreateSamplerYcbcrConversionKHR_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCreateSamplerYcbcrConversionKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkCreateSamplerYcbcrConversionKHR_VkResult_return, device, pCreateInfo,
                        pAllocator, pYcbcrConversion);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkDestroySamplerYcbcrConversionKHR: {
                android::base::beginTrace("vkDestroySamplerYcbcrConversionKHR decode");
                VkDevice device;
                VkSamplerYcbcrConversion ycbcrConversion;
                const VkAllocationCallbacks* pAllocator;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                // Begin manual non dispatchable handle destroy unboxing for ycbcrConversion;
                VkSamplerYcbcrConversion boxed_ycbcrConversion_preserve;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkSamplerYcbcrConversion*)&ycbcrConversion =
                    (VkSamplerYcbcrConversion)(VkSamplerYcbcrConversion)((
                        VkSamplerYcbcrConversion)(*&cgen_var_1));
                boxed_ycbcrConversion_preserve = ycbcrConversion;
                ycbcrConversion = unbox_VkSamplerYcbcrConversion(ycbcrConversion);
                // WARNING PTR CHECK
                memcpy((VkAllocationCallbacks**)&pAllocator, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pAllocator);
                *readStreamPtrPtr += 8;
                if (pAllocator) {
                    vkReadStream->alloc((void**)&pAllocator, sizeof(const VkAllocationCallbacks));
                    reservedunmarshal_VkAllocationCallbacks(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkAllocationCallbacks*)(pAllocator), readStreamPtrPtr);
                }
                if (pAllocator) {
                    transform_tohost_VkAllocationCallbacks(m_state,
                                                           (VkAllocationCallbacks*)(pAllocator));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkDestroySamplerYcbcrConversionKHR 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)ycbcrConversion, (unsigned long long)pAllocator);
                }
                m_state->on_vkDestroySamplerYcbcrConversionKHR(&m_pool, device, ycbcrConversion,
                                                               pAllocator);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkDestroySamplerYcbcrConversionKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device,
                        boxed_ycbcrConversion_preserve, pAllocator);
                }
                delete_VkSamplerYcbcrConversion(boxed_ycbcrConversion_preserve);
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_bind_memory2
            case OP_vkBindBufferMemory2KHR: {
                android::base::beginTrace("vkBindBufferMemory2KHR decode");
                VkDevice device;
                uint32_t bindInfoCount;
                const VkBindBufferMemoryInfo* pBindInfos;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                memcpy((uint32_t*)&bindInfoCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pBindInfos,
                                    ((bindInfoCount)) * sizeof(const VkBindBufferMemoryInfo));
                for (uint32_t i = 0; i < (uint32_t)((bindInfoCount)); ++i) {
                    reservedunmarshal_VkBindBufferMemoryInfo(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkBindBufferMemoryInfo*)(pBindInfos + i), readStreamPtrPtr);
                }
                if (pBindInfos) {
                    for (uint32_t i = 0; i < (uint32_t)((bindInfoCount)); ++i) {
                        transform_tohost_VkBindBufferMemoryInfo(
                            m_state, (VkBindBufferMemoryInfo*)(pBindInfos + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkBindBufferMemory2KHR 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)bindInfoCount,
                            (unsigned long long)pBindInfos);
                }
                VkResult vkBindBufferMemory2KHR_VkResult_return = (VkResult)0;
                vkBindBufferMemory2KHR_VkResult_return =
                    m_state->on_vkBindBufferMemory2KHR(&m_pool, device, bindInfoCount, pBindInfos);
                if ((vkBindBufferMemory2KHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkBindBufferMemory2KHR_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkBindBufferMemory2KHR_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkBindBufferMemory2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkBindBufferMemory2KHR_VkResult_return, device, bindInfoCount, pBindInfos);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkBindImageMemory2KHR: {
                android::base::beginTrace("vkBindImageMemory2KHR decode");
                VkDevice device;
                uint32_t bindInfoCount;
                const VkBindImageMemoryInfo* pBindInfos;
                // Begin global wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                memcpy((uint32_t*)&bindInfoCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pBindInfos,
                                    ((bindInfoCount)) * sizeof(const VkBindImageMemoryInfo));
                for (uint32_t i = 0; i < (uint32_t)((bindInfoCount)); ++i) {
                    reservedunmarshal_VkBindImageMemoryInfo(
                        vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                        (VkBindImageMemoryInfo*)(pBindInfos + i), readStreamPtrPtr);
                }
                if (pBindInfos) {
                    for (uint32_t i = 0; i < (uint32_t)((bindInfoCount)); ++i) {
                        transform_tohost_VkBindImageMemoryInfo(
                            m_state, (VkBindImageMemoryInfo*)(pBindInfos + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkBindImageMemory2KHR 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)bindInfoCount,
                            (unsigned long long)pBindInfos);
                }
                VkResult vkBindImageMemory2KHR_VkResult_return = (VkResult)0;
                vkBindImageMemory2KHR_VkResult_return =
                    m_state->on_vkBindImageMemory2KHR(&m_pool, device, bindInfoCount, pBindInfos);
                if ((vkBindImageMemory2KHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkBindImageMemory2KHR_VkResult_return, opcode,
                                             context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkBindImageMemory2KHR_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkBindImageMemory2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkBindImageMemory2KHR_VkResult_return, device, bindInfoCount, pBindInfos);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_maintenance3
            case OP_vkGetDescriptorSetLayoutSupportKHR: {
                android::base::beginTrace("vkGetDescriptorSetLayoutSupportKHR decode");
                VkDevice device;
                const VkDescriptorSetLayoutCreateInfo* pCreateInfo;
                VkDescriptorSetLayoutSupport* pSupport;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pCreateInfo,
                                    sizeof(const VkDescriptorSetLayoutCreateInfo));
                reservedunmarshal_VkDescriptorSetLayoutCreateInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDescriptorSetLayoutCreateInfo*)(pCreateInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pSupport;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pSupport, sizeof(VkDescriptorSetLayoutSupport));
                reservedunmarshal_VkDescriptorSetLayoutSupport(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDescriptorSetLayoutSupport*)(pSupport), readStreamPtrPtr);
                if (pCreateInfo) {
                    transform_tohost_VkDescriptorSetLayoutCreateInfo(
                        m_state, (VkDescriptorSetLayoutCreateInfo*)(pCreateInfo));
                }
                if (pSupport) {
                    transform_tohost_VkDescriptorSetLayoutSupport(
                        m_state, (VkDescriptorSetLayoutSupport*)(pSupport));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetDescriptorSetLayoutSupportKHR 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pCreateInfo,
                            (unsigned long long)pSupport);
                }
                vk->vkGetDescriptorSetLayoutSupportKHR(unboxed_device, pCreateInfo, pSupport);
                vkStream->unsetHandleMapping();
                if (pSupport) {
                    transform_fromhost_VkDescriptorSetLayoutSupport(
                        m_state, (VkDescriptorSetLayoutSupport*)(pSupport));
                }
                marshal_VkDescriptorSetLayoutSupport(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                     (VkDescriptorSetLayoutSupport*)(pSupport));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDescriptorSetLayoutSupportKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pCreateInfo,
                        pSupport);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_shader_subgroup_extended_types
#endif
#ifdef VK_KHR_vulkan_memory_model
#endif
#ifdef VK_KHR_shader_terminate_invocation
#endif
#ifdef VK_KHR_buffer_device_address
            case OP_vkGetBufferDeviceAddressKHR: {
                android::base::beginTrace("vkGetBufferDeviceAddressKHR decode");
                VkDevice device;
                const VkBufferDeviceAddressInfo* pInfo;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkBufferDeviceAddressInfo));
                reservedunmarshal_VkBufferDeviceAddressInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM, (VkBufferDeviceAddressInfo*)(pInfo),
                    readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkBufferDeviceAddressInfo(m_state,
                                                               (VkBufferDeviceAddressInfo*)(pInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkGetBufferDeviceAddressKHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo);
                }
                VkDeviceAddress vkGetBufferDeviceAddressKHR_VkDeviceAddress_return =
                    (VkDeviceAddress)0;
                vkGetBufferDeviceAddressKHR_VkDeviceAddress_return =
                    vk->vkGetBufferDeviceAddressKHR(unboxed_device, pInfo);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkGetBufferDeviceAddressKHR_VkDeviceAddress_return,
                                sizeof(VkDeviceAddress));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetBufferDeviceAddressKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetBufferDeviceAddressKHR_VkDeviceAddress_return, device, pInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetBufferOpaqueCaptureAddressKHR: {
                android::base::beginTrace("vkGetBufferOpaqueCaptureAddressKHR decode");
                VkDevice device;
                const VkBufferDeviceAddressInfo* pInfo;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkBufferDeviceAddressInfo));
                reservedunmarshal_VkBufferDeviceAddressInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM, (VkBufferDeviceAddressInfo*)(pInfo),
                    readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkBufferDeviceAddressInfo(m_state,
                                                               (VkBufferDeviceAddressInfo*)(pInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetBufferOpaqueCaptureAddressKHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo);
                }
                uint64_t vkGetBufferOpaqueCaptureAddressKHR_uint64_t_return = (uint64_t)0;
                vkGetBufferOpaqueCaptureAddressKHR_uint64_t_return =
                    vk->vkGetBufferOpaqueCaptureAddressKHR(unboxed_device, pInfo);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkGetBufferOpaqueCaptureAddressKHR_uint64_t_return,
                                sizeof(uint64_t));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetBufferOpaqueCaptureAddressKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetBufferOpaqueCaptureAddressKHR_uint64_t_return, device, pInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceMemoryOpaqueCaptureAddressKHR: {
                android::base::beginTrace("vkGetDeviceMemoryOpaqueCaptureAddressKHR decode");
                VkDevice device;
                const VkDeviceMemoryOpaqueCaptureAddressInfo* pInfo;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo,
                                    sizeof(const VkDeviceMemoryOpaqueCaptureAddressInfo));
                reservedunmarshal_VkDeviceMemoryOpaqueCaptureAddressInfo(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDeviceMemoryOpaqueCaptureAddressInfo*)(pInfo), readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkDeviceMemoryOpaqueCaptureAddressInfo(
                        m_state, (VkDeviceMemoryOpaqueCaptureAddressInfo*)(pInfo));
                }
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkGetDeviceMemoryOpaqueCaptureAddressKHR 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)device, (unsigned long long)pInfo);
                }
                uint64_t vkGetDeviceMemoryOpaqueCaptureAddressKHR_uint64_t_return = (uint64_t)0;
                vkGetDeviceMemoryOpaqueCaptureAddressKHR_uint64_t_return =
                    vk->vkGetDeviceMemoryOpaqueCaptureAddressKHR(unboxed_device, pInfo);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkGetDeviceMemoryOpaqueCaptureAddressKHR_uint64_t_return,
                                sizeof(uint64_t));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceMemoryOpaqueCaptureAddressKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetDeviceMemoryOpaqueCaptureAddressKHR_uint64_t_return, device, pInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_pipeline_executable_properties
            case OP_vkGetPipelineExecutablePropertiesKHR: {
                android::base::beginTrace("vkGetPipelineExecutablePropertiesKHR decode");
                VkDevice device;
                const VkPipelineInfoKHR* pPipelineInfo;
                uint32_t* pExecutableCount;
                VkPipelineExecutablePropertiesKHR* pProperties;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pPipelineInfo, sizeof(const VkPipelineInfoKHR));
                reservedunmarshal_VkPipelineInfoKHR(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkPipelineInfoKHR*)(pPipelineInfo),
                                                    readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pExecutableCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pExecutableCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pExecutableCount);
                *readStreamPtrPtr += 8;
                if (pExecutableCount) {
                    vkReadStream->alloc((void**)&pExecutableCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pExecutableCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pProperties;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkPipelineExecutablePropertiesKHR**)&pProperties, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pProperties);
                *readStreamPtrPtr += 8;
                if (pProperties) {
                    vkReadStream->alloc(
                        (void**)&pProperties,
                        (*(pExecutableCount)) * sizeof(VkPipelineExecutablePropertiesKHR));
                    for (uint32_t i = 0; i < (uint32_t)(*(pExecutableCount)); ++i) {
                        reservedunmarshal_VkPipelineExecutablePropertiesKHR(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkPipelineExecutablePropertiesKHR*)(pProperties + i),
                            readStreamPtrPtr);
                    }
                }
                if (pPipelineInfo) {
                    transform_tohost_VkPipelineInfoKHR(m_state,
                                                       (VkPipelineInfoKHR*)(pPipelineInfo));
                }
                if (pExecutableCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pExecutableCount)); ++i) {
                            transform_tohost_VkPipelineExecutablePropertiesKHR(
                                m_state, (VkPipelineExecutablePropertiesKHR*)(pProperties + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPipelineExecutablePropertiesKHR 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pPipelineInfo,
                            (unsigned long long)pExecutableCount, (unsigned long long)pProperties);
                }
                VkResult vkGetPipelineExecutablePropertiesKHR_VkResult_return = (VkResult)0;
                vkGetPipelineExecutablePropertiesKHR_VkResult_return =
                    vk->vkGetPipelineExecutablePropertiesKHR(unboxed_device, pPipelineInfo,
                                                             pExecutableCount, pProperties);
                if ((vkGetPipelineExecutablePropertiesKHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkGetPipelineExecutablePropertiesKHR_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pExecutableCount;
                vkStream->putBe64(cgen_var_3);
                if (pExecutableCount) {
                    vkStream->write((uint32_t*)pExecutableCount, sizeof(uint32_t));
                }
                if (pExecutableCount) {
                    if (pProperties) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pExecutableCount)); ++i) {
                            transform_fromhost_VkPipelineExecutablePropertiesKHR(
                                m_state, (VkPipelineExecutablePropertiesKHR*)(pProperties + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pProperties;
                vkStream->putBe64(cgen_var_4);
                if (pProperties) {
                    if (pExecutableCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pExecutableCount)); ++i) {
                            marshal_VkPipelineExecutablePropertiesKHR(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkPipelineExecutablePropertiesKHR*)(pProperties + i));
                        }
                    }
                }
                vkStream->write(&vkGetPipelineExecutablePropertiesKHR_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPipelineExecutablePropertiesKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetPipelineExecutablePropertiesKHR_VkResult_return, device, pPipelineInfo,
                        pExecutableCount, pProperties);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPipelineExecutableStatisticsKHR: {
                android::base::beginTrace("vkGetPipelineExecutableStatisticsKHR decode");
                VkDevice device;
                const VkPipelineExecutableInfoKHR* pExecutableInfo;
                uint32_t* pStatisticCount;
                VkPipelineExecutableStatisticKHR* pStatistics;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pExecutableInfo,
                                    sizeof(const VkPipelineExecutableInfoKHR));
                reservedunmarshal_VkPipelineExecutableInfoKHR(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPipelineExecutableInfoKHR*)(pExecutableInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pStatisticCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pStatisticCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pStatisticCount);
                *readStreamPtrPtr += 8;
                if (pStatisticCount) {
                    vkReadStream->alloc((void**)&pStatisticCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pStatisticCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pStatistics;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkPipelineExecutableStatisticKHR**)&pStatistics, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pStatistics);
                *readStreamPtrPtr += 8;
                if (pStatistics) {
                    vkReadStream->alloc(
                        (void**)&pStatistics,
                        (*(pStatisticCount)) * sizeof(VkPipelineExecutableStatisticKHR));
                    for (uint32_t i = 0; i < (uint32_t)(*(pStatisticCount)); ++i) {
                        reservedunmarshal_VkPipelineExecutableStatisticKHR(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkPipelineExecutableStatisticKHR*)(pStatistics + i), readStreamPtrPtr);
                    }
                }
                if (pExecutableInfo) {
                    transform_tohost_VkPipelineExecutableInfoKHR(
                        m_state, (VkPipelineExecutableInfoKHR*)(pExecutableInfo));
                }
                if (pStatisticCount) {
                    if (pStatistics) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pStatisticCount)); ++i) {
                            transform_tohost_VkPipelineExecutableStatisticKHR(
                                m_state, (VkPipelineExecutableStatisticKHR*)(pStatistics + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPipelineExecutableStatisticsKHR 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)pExecutableInfo,
                            (unsigned long long)pStatisticCount, (unsigned long long)pStatistics);
                }
                VkResult vkGetPipelineExecutableStatisticsKHR_VkResult_return = (VkResult)0;
                vkGetPipelineExecutableStatisticsKHR_VkResult_return =
                    vk->vkGetPipelineExecutableStatisticsKHR(unboxed_device, pExecutableInfo,
                                                             pStatisticCount, pStatistics);
                if ((vkGetPipelineExecutableStatisticsKHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkGetPipelineExecutableStatisticsKHR_VkResult_return,
                                             opcode, context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pStatisticCount;
                vkStream->putBe64(cgen_var_3);
                if (pStatisticCount) {
                    vkStream->write((uint32_t*)pStatisticCount, sizeof(uint32_t));
                }
                if (pStatisticCount) {
                    if (pStatistics) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pStatisticCount)); ++i) {
                            transform_fromhost_VkPipelineExecutableStatisticKHR(
                                m_state, (VkPipelineExecutableStatisticKHR*)(pStatistics + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pStatistics;
                vkStream->putBe64(cgen_var_4);
                if (pStatistics) {
                    if (pStatisticCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pStatisticCount)); ++i) {
                            marshal_VkPipelineExecutableStatisticKHR(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkPipelineExecutableStatisticKHR*)(pStatistics + i));
                        }
                    }
                }
                vkStream->write(&vkGetPipelineExecutableStatisticsKHR_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPipelineExecutableStatisticsKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetPipelineExecutableStatisticsKHR_VkResult_return, device,
                        pExecutableInfo, pStatisticCount, pStatistics);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetPipelineExecutableInternalRepresentationsKHR: {
                android::base::beginTrace(
                    "vkGetPipelineExecutableInternalRepresentationsKHR decode");
                VkDevice device;
                const VkPipelineExecutableInfoKHR* pExecutableInfo;
                uint32_t* pInternalRepresentationCount;
                VkPipelineExecutableInternalRepresentationKHR* pInternalRepresentations;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pExecutableInfo,
                                    sizeof(const VkPipelineExecutableInfoKHR));
                reservedunmarshal_VkPipelineExecutableInfoKHR(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkPipelineExecutableInfoKHR*)(pExecutableInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pInternalRepresentationCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pInternalRepresentationCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pInternalRepresentationCount);
                *readStreamPtrPtr += 8;
                if (pInternalRepresentationCount) {
                    vkReadStream->alloc((void**)&pInternalRepresentationCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pInternalRepresentationCount, *readStreamPtrPtr,
                           sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pInternalRepresentations;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkPipelineExecutableInternalRepresentationKHR**)&pInternalRepresentations,
                       (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pInternalRepresentations);
                *readStreamPtrPtr += 8;
                if (pInternalRepresentations) {
                    vkReadStream->alloc((void**)&pInternalRepresentations,
                                        (*(pInternalRepresentationCount)) *
                                            sizeof(VkPipelineExecutableInternalRepresentationKHR));
                    for (uint32_t i = 0; i < (uint32_t)(*(pInternalRepresentationCount)); ++i) {
                        reservedunmarshal_VkPipelineExecutableInternalRepresentationKHR(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkPipelineExecutableInternalRepresentationKHR*)(pInternalRepresentations +
                                                                             i),
                            readStreamPtrPtr);
                    }
                }
                if (pExecutableInfo) {
                    transform_tohost_VkPipelineExecutableInfoKHR(
                        m_state, (VkPipelineExecutableInfoKHR*)(pExecutableInfo));
                }
                if (pInternalRepresentationCount) {
                    if (pInternalRepresentations) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pInternalRepresentationCount)); ++i) {
                            transform_tohost_VkPipelineExecutableInternalRepresentationKHR(
                                m_state,
                                (VkPipelineExecutableInternalRepresentationKHR*)(pInternalRepresentations +
                                                                                 i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetPipelineExecutableInternalRepresentationsKHR "
                            "0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device,
                            (unsigned long long)pExecutableInfo,
                            (unsigned long long)pInternalRepresentationCount,
                            (unsigned long long)pInternalRepresentations);
                }
                VkResult vkGetPipelineExecutableInternalRepresentationsKHR_VkResult_return =
                    (VkResult)0;
                vkGetPipelineExecutableInternalRepresentationsKHR_VkResult_return =
                    vk->vkGetPipelineExecutableInternalRepresentationsKHR(
                        unboxed_device, pExecutableInfo, pInternalRepresentationCount,
                        pInternalRepresentations);
                if ((vkGetPipelineExecutableInternalRepresentationsKHR_VkResult_return) ==
                    VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(
                    vkGetPipelineExecutableInternalRepresentationsKHR_VkResult_return, opcode,
                    context);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pInternalRepresentationCount;
                vkStream->putBe64(cgen_var_3);
                if (pInternalRepresentationCount) {
                    vkStream->write((uint32_t*)pInternalRepresentationCount, sizeof(uint32_t));
                }
                if (pInternalRepresentationCount) {
                    if (pInternalRepresentations) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pInternalRepresentationCount)); ++i) {
                            transform_fromhost_VkPipelineExecutableInternalRepresentationKHR(
                                m_state,
                                (VkPipelineExecutableInternalRepresentationKHR*)(pInternalRepresentations +
                                                                                 i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pInternalRepresentations;
                vkStream->putBe64(cgen_var_4);
                if (pInternalRepresentations) {
                    if (pInternalRepresentationCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pInternalRepresentationCount)); ++i) {
                            marshal_VkPipelineExecutableInternalRepresentationKHR(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkPipelineExecutableInternalRepresentationKHR*)(pInternalRepresentations +
                                                                                 i));
                        }
                    }
                }
                vkStream->write(&vkGetPipelineExecutableInternalRepresentationsKHR_VkResult_return,
                                sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetPipelineExecutableInternalRepresentationsKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkGetPipelineExecutableInternalRepresentationsKHR_VkResult_return, device,
                        pExecutableInfo, pInternalRepresentationCount, pInternalRepresentations);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_shader_integer_dot_product
#endif
#ifdef VK_KHR_shader_non_semantic_info
#endif
#ifdef VK_KHR_synchronization2
            case OP_vkCmdSetEvent2KHR: {
                android::base::beginTrace("vkCmdSetEvent2KHR decode");
                VkCommandBuffer commandBuffer;
                VkEvent event;
                const VkDependencyInfo* pDependencyInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkEvent*)&event = (VkEvent)unbox_VkEvent((VkEvent)(*&cgen_var_1));
                vkReadStream->alloc((void**)&pDependencyInfo, sizeof(const VkDependencyInfo));
                reservedunmarshal_VkDependencyInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkDependencyInfo*)(pDependencyInfo),
                                                   readStreamPtrPtr);
                if (pDependencyInfo) {
                    transform_tohost_VkDependencyInfo(m_state,
                                                      (VkDependencyInfo*)(pDependencyInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdSetEvent2KHR 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)event,
                            (unsigned long long)pDependencyInfo);
                }
                vk->vkCmdSetEvent2KHR(unboxed_commandBuffer, event, pDependencyInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdSetEvent2KHR(snapshotTraceBegin, snapshotTraceBytes,
                                                           &m_pool, commandBuffer, event,
                                                           pDependencyInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdResetEvent2KHR: {
                android::base::beginTrace("vkCmdResetEvent2KHR decode");
                VkCommandBuffer commandBuffer;
                VkEvent event;
                VkPipelineStageFlags2 stageMask;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkEvent*)&event = (VkEvent)unbox_VkEvent((VkEvent)(*&cgen_var_1));
                memcpy((VkPipelineStageFlags2*)&stageMask, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags2));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags2);
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdResetEvent2KHR 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)event,
                            (unsigned long long)stageMask);
                }
                vk->vkCmdResetEvent2KHR(unboxed_commandBuffer, event, stageMask);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdResetEvent2KHR(snapshotTraceBegin, snapshotTraceBytes,
                                                             &m_pool, commandBuffer, event,
                                                             stageMask);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdWaitEvents2KHR: {
                android::base::beginTrace("vkCmdWaitEvents2KHR decode");
                VkCommandBuffer commandBuffer;
                uint32_t eventCount;
                const VkEvent* pEvents;
                const VkDependencyInfo* pDependencyInfos;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((uint32_t*)&eventCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pEvents, ((eventCount)) * sizeof(const VkEvent));
                if (((eventCount))) {
                    uint8_t* cgen_var_1_ptr = (uint8_t*)(*readStreamPtrPtr);
                    *readStreamPtrPtr += 8 * ((eventCount));
                    for (uint32_t k = 0; k < ((eventCount)); ++k) {
                        uint64_t tmpval;
                        memcpy(&tmpval, cgen_var_1_ptr + k * 8, sizeof(uint64_t));
                        *(((VkEvent*)pEvents) + k) = (VkEvent)unbox_VkEvent((VkEvent)tmpval);
                    }
                }
                vkReadStream->alloc((void**)&pDependencyInfos,
                                    ((eventCount)) * sizeof(const VkDependencyInfo));
                for (uint32_t i = 0; i < (uint32_t)((eventCount)); ++i) {
                    reservedunmarshal_VkDependencyInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                       (VkDependencyInfo*)(pDependencyInfos + i),
                                                       readStreamPtrPtr);
                }
                if (pDependencyInfos) {
                    for (uint32_t i = 0; i < (uint32_t)((eventCount)); ++i) {
                        transform_tohost_VkDependencyInfo(
                            m_state, (VkDependencyInfo*)(pDependencyInfos + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdWaitEvents2KHR 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)eventCount, (unsigned long long)pEvents,
                            (unsigned long long)pDependencyInfos);
                }
                vk->vkCmdWaitEvents2KHR(unboxed_commandBuffer, eventCount, pEvents,
                                        pDependencyInfos);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdWaitEvents2KHR(snapshotTraceBegin, snapshotTraceBytes,
                                                             &m_pool, commandBuffer, eventCount,
                                                             pEvents, pDependencyInfos);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdPipelineBarrier2KHR: {
                android::base::beginTrace("vkCmdPipelineBarrier2KHR decode");
                VkCommandBuffer commandBuffer;
                const VkDependencyInfo* pDependencyInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pDependencyInfo, sizeof(const VkDependencyInfo));
                reservedunmarshal_VkDependencyInfo(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkDependencyInfo*)(pDependencyInfo),
                                                   readStreamPtrPtr);
                if (pDependencyInfo) {
                    transform_tohost_VkDependencyInfo(m_state,
                                                      (VkDependencyInfo*)(pDependencyInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdPipelineBarrier2KHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pDependencyInfo);
                }
                vk->vkCmdPipelineBarrier2KHR(unboxed_commandBuffer, pDependencyInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdPipelineBarrier2KHR(snapshotTraceBegin,
                                                                  snapshotTraceBytes, &m_pool,
                                                                  commandBuffer, pDependencyInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdWriteTimestamp2KHR: {
                android::base::beginTrace("vkCmdWriteTimestamp2KHR decode");
                VkCommandBuffer commandBuffer;
                VkPipelineStageFlags2 stage;
                VkQueryPool queryPool;
                uint32_t query;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkPipelineStageFlags2*)&stage, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags2));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags2);
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueryPool*)&queryPool =
                    (VkQueryPool)unbox_VkQueryPool((VkQueryPool)(*&cgen_var_1));
                memcpy((uint32_t*)&query, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(
                        stderr,
                        "stream %p: call vkCmdWriteTimestamp2KHR 0x%llx 0x%llx 0x%llx 0x%llx \n",
                        ioStream, (unsigned long long)commandBuffer, (unsigned long long)stage,
                        (unsigned long long)queryPool, (unsigned long long)query);
                }
                vk->vkCmdWriteTimestamp2KHR(unboxed_commandBuffer, stage, queryPool, query);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdWriteTimestamp2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, stage,
                        queryPool, query);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkQueueSubmit2KHR: {
                android::base::beginTrace("vkQueueSubmit2KHR decode");
                VkQueue queue;
                uint32_t submitCount;
                const VkSubmitInfo2* pSubmits;
                VkFence fence;
                // Begin non wrapped dispatchable handle unboxing for queue;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueue*)&queue = (VkQueue)(VkQueue)((VkQueue)(*&cgen_var_0));
                auto unboxed_queue = unbox_VkQueue(queue);
                auto vk = dispatch_VkQueue(queue);
                // End manual dispatchable handle unboxing for queue;
                memcpy((uint32_t*)&submitCount, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                vkReadStream->alloc((void**)&pSubmits,
                                    ((submitCount)) * sizeof(const VkSubmitInfo2));
                for (uint32_t i = 0; i < (uint32_t)((submitCount)); ++i) {
                    reservedunmarshal_VkSubmitInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkSubmitInfo2*)(pSubmits + i),
                                                    readStreamPtrPtr);
                }
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkFence*)&fence = (VkFence)unbox_VkFence((VkFence)(*&cgen_var_1));
                if (pSubmits) {
                    for (uint32_t i = 0; i < (uint32_t)((submitCount)); ++i) {
                        transform_tohost_VkSubmitInfo2(m_state, (VkSubmitInfo2*)(pSubmits + i));
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkQueueSubmit2KHR 0x%llx 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)queue, (unsigned long long)submitCount,
                            (unsigned long long)pSubmits, (unsigned long long)fence);
                }
                VkResult vkQueueSubmit2KHR_VkResult_return = (VkResult)0;
                vkQueueSubmit2KHR_VkResult_return =
                    vk->vkQueueSubmit2KHR(unboxed_queue, submitCount, pSubmits, fence);
                if ((vkQueueSubmit2KHR_VkResult_return) == VK_ERROR_DEVICE_LOST)
                    m_state->on_DeviceLost();
                m_state->on_CheckOutOfMemory(vkQueueSubmit2KHR_VkResult_return, opcode, context);
                vkStream->unsetHandleMapping();
                vkStream->write(&vkQueueSubmit2KHR_VkResult_return, sizeof(VkResult));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkQueueSubmit2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool,
                        vkQueueSubmit2KHR_VkResult_return, queue, submitCount, pSubmits, fence);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdWriteBufferMarker2AMD: {
                android::base::beginTrace("vkCmdWriteBufferMarker2AMD decode");
                VkCommandBuffer commandBuffer;
                VkPipelineStageFlags2 stage;
                VkBuffer dstBuffer;
                VkDeviceSize dstOffset;
                uint32_t marker;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                memcpy((VkPipelineStageFlags2*)&stage, *readStreamPtrPtr,
                       sizeof(VkPipelineStageFlags2));
                *readStreamPtrPtr += sizeof(VkPipelineStageFlags2);
                uint64_t cgen_var_1;
                memcpy((uint64_t*)&cgen_var_1, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkBuffer*)&dstBuffer = (VkBuffer)unbox_VkBuffer((VkBuffer)(*&cgen_var_1));
                memcpy((VkDeviceSize*)&dstOffset, *readStreamPtrPtr, sizeof(VkDeviceSize));
                *readStreamPtrPtr += sizeof(VkDeviceSize);
                memcpy((uint32_t*)&marker, *readStreamPtrPtr, sizeof(uint32_t));
                *readStreamPtrPtr += sizeof(uint32_t);
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkCmdWriteBufferMarker2AMD 0x%llx 0x%llx 0x%llx "
                            "0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer, (unsigned long long)stage,
                            (unsigned long long)dstBuffer, (unsigned long long)dstOffset,
                            (unsigned long long)marker);
                }
                vk->vkCmdWriteBufferMarker2AMD(unboxed_commandBuffer, stage, dstBuffer, dstOffset,
                                               marker);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdWriteBufferMarker2AMD(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer, stage,
                        dstBuffer, dstOffset, marker);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetQueueCheckpointData2NV: {
                android::base::beginTrace("vkGetQueueCheckpointData2NV decode");
                VkQueue queue;
                uint32_t* pCheckpointDataCount;
                VkCheckpointData2NV* pCheckpointData;
                // Begin non wrapped dispatchable handle unboxing for queue;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkQueue*)&queue = (VkQueue)(VkQueue)((VkQueue)(*&cgen_var_0));
                auto unboxed_queue = unbox_VkQueue(queue);
                auto vk = dispatch_VkQueue(queue);
                // End manual dispatchable handle unboxing for queue;
                // Begin manual dispatchable handle unboxing for pCheckpointDataCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pCheckpointDataCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pCheckpointDataCount);
                *readStreamPtrPtr += 8;
                if (pCheckpointDataCount) {
                    vkReadStream->alloc((void**)&pCheckpointDataCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pCheckpointDataCount, *readStreamPtrPtr, sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pCheckpointData;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkCheckpointData2NV**)&pCheckpointData, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pCheckpointData);
                *readStreamPtrPtr += 8;
                if (pCheckpointData) {
                    vkReadStream->alloc((void**)&pCheckpointData,
                                        (*(pCheckpointDataCount)) * sizeof(VkCheckpointData2NV));
                    for (uint32_t i = 0; i < (uint32_t)(*(pCheckpointDataCount)); ++i) {
                        reservedunmarshal_VkCheckpointData2NV(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkCheckpointData2NV*)(pCheckpointData + i), readStreamPtrPtr);
                    }
                }
                if (pCheckpointDataCount) {
                    if (pCheckpointData) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pCheckpointDataCount)); ++i) {
                            transform_tohost_VkCheckpointData2NV(
                                m_state, (VkCheckpointData2NV*)(pCheckpointData + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetQueueCheckpointData2NV 0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)queue,
                            (unsigned long long)pCheckpointDataCount,
                            (unsigned long long)pCheckpointData);
                }
                vk->vkGetQueueCheckpointData2NV(unboxed_queue, pCheckpointDataCount,
                                                pCheckpointData);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pCheckpointDataCount;
                vkStream->putBe64(cgen_var_3);
                if (pCheckpointDataCount) {
                    vkStream->write((uint32_t*)pCheckpointDataCount, sizeof(uint32_t));
                }
                if (pCheckpointDataCount) {
                    if (pCheckpointData) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pCheckpointDataCount)); ++i) {
                            transform_fromhost_VkCheckpointData2NV(
                                m_state, (VkCheckpointData2NV*)(pCheckpointData + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pCheckpointData;
                vkStream->putBe64(cgen_var_4);
                if (pCheckpointData) {
                    if (pCheckpointDataCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pCheckpointDataCount)); ++i) {
                            marshal_VkCheckpointData2NV(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkCheckpointData2NV*)(pCheckpointData + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetQueueCheckpointData2NV(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, queue,
                        pCheckpointDataCount, pCheckpointData);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_zero_initialize_workgroup_memory
#endif
#ifdef VK_KHR_copy_commands2
            case OP_vkCmdCopyBuffer2KHR: {
                android::base::beginTrace("vkCmdCopyBuffer2KHR decode");
                VkCommandBuffer commandBuffer;
                const VkCopyBufferInfo2* pCopyBufferInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pCopyBufferInfo, sizeof(const VkCopyBufferInfo2));
                reservedunmarshal_VkCopyBufferInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                    (VkCopyBufferInfo2*)(pCopyBufferInfo),
                                                    readStreamPtrPtr);
                if (pCopyBufferInfo) {
                    transform_tohost_VkCopyBufferInfo2(m_state,
                                                       (VkCopyBufferInfo2*)(pCopyBufferInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdCopyBuffer2KHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pCopyBufferInfo);
                }
                vk->vkCmdCopyBuffer2KHR(unboxed_commandBuffer, pCopyBufferInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyBuffer2KHR(snapshotTraceBegin, snapshotTraceBytes,
                                                             &m_pool, commandBuffer,
                                                             pCopyBufferInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyImage2KHR: {
                android::base::beginTrace("vkCmdCopyImage2KHR decode");
                VkCommandBuffer commandBuffer;
                const VkCopyImageInfo2* pCopyImageInfo;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCopyImageInfo, sizeof(const VkCopyImageInfo2));
                reservedunmarshal_VkCopyImageInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkCopyImageInfo2*)(pCopyImageInfo),
                                                   readStreamPtrPtr);
                if (pCopyImageInfo) {
                    transform_tohost_VkCopyImageInfo2(m_state, (VkCopyImageInfo2*)(pCopyImageInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdCopyImage2KHR 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer, (unsigned long long)pCopyImageInfo);
                }
                m_state->on_vkCmdCopyImage2KHR(&m_pool, commandBuffer, pCopyImageInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyImage2KHR(snapshotTraceBegin, snapshotTraceBytes,
                                                            &m_pool, commandBuffer, pCopyImageInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyBufferToImage2KHR: {
                android::base::beginTrace("vkCmdCopyBufferToImage2KHR decode");
                VkCommandBuffer commandBuffer;
                const VkCopyBufferToImageInfo2* pCopyBufferToImageInfo;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCopyBufferToImageInfo,
                                    sizeof(const VkCopyBufferToImageInfo2));
                reservedunmarshal_VkCopyBufferToImageInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkCopyBufferToImageInfo2*)(pCopyBufferToImageInfo), readStreamPtrPtr);
                if (pCopyBufferToImageInfo) {
                    transform_tohost_VkCopyBufferToImageInfo2(
                        m_state, (VkCopyBufferToImageInfo2*)(pCopyBufferToImageInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdCopyBufferToImage2KHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pCopyBufferToImageInfo);
                }
                m_state->on_vkCmdCopyBufferToImage2KHR(&m_pool, commandBuffer,
                                                       pCopyBufferToImageInfo, context);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyBufferToImage2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        pCopyBufferToImageInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdCopyImageToBuffer2KHR: {
                android::base::beginTrace("vkCmdCopyImageToBuffer2KHR decode");
                VkCommandBuffer commandBuffer;
                const VkCopyImageToBufferInfo2* pCopyImageToBufferInfo;
                // Begin global wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                vkReadStream->alloc((void**)&pCopyImageToBufferInfo,
                                    sizeof(const VkCopyImageToBufferInfo2));
                reservedunmarshal_VkCopyImageToBufferInfo2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkCopyImageToBufferInfo2*)(pCopyImageToBufferInfo), readStreamPtrPtr);
                if (pCopyImageToBufferInfo) {
                    transform_tohost_VkCopyImageToBufferInfo2(
                        m_state, (VkCopyImageToBufferInfo2*)(pCopyImageToBufferInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdCopyImageToBuffer2KHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pCopyImageToBufferInfo);
                }
                m_state->on_vkCmdCopyImageToBuffer2KHR(&m_pool, commandBuffer,
                                                       pCopyImageToBufferInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdCopyImageToBuffer2KHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, commandBuffer,
                        pCopyImageToBufferInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdBlitImage2KHR: {
                android::base::beginTrace("vkCmdBlitImage2KHR decode");
                VkCommandBuffer commandBuffer;
                const VkBlitImageInfo2* pBlitImageInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pBlitImageInfo, sizeof(const VkBlitImageInfo2));
                reservedunmarshal_VkBlitImageInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                   (VkBlitImageInfo2*)(pBlitImageInfo),
                                                   readStreamPtrPtr);
                if (pBlitImageInfo) {
                    transform_tohost_VkBlitImageInfo2(m_state, (VkBlitImageInfo2*)(pBlitImageInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdBlitImage2KHR 0x%llx 0x%llx \n", ioStream,
                            (unsigned long long)commandBuffer, (unsigned long long)pBlitImageInfo);
                }
                vk->vkCmdBlitImage2KHR(unboxed_commandBuffer, pBlitImageInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdBlitImage2KHR(snapshotTraceBegin, snapshotTraceBytes,
                                                            &m_pool, commandBuffer, pBlitImageInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkCmdResolveImage2KHR: {
                android::base::beginTrace("vkCmdResolveImage2KHR decode");
                VkCommandBuffer commandBuffer;
                const VkResolveImageInfo2* pResolveImageInfo;
                // Begin non wrapped dispatchable handle unboxing for commandBuffer;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkCommandBuffer*)&commandBuffer =
                    (VkCommandBuffer)(VkCommandBuffer)((VkCommandBuffer)(*&cgen_var_0));
                auto unboxed_commandBuffer = unbox_VkCommandBuffer(commandBuffer);
                auto vk = dispatch_VkCommandBuffer(commandBuffer);
                // End manual dispatchable handle unboxing for commandBuffer;
                vkReadStream->alloc((void**)&pResolveImageInfo, sizeof(const VkResolveImageInfo2));
                reservedunmarshal_VkResolveImageInfo2(vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                                      (VkResolveImageInfo2*)(pResolveImageInfo),
                                                      readStreamPtrPtr);
                if (pResolveImageInfo) {
                    transform_tohost_VkResolveImageInfo2(m_state,
                                                         (VkResolveImageInfo2*)(pResolveImageInfo));
                }
                if (m_logCalls) {
                    fprintf(stderr, "stream %p: call vkCmdResolveImage2KHR 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)commandBuffer,
                            (unsigned long long)pResolveImageInfo);
                }
                vk->vkCmdResolveImage2KHR(unboxed_commandBuffer, pResolveImageInfo);
                vkStream->unsetHandleMapping();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkCmdResolveImage2KHR(snapshotTraceBegin,
                                                               snapshotTraceBytes, &m_pool,
                                                               commandBuffer, pResolveImageInfo);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_KHR_format_feature_flags2
#endif
#ifdef VK_KHR_maintenance4
            case OP_vkGetDeviceBufferMemoryRequirementsKHR: {
                android::base::beginTrace("vkGetDeviceBufferMemoryRequirementsKHR decode");
                VkDevice device;
                const VkDeviceBufferMemoryRequirements* pInfo;
                VkMemoryRequirements2* pMemoryRequirements;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkDeviceBufferMemoryRequirements));
                reservedunmarshal_VkDeviceBufferMemoryRequirements(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDeviceBufferMemoryRequirements*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryRequirements, sizeof(VkMemoryRequirements2));
                reservedunmarshal_VkMemoryRequirements2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkMemoryRequirements2*)(pMemoryRequirements), readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkDeviceBufferMemoryRequirements(
                        m_state, (VkDeviceBufferMemoryRequirements*)(pInfo));
                }
                if (pMemoryRequirements) {
                    transform_tohost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetDeviceBufferMemoryRequirementsKHR 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                            (unsigned long long)pMemoryRequirements);
                }
                vk->vkGetDeviceBufferMemoryRequirementsKHR(unboxed_device, pInfo,
                                                           pMemoryRequirements);
                vkStream->unsetHandleMapping();
                if (pMemoryRequirements) {
                    transform_fromhost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                marshal_VkMemoryRequirements2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                              (VkMemoryRequirements2*)(pMemoryRequirements));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceBufferMemoryRequirementsKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceImageMemoryRequirementsKHR: {
                android::base::beginTrace("vkGetDeviceImageMemoryRequirementsKHR decode");
                VkDevice device;
                const VkDeviceImageMemoryRequirements* pInfo;
                VkMemoryRequirements2* pMemoryRequirements;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkDeviceImageMemoryRequirements));
                reservedunmarshal_VkDeviceImageMemoryRequirements(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDeviceImageMemoryRequirements*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                vkReadStream->alloc((void**)&pMemoryRequirements, sizeof(VkMemoryRequirements2));
                reservedunmarshal_VkMemoryRequirements2(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkMemoryRequirements2*)(pMemoryRequirements), readStreamPtrPtr);
                if (pInfo) {
                    transform_tohost_VkDeviceImageMemoryRequirements(
                        m_state, (VkDeviceImageMemoryRequirements*)(pInfo));
                }
                if (pMemoryRequirements) {
                    transform_tohost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetDeviceImageMemoryRequirementsKHR 0x%llx 0x%llx "
                            "0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                            (unsigned long long)pMemoryRequirements);
                }
                vk->vkGetDeviceImageMemoryRequirementsKHR(unboxed_device, pInfo,
                                                          pMemoryRequirements);
                vkStream->unsetHandleMapping();
                if (pMemoryRequirements) {
                    transform_fromhost_VkMemoryRequirements2(
                        m_state, (VkMemoryRequirements2*)(pMemoryRequirements));
                }
                marshal_VkMemoryRequirements2(vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                              (VkMemoryRequirements2*)(pMemoryRequirements));
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceImageMemoryRequirementsKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
            case OP_vkGetDeviceImageSparseMemoryRequirementsKHR: {
                android::base::beginTrace("vkGetDeviceImageSparseMemoryRequirementsKHR decode");
                VkDevice device;
                const VkDeviceImageMemoryRequirements* pInfo;
                uint32_t* pSparseMemoryRequirementCount;
                VkSparseImageMemoryRequirements2* pSparseMemoryRequirements;
                // Begin non wrapped dispatchable handle unboxing for device;
                uint64_t cgen_var_0;
                memcpy((uint64_t*)&cgen_var_0, *readStreamPtrPtr, 1 * 8);
                *readStreamPtrPtr += 1 * 8;
                *(VkDevice*)&device = (VkDevice)(VkDevice)((VkDevice)(*&cgen_var_0));
                auto unboxed_device = unbox_VkDevice(device);
                auto vk = dispatch_VkDevice(device);
                // End manual dispatchable handle unboxing for device;
                vkReadStream->alloc((void**)&pInfo, sizeof(const VkDeviceImageMemoryRequirements));
                reservedunmarshal_VkDeviceImageMemoryRequirements(
                    vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                    (VkDeviceImageMemoryRequirements*)(pInfo), readStreamPtrPtr);
                // Begin manual dispatchable handle unboxing for pSparseMemoryRequirementCount;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((uint32_t**)&pSparseMemoryRequirementCount, (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSparseMemoryRequirementCount);
                *readStreamPtrPtr += 8;
                if (pSparseMemoryRequirementCount) {
                    vkReadStream->alloc((void**)&pSparseMemoryRequirementCount, sizeof(uint32_t));
                    memcpy((uint32_t*)pSparseMemoryRequirementCount, *readStreamPtrPtr,
                           sizeof(uint32_t));
                    *readStreamPtrPtr += sizeof(uint32_t);
                }
                // Begin manual dispatchable handle unboxing for pSparseMemoryRequirements;
                vkReadStream->unsetHandleMapping();
                // WARNING PTR CHECK
                memcpy((VkSparseImageMemoryRequirements2**)&pSparseMemoryRequirements,
                       (*readStreamPtrPtr), 8);
                android::base::Stream::fromBe64((uint8_t*)&pSparseMemoryRequirements);
                *readStreamPtrPtr += 8;
                if (pSparseMemoryRequirements) {
                    vkReadStream->alloc((void**)&pSparseMemoryRequirements,
                                        (*(pSparseMemoryRequirementCount)) *
                                            sizeof(VkSparseImageMemoryRequirements2));
                    for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount)); ++i) {
                        reservedunmarshal_VkSparseImageMemoryRequirements2(
                            vkReadStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                            (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i),
                            readStreamPtrPtr);
                    }
                }
                if (pInfo) {
                    transform_tohost_VkDeviceImageMemoryRequirements(
                        m_state, (VkDeviceImageMemoryRequirements*)(pInfo));
                }
                if (pSparseMemoryRequirementCount) {
                    if (pSparseMemoryRequirements) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            transform_tohost_VkSparseImageMemoryRequirements2(
                                m_state,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                if (m_logCalls) {
                    fprintf(stderr,
                            "stream %p: call vkGetDeviceImageSparseMemoryRequirementsKHR 0x%llx "
                            "0x%llx 0x%llx 0x%llx \n",
                            ioStream, (unsigned long long)device, (unsigned long long)pInfo,
                            (unsigned long long)pSparseMemoryRequirementCount,
                            (unsigned long long)pSparseMemoryRequirements);
                }
                vk->vkGetDeviceImageSparseMemoryRequirementsKHR(unboxed_device, pInfo,
                                                                pSparseMemoryRequirementCount,
                                                                pSparseMemoryRequirements);
                vkStream->unsetHandleMapping();
                // WARNING PTR CHECK
                uint64_t cgen_var_3 = (uint64_t)(uintptr_t)pSparseMemoryRequirementCount;
                vkStream->putBe64(cgen_var_3);
                if (pSparseMemoryRequirementCount) {
                    vkStream->write((uint32_t*)pSparseMemoryRequirementCount, sizeof(uint32_t));
                }
                if (pSparseMemoryRequirementCount) {
                    if (pSparseMemoryRequirements) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            transform_fromhost_VkSparseImageMemoryRequirements2(
                                m_state,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                // WARNING PTR CHECK
                uint64_t cgen_var_4 = (uint64_t)(uintptr_t)pSparseMemoryRequirements;
                vkStream->putBe64(cgen_var_4);
                if (pSparseMemoryRequirements) {
                    if (pSparseMemoryRequirementCount) {
                        for (uint32_t i = 0; i < (uint32_t)(*(pSparseMemoryRequirementCount));
                             ++i) {
                            marshal_VkSparseImageMemoryRequirements2(
                                vkStream, VK_STRUCTURE_TYPE_MAX_ENUM,
                                (VkSparseImageMemoryRequirements2*)(pSparseMemoryRequirements + i));
                        }
                    }
                }
                vkStream->commitWrite();
                vkReadStream->setReadPos((uintptr_t)(*readStreamPtrPtr) -
                                         (uintptr_t)snapshotTraceBegin);
                size_t snapshotTraceBytes = vkReadStream->endTrace();
                if (m_state->snapshotsEnabled()) {
                    m_state->snapshot()->vkGetDeviceImageSparseMemoryRequirementsKHR(
                        snapshotTraceBegin, snapshotTraceBytes, &m_pool, device, pInfo,
                        pSparseMemoryRequirementCount, pSparseMemoryRequirements);
                }
                vkReadStream->clearPool();
                if (queueSubmitWithCommandsEnabled)
                    seqnoPtr->fetch_add(1, std::memory_order_seq_cst);
                android::base::endTrace();
                break;
            }
#endif
#ifdef VK_ANDROID_native_buffer
            case OP_vkGetSwapchainGrallocUsag