#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#

import subprocess
import re
import os
import sys
if sys.version_info.major == 2:
    from pipes import quote
else:
    from shlex import quote

from shlex import split

advixe_cl_path = 'advixe-cl'


class CommandRunner(object):
    def __init__(self, commands, stdout, stderr, dry_run):
        self.__commands = commands
        self.__stdout = stdout
        self.__stderr = stderr
        self.__dry_run = dry_run

    def __call(self, command):
        proc = subprocess.Popen(command, stdout=self.__stdout, stderr=self.__stderr)
        stddata = proc.communicate()
        return proc.returncode, stddata[0], stddata[1]

    def run(self):
        if len(self.__commands) == 0:
            return None

        results = []
        for command in self.__commands:
            if self.__dry_run:
                if type(command) is list:
                    double_dash_idx = command.index('--')
                    before_double_dash = command[:double_dash_idx]
                    after_double_dash = command[double_dash_idx:]

                    keyword = ''
                    result_command = ''
                    for arg in before_double_dash:
                        if arg.startswith('-'): # arg is a keyword
                            if keyword:
                                result_command += '{} '.format(keyword)
                            keyword = arg if arg not in ['-return-app-exitcode', '-auto-finalize'] else ''

                            if not keyword.startswith('--') and len(keyword) > 2:
                                keyword = '-{}'.format(keyword)
                        else:
                            if len(keyword) > 2: # Long keyword form
                                result_command += '{}={} '.format(keyword, quote(arg))
                            else: # Short keyword form
                                result_command += '{} {} '.format(keyword, quote(arg))
                            keyword = ''
                    if keyword: # Last arg was a keyword
                        result_command += '{} '.format(keyword)

                    # Leave commands after double dash untouched
                    result_command += ' '.join([quote(arg) for arg in after_double_dash])

                    print(result_command)
                else:
                    print(command)

                results.append([0, '', ''])
            else:
                res = self.__call(command)
                if res[0] != 0:
                    return res
                results.append(res)

        splitter = '\n'
        stdout_data = splitter.join([res[1] if res[1] is not None else '' for res in results])
        stderr_data = splitter.join([res[2] if res[2] is not None else '' for res in results])
        return results[-1][0], stdout_data, stderr_data


class CollectionByType(CommandRunner):
    def __init__(self, collection_type, app_path, collection_args, app_args, project_dir, stdout, stderr, dry_run):
        collection = collection_type.name.lower()
        self.__project_dir = project_dir
        self.__app_path = app_path
        self.__collection_args = [self.__create_option(arg) for arg in collection_args] if collection_args is not None else []
        self.__app_args = app_args if app_args is not None else []

        commands = []

        if 'roofline' in collection:
            tripcounts_args = ['-flop']

            commands.append(self.__create_command('survey'))
            commands.append(self.__create_command('tripcounts', tripcounts_args))
        else:
            commands.append(self.__create_command(collection))

        CommandRunner.__init__(self, commands, stdout, stderr, dry_run)

    def __create_command(self, collection, additonal_coll_args=[]):
        command = [advixe_cl_path, '-collect', collection, '-project-dir', self.__project_dir, '-return-app-exitcode']
        command += self.__collection_args
        command += additonal_coll_args
        command += ['--', self.__app_path]
        command += self.__app_args

        return command

    def __create_option(self, arg):
        res = re.match(r'\s*-{1,2}', arg)
        return '-{}'.format(arg) if res is None else arg


class CollectionByCLString(CommandRunner):
    def __init__(self, collect_command, stdout, stderr, dry_run):
        if sys.platform != 'win32':
            command = [advixe_cl_path, '-collect'] + split(collect_command)
        else:
            command = '"{}" -collect {}'.format(advixe_cl_path, collect_command)
        CommandRunner.__init__(self, [command], stdout, stderr, dry_run)


def collect(self, collection_type, app_path, collection_args=None, app_args=None, stdout=None, stderr=None, dry_run=False):
    """
    Run the project collection.

    Required args:
       collection_type: type of advisor collection (advisor.SURVEY|advisor.TRIPCOUNTS|advisor.MAP|advisor.DEPENDENCIES|advisor.ROOFLINE)
       app_path: absolute or relative path to binary file

    Optional args:
       collection_args: list of collection arguments, for example: ['delete-tripcounts', 'callstack-flops']. Default value is None.
       app_args: list of application arguments. Default value is None.
       stdout: stdout has arguments like subprocess.Popen.stdout: None|subprocess.PIPE|subprocess.STDOUT|open(path_to_log_file, 'w'). Default value is None.
       stderr: stderr has arguments like subprocess.Popen.stderr: None|subprocess.PIPE|subprocess.STDOUT|open(path_to_log_file, 'w'). Default value is None.

    Returned value:
    tuple: return_code, stdout_data, stderr_data

    Examples:
        >>> res = project.collect(advisor.SURVEY, '/home/main')
    """
    return CollectionByType(collection_type, app_path, collection_args, app_args, self._get_project_dir(), stdout, stderr, dry_run).run()


def collect_ex(self, collection_string, stdout=None, stderr=None, dry_run=False):
    """
    Run the project collection.

    Required args:
       collection_string: collection string for execution through the advixe-cl (without specifying advixe-cl and 'collect' option), for example: 'survey -project-dir home/new_test -- /home/main'

    Optional args:
       stdout: stdout has arguments like subprocess.Popen.stdout: None|subprocess.PIPE|subprocess.STDOUT|open(path_to_log_file, 'w'). Default value is None.
       stderr: stderr has arguments like subprocess.Popen.stderr: None|subprocess.PIPE|subprocess.STDOUT|open(path_to_log_file, 'w'). Default value is None.

    Returned value:
    tuple: return_code, stdout_data, stderr_data

    Examples:
        >>> res = project.collect('survey -project-dir home/new_test -- /home/main')
    """
    return CollectionByCLString(collection_string, stdout, stderr, dry_run).run()
