#include "clad/Differentiator/BaseForwardModeVisitor.h"
#include "clad/Differentiator/Compatibility.h"

#include "clang/AST/DeclarationName.h"
#include "clang/AST/OpenMPClause.h"
#include "clang/AST/Stmt.h"
#include "clang/AST/StmtOpenMP.h"
#include "clang/Basic/LLVM.h"
#include "clang/Basic/OpenMPKinds.h"
#include "clang/Sema/DeclSpec.h"
#include "clang/Sema/Sema.h"
#include "llvm/ADT/SmallVector.h"
#include "llvm/Frontend/OpenMP/OMP.h.inc"

#include <cassert>

using namespace clang;
using namespace llvm::omp;

namespace clad {

OMPClause*
BaseForwardModeVisitor::VisitOMPPrivateClause(const OMPPrivateClause* C) {
  llvm::SmallVector<Expr*, 16> Vars;
  Vars.reserve(C->varlist_size());
  for (const auto* Var : CLAD_COMPAT_CLANG20_getvarlist(C)) {
    Vars.push_back(Visit(Var).getExpr_dx());
    Vars.push_back(Clone(Var));
  }
  return CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).ActOnOpenMPPrivateClause(
      Vars, C->getBeginLoc(), C->getLParenLoc(), C->getEndLoc());
}

OMPClause* BaseForwardModeVisitor::VisitOMPFirstprivateClause(
    const OMPFirstprivateClause* C) {
  llvm::SmallVector<Expr*, 16> Vars;
  Vars.reserve(C->varlist_size());
  for (const auto* Var : CLAD_COMPAT_CLANG20_getvarlist(C)) {
    Vars.push_back(Visit(Var).getExpr_dx());
    Vars.push_back(Clone(Var));
  }
  return CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).ActOnOpenMPFirstprivateClause(
      Vars, C->getBeginLoc(), C->getLParenLoc(), C->getEndLoc());
}

OMPClause* BaseForwardModeVisitor::VisitOMPLastprivateClause(
    const OMPLastprivateClause* C) {
  llvm::SmallVector<Expr*, 16> Vars;
  Vars.reserve(C->varlist_size());
  for (const auto* Var : CLAD_COMPAT_CLANG20_getvarlist(C)) {
    Vars.push_back(Visit(Var).getExpr_dx());
    Vars.push_back(Clone(Var));
  }
  return CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).ActOnOpenMPLastprivateClause(
      Vars, C->getKind(), C->getKindLoc(), C->getColonLoc(), C->getBeginLoc(),
      C->getLParenLoc(), C->getEndLoc());
}

OMPClause*
BaseForwardModeVisitor::VisitOMPSharedClause(const OMPSharedClause* C) {
  llvm::SmallVector<Expr*, 16> Vars;
  Vars.reserve(C->varlist_size());
  for (const auto* Var : CLAD_COMPAT_CLANG20_getvarlist(C)) {
    Vars.push_back(Visit(Var).getExpr_dx());
    Vars.push_back(Clone(Var));
  }
  return CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).ActOnOpenMPSharedClause(
      Vars, C->getBeginLoc(), C->getLParenLoc(), C->getEndLoc());
}

OMPClause*
BaseForwardModeVisitor::VisitOMPReductionClause(const OMPReductionClause* C) {
  llvm::SmallVector<Expr*, 16> Vars;
  Vars.reserve(C->varlist_size());
  for (const auto* Var : CLAD_COMPAT_CLANG20_getvarlist(C)) {
    Vars.push_back(Visit(Var).getExpr_dx());
    Vars.push_back(Clone(Var));
  }
  CXXScopeSpec ReductionIdScopeSpec;
  ReductionIdScopeSpec.Adopt(C->getQualifierLoc());
  DeclarationNameInfo NameInfo = C->getNameInfo();
  return CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).ActOnOpenMPReductionClause(
      Vars, CLAD_COMPAT_CLANG21_getModifier(C), C->getBeginLoc(),
      C->getLParenLoc(), C->getModifierLoc(), C->getColonLoc(), C->getEndLoc(),
      ReductionIdScopeSpec, NameInfo);
}

StmtDiff BaseForwardModeVisitor::VisitOMPExecutableDirective(
    const OMPExecutableDirective* D) {
  // Transform the clauses
  llvm::SmallVector<OMPClause*, 16> TClauses;
  ArrayRef<OMPClause*> Clauses = D->clauses();
  TClauses.reserve(Clauses.size());
  for (auto* I : Clauses) {
    assert(I);
    CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).StartOpenMPClause(
        I->getClauseKind());
    OMPClause* Clause = Visit(I);
    CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).EndOpenMPClause();
    assert(Clause);
    TClauses.push_back(Clause);
  }
  StmtDiff AssociatedStmt;
  if (D->hasAssociatedStmt() && D->getAssociatedStmt()) {
    CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).ActOnOpenMPRegionStart(
        D->getDirectiveKind(), getCurrentScope());
    StmtDiff Body;
    {
      Sema::CompoundScopeRAII CompoundScope(m_Sema);
      const auto* CS = D->getInnermostCapturedStmt()->getCapturedStmt();
      Body = Visit(CS);
      if (isOpenMPLoopDirective(D->getDirectiveKind())) {
        // Ignore the differential variable generated by loop variables to
        // obtain a standard ForStmt.
        for (auto* SubStmt : Body.getStmt()->children()) {
          if (isa<ForStmt>(SubStmt)) {
            Body = SubStmt;
            break;
          }
        }
      }
    }
    AssociatedStmt = CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema)
                         .ActOnOpenMPRegionEnd(Body.getStmt(), TClauses)
                         .get();
  }
  DeclarationNameInfo DirName;
  OpenMPDirectiveKind CancelRegion = OMPD_unknown;
  return CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema)
      .ActOnOpenMPExecutableDirective(
          D->getDirectiveKind(), DirName, CancelRegion, TClauses,
          AssociatedStmt.getStmt(), D->getBeginLoc(), D->getEndLoc())
      .get();
}

StmtDiff BaseForwardModeVisitor::VisitOMPParallelDirective(
    const OMPParallelDirective* D) {
  DeclarationNameInfo DirName;
  CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).StartOpenMPDSABlock(
      OMPD_parallel, DirName, getCurrentScope(), D->getBeginLoc());
  StmtDiff Res = VisitOMPExecutableDirective(D);
  CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).EndOpenMPDSABlock(Res.getStmt());
  return Res;
}

StmtDiff BaseForwardModeVisitor::VisitOMPParallelForDirective(
    const OMPParallelForDirective* D) {
  DeclarationNameInfo DirName;
  CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).StartOpenMPDSABlock(
      OMPD_parallel_for, DirName, getCurrentScope(), D->getBeginLoc());
  StmtDiff Res = VisitOMPExecutableDirective(D);
  CLAD_COMPAT_CLANG19_SemaOpenMP(m_Sema).EndOpenMPDSABlock(Res.getStmt());
  return Res;
}
} // namespace clad