use super::GArticleID;
use glib::{Object, Properties, prelude::*};
use gtk4::subclass::prelude::*;
use news_flash::models::Enclosure;
use std::cell::{Cell, RefCell};

mod imp {
    pub(super) const INVALID_VALUE: i32 = -1;
    pub(super) const MIN_IMAGE_WIDTH: i32 = 200;
    pub(super) const MIN_IMAGE_HEIGHT: i32 = 150;

    use super::*;

    #[derive(Default, Properties)]
    #[properties(wrapper_type = super::GEnclosure)]
    pub struct GEnclosure {
        #[property(get, set, name = "article-id")]
        pub article_id: RefCell<GArticleID>,

        #[property(get, set)]
        pub url: RefCell<String>,

        #[property(name = "is-video", type = bool, get = Self::is_video)]
        #[property(name = "is-audio", type = bool, get = Self::is_audio)]
        #[property(name = "is-image", type = bool, get = Self::is_image)]
        #[property(name = "hide-preview", type = bool, get = Self::hide_preview)]
        #[property(get, set, nullable, name = "mime-type")]
        pub mime_type: RefCell<Option<String>>,

        #[property(get, set, nullable)]
        pub title: RefCell<Option<String>>,

        #[property(get, set, nullable)]
        pub summary: RefCell<Option<String>>,

        #[property(get, set, nullable, name = "article-title")]
        pub article_title: RefCell<Option<String>>,

        #[property(get, set, nullable, name = "feed-title")]
        pub feed_title: RefCell<Option<String>>,

        #[property(get, set, nullable)]
        pub author: RefCell<Option<String>>,

        #[property(get, set, nullable, name = "thumbnail-url")]
        pub thumbnail_url: RefCell<Option<String>>,

        #[property(get, set)]
        pub filesize: Cell<i32>,

        #[property(get, set)]
        pub position: Cell<f64>,

        #[property(get, set)]
        pub framerate: Cell<f64>,

        #[property(get, set)]
        pub width: Cell<i32>,

        #[property(get, set)]
        pub height: Cell<i32>,

        #[property(get, set)]
        pub duration: Cell<i32>,

        #[property(get, set)]
        pub is_default: Cell<bool>,

        #[property(get, set, nullable, name = "alternative-to")]
        pub alterative_to: RefCell<Option<String>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for GEnclosure {
        const NAME: &'static str = "NewsFlashGEnclosure";
        type Type = super::GEnclosure;
    }

    #[glib::derived_properties]
    impl ObjectImpl for GEnclosure {}

    impl GEnclosure {
        fn is_video(&self) -> bool {
            self.mime_type
                .borrow()
                .as_deref()
                .map(|mime| mime.starts_with("video") || mime == "application/x-shockwave-flash")
                .unwrap_or(false)
        }

        fn is_image(&self) -> bool {
            self.mime_type
                .borrow()
                .as_deref()
                .map(|mime| mime.starts_with("image"))
                .unwrap_or(false)
        }

        fn is_audio(&self) -> bool {
            self.mime_type
                .borrow()
                .as_deref()
                .map(|mime| mime.starts_with("audio"))
                .unwrap_or(false)
        }

        fn hide_preview(&self) -> bool {
            let width = self.width.get();
            let height = self.height.get();

            if self.is_image()
                && width > imp::INVALID_VALUE
                && width < imp::MIN_IMAGE_WIDTH
                && height > imp::INVALID_VALUE
                && height < imp::MIN_IMAGE_HEIGHT
            {
                return true;
            }

            false
        }
    }
}

glib::wrapper! {
    pub struct GEnclosure(ObjectSubclass<imp::GEnclosure>);
}

impl Default for GEnclosure {
    fn default() -> Self {
        Object::new()
    }
}

impl From<Enclosure> for GEnclosure {
    fn from(value: Enclosure) -> Self {
        GEnclosure::with_additional_data(value, None, None, None)
    }
}

impl From<GEnclosure> for Enclosure {
    fn from(value: GEnclosure) -> Self {
        Enclosure {
            article_id: value.article_id().into(),
            url: news_flash::models::Url::parse(&value.url()).unwrap(),
            mime_type: value.mime_type(),
            title: value.title(),
            summary: value.summary(),
            position: Some(value.position()),
            thumbnail_url: value.thumbnail_url(),
            filesize: if value.filesize() > 0 {
                Some(value.filesize())
            } else {
                None
            },
            duration: if value.duration() > 0 {
                Some(value.duration())
            } else {
                None
            },
            framerate: if value.framerate() > 0.0 {
                Some(value.framerate())
            } else {
                None
            },
            width: if value.width() > 0 { Some(value.width()) } else { None },
            height: if value.height() > 0 { Some(value.height()) } else { None },
            is_default: value.is_default(),
            alternative: value
                .alternative_to()
                .and_then(|url| news_flash::models::Url::parse(&url).ok()),
        }
    }
}

impl GEnclosure {
    pub fn with_additional_data(
        enclosure: Enclosure,
        article_title: Option<String>,
        author: Option<String>,
        feed_title: Option<String>,
    ) -> Self {
        let obj = GEnclosure::default();
        let imp = obj.imp();
        imp.article_id.replace(enclosure.article_id.into());
        imp.url.replace(enclosure.url.as_str().to_string());
        imp.mime_type.replace(enclosure.mime_type);
        imp.title.replace(enclosure.title);
        imp.title.replace(enclosure.summary);
        imp.position.set(enclosure.position.unwrap_or(0.0));
        imp.article_title.replace(article_title);
        imp.feed_title.replace(feed_title);
        imp.author.replace(author);
        imp.thumbnail_url.replace(enclosure.thumbnail_url);
        imp.filesize.set(enclosure.filesize.unwrap_or(imp::INVALID_VALUE));
        imp.width.set(enclosure.width.unwrap_or(imp::INVALID_VALUE));
        imp.height.set(enclosure.height.unwrap_or(imp::INVALID_VALUE));
        imp.duration.set(enclosure.duration.unwrap_or(imp::INVALID_VALUE));
        imp.framerate.set(enclosure.framerate.unwrap_or(-1.0));
        imp.is_default.set(enclosure.is_default);
        imp.alterative_to
            .replace(enclosure.alternative.map(|alt| alt.as_str().to_string()));
        obj
    }

    pub fn eq(&self, other: &Self) -> bool {
        let imp = self.imp();
        let other_imp = other.imp();
        *imp.article_id.borrow() == *other_imp.article_id.borrow() && *imp.url.borrow() == *other_imp.url.borrow()
    }
}
