use crate::app::App;
use crate::article_list::{MarkUpdate, ReadUpdate};
use crate::article_view::{ArticleLoader, ArticleView};
use crate::gobject_models::{GArticle, GArticleID, GMarked, GRead};
use crate::infrastructure::TokioRuntime;
use crate::main_window::MainWindow;
use crate::share::share_popover::SharePopover;
use crate::tag_popover::TagPopover;
use crate::util::constants;
use gio::{SimpleAction, SimpleActionGroup};
use glib::{Properties, SignalHandlerId, clone, subclass::prelude::*, subclass::*};
use gtk4::{
    Button, CompositeTemplate, GestureClick, MenuButton, NamedAction, Orientation, PopoverMenu, Shortcut, Stack,
    ToggleButton, prelude::*, subclass::prelude::*,
};
use news_flash::models::{ArticleID, PluginCapabilities};
use std::cell::{Cell, RefCell};

use super::ContentPage;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::ArticleViewColumn)]
    #[template(file = "data/resources/ui_templates/article_view/column.blp")]
    pub struct ArticleViewColumn {
        #[template_child]
        pub article_view: TemplateChild<ArticleView>,
        #[template_child]
        pub scrape_content_button: TemplateChild<ToggleButton>,
        #[template_child]
        pub share_button: TemplateChild<MenuButton>,
        #[template_child]
        pub scrape_content_stack: TemplateChild<Stack>,
        #[template_child]
        pub tag_button: TemplateChild<MenuButton>,
        #[template_child]
        pub tag_button_click: TemplateChild<GestureClick>,
        #[template_child]
        pub more_actions_button: TemplateChild<MenuButton>,
        #[template_child]
        pub more_actions_stack: TemplateChild<Stack>,
        #[template_child]
        pub tag_popover: TemplateChild<TagPopover>,
        #[template_child]
        pub share_popover: TemplateChild<SharePopover>,

        #[template_child]
        pub zoom_in_shortcut: TemplateChild<Shortcut>,
        #[template_child]
        pub zoom_out_shortcut: TemplateChild<Shortcut>,
        #[template_child]
        pub zoom_reset_shortcut: TemplateChild<Shortcut>,

        #[property(get, set, nullable)]
        pub article: RefCell<Option<GArticle>>,

        #[property(get, set, name = "have-article-and-online")]
        pub have_article_and_online: Cell<bool>,

        #[property(get, set, name = "can-tag")]
        pub can_tag: Cell<bool>,

        #[property(get, set, name = "block-next-view-switch")]
        pub block_next_view_switch: Cell<bool>,

        pub scrape_content_button_signal_id: RefCell<Option<SignalHandlerId>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ArticleViewColumn {
        const NAME: &'static str = "ArticleViewColumn";
        type ParentType = gtk4::Box;
        type Type = super::ArticleViewColumn;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for ArticleViewColumn {
        fn constructed(&self) {
            self.insert_zoom_controls();
            self.bind_article_and_online();
            self.bind_scrape_content_button();
            self.bind_more_actions_stack();
            self.bind_can_tag();

            let zoom_in_action = SimpleAction::new("zoom-in", None);
            zoom_in_action.connect_activate(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |_action, _parameter| {
                    let view = imp.article_view.get();
                    let zoom = view.zoom();
                    if zoom < constants::ARTICLE_ZOOM_UPPER {
                        view.set_zoom(zoom + 0.25);
                    }
                }
            ));

            let zoom_out_action = SimpleAction::new("zoom-out", None);
            zoom_out_action.connect_activate(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |_action, _parameter| {
                    let view = imp.article_view.get();
                    let zoom = view.zoom();
                    if zoom > constants::ARTICLE_ZOOM_LOWER {
                        view.set_zoom(zoom - 0.25);
                    }
                }
            ));

            let reset_zoom_action = SimpleAction::new("reset-zoom", None);
            reset_zoom_action.connect_activate(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |_action, _parameter| {
                    imp.article_view.set_zoom(1.0);
                }
            ));

            let obj = self.obj();
            let action_group = SimpleActionGroup::new();
            action_group.add_action(&zoom_in_action);
            action_group.add_action(&zoom_out_action);
            action_group.add_action(&reset_zoom_action);
            obj.insert_action_group("articleview", Some(&action_group));

            self.zoom_in_shortcut
                .set_action(Some(NamedAction::new("articleview.zoom-in")));
            self.zoom_out_shortcut
                .set_action(Some(NamedAction::new("articleview.zoom-out")));
            self.zoom_reset_shortcut
                .set_action(Some(NamedAction::new("articleview.reset-zoom")));
        }
    }

    impl WidgetImpl for ArticleViewColumn {}

    impl BoxImpl for ArticleViewColumn {}

    #[gtk4::template_callbacks]
    impl ArticleViewColumn {
        #[template_callback]
        fn have_article(&self, article: Option<GArticle>) -> bool {
            article.is_some()
        }

        #[template_callback]
        fn is_unread(&self, article: Option<GArticle>) -> bool {
            article.map(|a| a.read() == GRead::Unread).unwrap_or(false)
        }

        #[template_callback]
        fn is_marked(&self, article: Option<GArticle>) -> bool {
            article.map(|a| a.marked() == GMarked::Marked).unwrap_or(false)
        }

        #[template_callback]
        fn have_enclosures(&self, article: Option<GArticle>) -> bool {
            article.map(|a| !a.enclosures().is_empty()).unwrap_or(false)
        }

        #[template_callback]
        fn on_marked_clicked(&self) {
            let Some(article) = &*self.article.borrow() else {
                return;
            };

            let update = MarkUpdate {
                article_id: article.article_id(),
                marked: article.marked().invert(),
            };

            MainWindow::activate_action("set-article-marked", Some(&update.to_variant()));
        }

        #[template_callback]
        fn on_read_clicked(&self) {
            let Some(article) = &*self.article.borrow() else {
                return;
            };

            let update = ReadUpdate {
                article_id: article.article_id(),
                read: article.read().invert(),
            };

            MainWindow::activate_action("set-article-read", Some(&update.to_variant()));
        }

        #[template_callback]
        fn active_to_stack(&self, active: bool) -> &'static str {
            Self::bool_to_scrape_page(active)
        }

        fn bool_to_scrape_page(active: bool) -> &'static str {
            if active { "active" } else { "inactive" }
        }

        fn bool_to_actions_page(active: bool) -> &'static str {
            if active { "spinner" } else { "button" }
        }

        fn on_scrape_toggled(&self) {
            let active = self.scrape_content_button.is_active();

            if active {
                MainWindow::activate_action("scrape-content", None);
            } else {
                self.article_view.set_prefer_scraped_content(false);
                self.article_view.redraw_article();
            }
        }

        fn bind_scrape_content_button(&self) {
            App::default()
                .bind_property("is-scraping-content", &*self.scrape_content_stack, "visible-child-name")
                .transform_to(|_binding, active: bool| Some(Self::bool_to_scrape_page(active)))
                .build();

            let signal_id = self.scrape_content_button.connect_toggled(clone!(
                #[weak(rename_to = imp)]
                self,
                move |_button| imp.on_scrape_toggled()
            ));
            self.scrape_content_button_signal_id.replace(Some(signal_id));
        }

        fn bind_more_actions_stack(&self) {
            App::default()
                .bind_property("is-exporting-article", &*self.more_actions_stack, "visible-child-name")
                .transform_to(|_binding, active: bool| Some(Self::bool_to_actions_page(active)))
                .build();
        }

        fn bind_article_and_online(&self) {
            let obj = self.obj();

            obj.connect_article_notify(|obj| {
                let have_article = obj.article().is_some();
                let value = if App::default().is_offline() {
                    false
                } else {
                    have_article
                };
                obj.set_have_article_and_online(value);
            });

            App::default().connect_is_offline_notify(clone!(
                #[weak]
                obj,
                move |app| {
                    let have_article = obj.article().is_some();
                    let value = if app.is_offline() { false } else { have_article };
                    obj.set_have_article_and_online(value);
                }
            ));
        }

        fn bind_can_tag(&self) {
            let obj = self.obj();

            obj.connect_article_notify(|obj| {
                let have_article_and_online = obj.have_article_and_online();
                let support_tags = App::default()
                    .features()
                    .as_ref()
                    .contains(PluginCapabilities::SUPPORT_TAGS);
                obj.set_can_tag(have_article_and_online && support_tags);
            });

            App::default().connect_features_notify(clone!(
                #[weak]
                obj,
                move |app| {
                    let have_article_and_online = obj.have_article_and_online();
                    let support_tags = app.features().as_ref().contains(PluginCapabilities::SUPPORT_TAGS);
                    obj.set_can_tag(have_article_and_online && support_tags);
                }
            ));
        }

        fn insert_zoom_controls(&self) {
            let Some(popover) = self.more_actions_button.popover().and_downcast::<PopoverMenu>() else {
                return;
            };

            let zoom_plus = Button::builder()
                .icon_name("zoom-in-symbolic")
                .tooltip_text("Zoom In")
                .css_classes(vec!["circular", "flat"])
                .action_name("articleview.zoom-in")
                .build();
            let zoom_minus = Button::builder()
                .icon_name("zoom-out-symbolic")
                .tooltip_text("Zoom Out")
                .css_classes(vec!["circular", "flat"])
                .action_name("articleview.zoom-out")
                .build();
            let zoom_reset = Button::builder()
                .label("100%")
                .tooltip_text("Reset Zoom")
                .css_classes(vec!["flat"])
                .action_name("articleview.reset-zoom")
                .hexpand(true)
                .build();
            zoom_reset
                .bind_property("label", &*self.article_view, "zoom")
                .bidirectional()
                .transform_from(|_binding, b: f64| Some(format!("{:.0}%", b * 100.0)))
                .build();
            let zoom_box = gtk4::Box::builder()
                .spacing(12)
                .orientation(Orientation::Horizontal)
                .margin_start(18)
                .margin_end(18)
                .build();
            zoom_box.append(&zoom_minus);
            zoom_box.append(&zoom_reset);
            zoom_box.append(&zoom_plus);

            popover.add_child(&zoom_box, "zoom");
        }
    }
}

glib::wrapper! {
    pub struct ArticleViewColumn(ObjectSubclass<imp::ArticleViewColumn>)
        @extends gtk4::Widget, gtk4::Box;
}

impl Default for ArticleViewColumn {
    fn default() -> Self {
        glib::Object::new()
    }
}

impl ArticleViewColumn {
    pub fn instance() -> Self {
        ContentPage::instance().imp().articleview_column.get()
    }

    pub fn show_article(&self, article_id: GArticleID, overwrite_read: bool) {
        TokioRuntime::execute_with_callback(
            move || async move {
                let news_flash = App::news_flash();
                let news_flash_guard = news_flash.read().await;
                let news_flash = news_flash_guard.as_ref()?;
                ArticleLoader::load(&article_id, news_flash)
            },
            clone!(
                #[weak(rename_to = this)]
                self,
                move |res: Option<ArticleLoader>| {
                    let Some(loader) = res else {
                        return;
                    };

                    let imp = this.imp();
                    let new_article = loader.build();

                    if overwrite_read {
                        new_article.set_read(GRead::Read);
                    }

                    // block scrape content button toggled signal
                    if let Some(handler_id) = imp.scrape_content_button_signal_id.borrow().as_ref() {
                        imp.scrape_content_button.block_signal(handler_id);
                    }

                    this.set_article(Some(new_article));

                    // unblock scrape content button toggled signal
                    if let Some(handler_id) = imp.scrape_content_button_signal_id.borrow().as_ref() {
                        imp.scrape_content_button.unblock_signal(handler_id);
                    }

                    if !MainWindow::instance().is_fullscreen() {
                        let inner_splitview = ContentPage::instance().inner().clone();
                        let show_article_view = inner_splitview.is_collapsed()
                            && !inner_splitview.shows_content()
                            && !this.block_next_view_switch();
                        log::debug!("set show article view: {show_article_view}");
                        inner_splitview.set_show_content(show_article_view);
                    }

                    this.set_block_next_view_switch(false);
                },
            ),
        );
    }

    pub fn refresh_article_metadata_from_db(&self) {
        let Some(article) = self.article() else {
            return;
        };

        let article_id: ArticleID = article.article_id().into();

        TokioRuntime::execute_with_callback(
            move || async move {
                let news_flash = App::news_flash();
                let news_flash_guard = news_flash.read().await;
                let news_flash = news_flash_guard.as_ref()?;
                news_flash.get_article(&article_id).ok()
            },
            move |res| {
                let Some(loaded_article) = res else {
                    return;
                };

                let read: GRead = loaded_article.unread.into();
                let marked: GMarked = loaded_article.marked.into();

                article.set_read(read);
                article.set_marked(marked);
            },
        );
    }

    pub fn popup_tag_popover(&self) {
        self.imp().tag_button.popup();
    }
}
