/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */
#pragma once

#include "DrawViewWrapper.hxx"
#include <ObjectIdentifier.hxx>

class SdrObject;

namespace com::sun::star::drawing { class XShape; }

namespace chart
{

class Selection
{
public: //methods
    bool hasSelection() const;

    OUString const & getSelectedCID() const;
    css::uno::Reference< css::drawing::XShape > const & getSelectedAdditionalShape() const;
    const ObjectIdentifier& getSelectedOID() const { return m_aSelectedOID;}

    bool isResizeableObjectSelected() const;
    bool isRotateableObjectSelected( const rtl::Reference<::chart::ChartModel>& xChartModel ) const;
    bool isTitleObjectSelected() const;
    bool isDragableObjectSelected() const;

    bool isAdditionalShapeSelected() const;

    //returns true if selection has changed
    bool setSelection( const OUString& rCID );
    bool setSelection( const css::uno::Reference< css::drawing::XShape >& xShape );

    void clearSelection();

    //returns true if the selection has changed
    bool maybeSwitchSelectionAfterSingleClickWasEnsured();
    void resetPossibleSelectionAfterSingleClickWasEnsured();

    void remindSelectionBeforeMouseDown();
    bool isSelectionDifferentFromBeforeMouseDown() const;

    void adaptSelectionToNewPos( const Point& rMousePos, DrawViewWrapper const * pDrawViewWrapper
                                       , bool bIsRightMouse, bool bWaitingForDoubleClick );

    void applySelection( DrawViewWrapper* pDrawViewWrapper );

private: //member
    //the selection could be given by a CID or by a shape
    //if m_aSelectedObjectCID is not empty this indicates the selection
    //the content of m_xSelectedShape is ignored in that case
    //the strings are used for autogenerated chart specific objects
    //the shape reference is used for additional shapes
    ObjectIdentifier    m_aSelectedOID; //only single object selection so far
    ObjectIdentifier    m_aSelectedOID_beforeMouseDown;
    ObjectIdentifier    m_aSelectedOID_selectOnlyIfNoDoubleClickIsFollowing;
};

class SelectionHelper final : public MarkHandleProvider
{
public:
    static bool             findNamedParent( SdrObject*& pInOutObject
                                , OUString& rOutName
                                , bool bGivenObjectMayBeResult );
    static bool             findNamedParent( SdrObject*& pInOutObject
                                , ObjectIdentifier& rOutObject
                                , bool bGivenObjectMayBeResult );
    static SdrObject*       getMarkHandlesObject( SdrObject* pObj );
    static E3dScene*        getSceneToRotate( SdrObject* pObj );
    static bool             isDragableObjectHitTwice( const Point& rMPos
                                , const OUString& rNameOfSelectedObject
                                , const DrawViewWrapper& rDrawViewWrapper );

    static OUString getHitObjectCID(
        const Point& rMPos,
        DrawViewWrapper const & rDrawViewWrapper,
        bool bGetDiagramInsteadOf_Wall=false );

    static bool isRotateableObject( std::u16string_view rCID
                    , const rtl::Reference<::chart::ChartModel>& xChartModel );

    explicit SelectionHelper( SdrObject* pSelectedObj );
    ~SelectionHelper();

    //MarkHandleProvider:
    virtual bool getMarkHandles( SdrHdlList& rHdlList ) override;
    virtual bool getFrameDragSingles() override;

    SdrObject*   getObjectToMark();//sets also internally the mark object
    //-> getMarkHandles will behave different if this method has found a Mark Object different from m_pSelectedObj

private:
    SdrObject*      m_pSelectedObj;//hit and logically selected object
    SdrObject*      m_pMarkObj;//object that is marked instead to have more pretty handles
};

} //namespace chart

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
