#ifndef IGCC_SPIRV_EXTENSIONS_SUPPORT_H
#define IGCC_SPIRV_EXTENSIONS_SUPPORT_H

#include <vector>
#include <string>
#include "igfxfmid.h"

namespace IGC {
namespace SPIRVExtensionsSupport {

// Helper function for core family hierarchy checks
inline bool isCoreChildOf(PLATFORM Platform, GFXCORE_FAMILY Core) {
  return Platform.eRenderCoreFamily >= Core;
}

// Helper function for product family hierarchy checks
inline bool isProductChildOf(PLATFORM Platform, PRODUCT_FAMILY Product) {
  return Platform.eProductFamily >= Product;
}

// SPIR-V Extension and Capability structures
struct SPIRVCapability {
  std::string Name;
};

struct SPIRVExtension {
  std::string Name;
  std::string SpecURL;
  std::vector<SPIRVCapability> Capabilities;
  bool IsPublished;
};

// Forward declarations
inline bool isExtensionSupported(const std::string& ExtensionName, PLATFORM Platform);
inline bool isCapabilitySupported(const std::string& CapabilityName, PLATFORM Platform);

// Static vector of all defined extensions with their capabilities
static const std::vector<SPIRVExtension> AllExtensions = {
  {
    "SPV_INTEL_2d_block_io",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_2d_block_io.asciidoc",
    {
      {"Subgroup2DBlockIOINTEL"},
      {"Subgroup2DBlockTransformINTEL"},
      {"Subgroup2DBlockTransposeINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_bfloat16_conversion",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_bfloat16_conversion.asciidoc",
    {
      {"BFloat16ConversionINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_cache_controls",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_cache_controls.asciidoc",
    {
      {"CacheControlsINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_fp_fast_math_mode",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_fp_fast_math_mode.asciidoc",
    {
      {"FPFastMathModeINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_subgroups",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_subgroups.asciidoc",
    {
      {"SubgroupShuffleINTEL"},
      {"SubgroupBufferBlockIOINTEL"},
      {"SubgroupImageBlockIOINTEL"}
    },
    true
  }
};

// Individual extension/capability query functions
inline bool isExtensionSupported(const std::string& ExtensionName, PLATFORM Platform) {
  if (ExtensionName == "SPV_INTEL_2d_block_io") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_bfloat16_conversion") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_cache_controls") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_fp_fast_math_mode") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_subgroups") {
    return true /* Supported on all platforms */;
  }
  return false;
}

inline bool isCapabilitySupported(const std::string& CapabilityName, PLATFORM Platform) {
  if (CapabilityName == "Subgroup2DBlockIOINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "Subgroup2DBlockTransformINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "Subgroup2DBlockTransposeINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "BFloat16ConversionINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "CacheControlsINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "FPFastMathModeINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "SubgroupShuffleINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "SubgroupBufferBlockIOINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "SubgroupImageBlockIOINTEL") {
    return true /* Supported on all platforms */;
  }
  return false;
}

// Get extension info with capabilities for a platform
inline std::vector<SPIRVExtension> getSupportedExtensionInfo(PLATFORM Platform, bool includeUnpublished = false) {
  std::vector<SPIRVExtension> SupportedExtensions;
  for (const auto& Ext : AllExtensions) {
    if (!includeUnpublished && !Ext.IsPublished) {
      continue;
    }
    if (isExtensionSupported(Ext.Name, Platform)) {
      SPIRVExtension SupportedExt;
      SupportedExt.Name = Ext.Name;
      SupportedExt.SpecURL = Ext.SpecURL;
      SupportedExt.IsPublished = Ext.IsPublished;
      for (const auto& Cap : Ext.Capabilities) {
        if (isCapabilitySupported(Cap.Name, Platform)) {
          SupportedExt.Capabilities.push_back(Cap);
        }
      }
      SupportedExtensions.push_back(SupportedExt);
    }
  }
  return SupportedExtensions;
}
} // namespace SPIRVExtensionsSupport
} // namespace IGC

#endif // IGCC_SPIRV_EXTENSIONS_SUPPORT_H
