/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

use num_derive::FromPrimitive;

// Enumeration representing all crash annotations
#[allow(dead_code, non_camel_case_types, clippy::upper_case_acronyms)]
#[repr(u32)]
#[derive(Clone, Copy, Debug, FromPrimitive, PartialEq)]
pub(crate) enum CrashAnnotation {
    AbortMessage = 0,
    Accessibility = 1,
    AccessibilityClient = 2,
    AccessibilityInProcClient = 3,
    AdapterDeviceID = 4,
    AdapterDriverVendor = 5,
    AdapterDriverVersion = 6,
    AdapterSubsysID = 7,
    AdapterVendorID = 8,
    additional_minidumps = 9,
    Addons = 10,
    Android_Board = 11,
    Android_Brand = 12,
    Android_CPU_ABI = 13,
    Android_CPU_ABI2 = 14,
    Android_Device = 15,
    Android_Display = 16,
    Android_Fingerprint = 17,
    Android_Hardware = 18,
    Android_Manufacturer = 19,
    Android_Model = 20,
    Android_PackageName = 21,
    Android_ProcessName = 22,
    Android_Version = 23,
    AndroidComponentVersion = 24,
    AppInitDLLs = 25,
    ApplicationBuildID = 26,
    ApplicationServicesVersion = 27,
    AsyncShutdownTimeout = 28,
    AvailablePageFile = 29,
    AvailablePhysicalMemory = 30,
    AvailableSwapMemory = 31,
    AvailableVirtualMemory = 32,
    BackgroundTaskMode = 33,
    BackgroundTaskName = 34,
    BlockedDllList = 35,
    BlocklistInitFailed = 36,
    Breadcrumbs = 37,
    BuildID = 38,
    Comments = 39,
    ContentSandboxCapabilities = 40,
    ContentSandboxCapable = 41,
    ContentSandboxEnabled = 42,
    ContentSandboxLevel = 43,
    ContentSandboxWin32kState = 44,
    CPUMicrocodeVersion = 45,
    CrashTime = 46,
    CrashType = 47,
    CycleCollector = 48,
    DesktopEnvironment = 49,
    DeviceResetReason = 50,
    DistributionID = 51,
    DOMFissionEnabled = 52,
    DOMIPCEnabled = 53,
    DumperError = 54,
    EMCheckCompatibility = 55,
    EventLoopNestingLevel = 56,
    FontName = 57,
    GeckoViewVersion = 58,
    GleanVersion = 59,
    GMPLibraryPath = 60,
    GMPPlugin = 61,
    GPUProcessLaunchCount = 62,
    GPUProcessStatus = 63,
    GpuSandboxLevel = 64,
    GraphicsCompileShader = 65,
    GraphicsCriticalError = 66,
    GraphicsDrawShader = 67,
    GraphicsFontFile = 68,
    GraphicsNumActiveRenderers = 69,
    GraphicsNumRenderers = 70,
    GraphicsStartupTest = 71,
    Hang = 72,
    HasDeviceTouchScreen = 73,
    HeadlessMode = 74,
    InstallTime = 75,
    ipc_channel_error = 76,
    IpcCreatePipeCloExecErrno = 77,
    IpcCreatePipeFcntlErrno = 78,
    IpcCreatePipeSocketPairErrno = 79,
    IPCFatalErrorMsg = 80,
    IPCFatalErrorProtocol = 81,
    IPCMessageLargeBufferShmemFailureSize = 82,
    IPCMessageName = 83,
    IPCMessageSize = 84,
    IPCReadErrorReason = 85,
    IPCShutdownState = 86,
    IPCSystemError = 87,
    IsGarbageCollecting = 88,
    IsWayland = 89,
    IsWebRenderResourcePathOverridden = 90,
    JavaException = 91,
    JavaStackTrace = 92,
    JSActorMessage = 93,
    JSActorName = 94,
    JSLargeAllocationFailure = 95,
    JSModuleLoadError = 96,
    JSOutOfMemory = 97,
    LastInteractionDuration = 98,
    LastStartupWasCrash = 99,
    LauncherProcessState = 100,
    LinuxMemoryPSI = 101,
    LinuxUnderMemoryPressure = 102,
    LowPhysicalMemoryEvents = 103,
    MacAvailableMemorySysctl = 104,
    MacMemoryPressure = 105,
    MacMemoryPressureCriticalTime = 106,
    MacMemoryPressureNormalTime = 107,
    MacMemoryPressureSysctl = 108,
    MacMemoryPressureWarningTime = 109,
    MainThreadRunnableName = 110,
    Marionette = 111,
    MemtestOutput = 112,
    MinidumpSha256Hash = 113,
    ModuleSignatureInfo = 114,
    MozCrashReason = 115,
    NimbusEnrollments = 116,
    Notes = 117,
    OOMAllocationSize = 118,
    PHCAllocStack = 119,
    PHCBaseAddress = 120,
    PHCFreeStack = 121,
    PHCKind = 122,
    PHCUsableSize = 123,
    PluginFilename = 124,
    PluginName = 125,
    PluginVersion = 126,
    ProcessType = 127,
    ProductID = 128,
    ProductName = 129,
    ProfileDirectory = 130,
    ProfilerChildShutdownPhase = 131,
    PurgeablePhysicalMemory = 132,
    QuotaManagerShutdownTimeout = 133,
    QuotaManagerStorageIsNetworkResource = 134,
    RDDProcessStatus = 135,
    ReleaseChannel = 136,
    RemoteAgent = 137,
    RemoteType = 138,
    SafeMode = 139,
    SecondsSinceLastCrash = 140,
    ServerURL = 141,
    ShutdownProgress = 142,
    ShutdownReason = 143,
    StackTraces = 144,
    StartupCacheValid = 145,
    StartupCrash = 146,
    StartupTime = 147,
    StorageConnectionNotClosed = 148,
    SubmittedFrom = 149,
    SystemMemoryUsePercentage = 150,
    TelemetryClientId = 151,
    TelemetryEnvironment = 152,
    TelemetryProfileGroupId = 153,
    TelemetryServerURL = 154,
    TelemetrySessionId = 155,
    TestBoolean = 156,
    TestInteger = 157,
    TestKey = 158,
    TestUnicode = 159,
    TextureUsage = 160,
    Throttleable = 161,
    TotalPageFile = 162,
    TotalPhysicalMemory = 163,
    TotalVirtualMemory = 164,
    UnknownNetAddrSocketFamily = 165,
    UptimeTS = 166,
    URL = 167,
    URLSegments = 168,
    User32BeforeBlocklist = 169,
    useragent_locale = 170,
    UserFontRulesExhausted = 171,
    UtilityActorsName = 172,
    UtilityProcessStatus = 173,
    Vendor = 174,
    Version = 175,
    VRProcessStatus = 176,
    WasmLibrarySandboxMallocFailed = 177,
    WindowsErrorReporting = 178,
    WindowsFileDialogErrorCode = 179,
    WindowsPackageFamilyName = 180,
    Winsock_LSP = 181,
    XPCOMSpinEventLoopStack = 182,
    Count = 183,
}

impl std::fmt::Display for CrashAnnotation {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", CRASH_ANNOTATION_STRINGS[*self as usize])
    }
}

// Type of each annotation
#[allow(dead_code)]
#[derive(Clone, Copy, PartialEq)]
pub(crate) enum CrashAnnotationType {
  String = 0,   // Any type of string, const char*, nsCString, etc...
  Boolean = 1,  // Stored as a byte
  U32 = 2,      // C/C++'s uint32_t or Rust's u32
  U64 = 3,      // C/C++'s uint64_t or Rust's u64
  USize = 4,    // C/C++'s size_t or Rust's usize
  Object = 5,   // Not usable via the Rust API
}

// Type of each annotation
static CRASH_ANNOTATION_TYPES: &[CrashAnnotationType] = &[
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::U32,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U64,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::USize,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::USize,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U64,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Object,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U64,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::USize,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
];

// Stringified representation of each annotation. Most of these will just match
// the corresponding enum values, but for historical reasons some of them are
// different in string form when stored in the .extra file.
static CRASH_ANNOTATION_STRINGS: &[&str] = &[
    "AbortMessage",
    "Accessibility",
    "AccessibilityClient",
    "AccessibilityInProcClient",
    "AdapterDeviceID",
    "AdapterDriverVendor",
    "AdapterDriverVersion",
    "AdapterSubsysID",
    "AdapterVendorID",
    "additional_minidumps",
    "Add-ons",
    "Android_Board",
    "Android_Brand",
    "Android_CPU_ABI",
    "Android_CPU_ABI2",
    "Android_Device",
    "Android_Display",
    "Android_Fingerprint",
    "Android_Hardware",
    "Android_Manufacturer",
    "Android_Model",
    "Android_PackageName",
    "Android_ProcessName",
    "Android_Version",
    "AndroidComponentVersion",
    "AppInitDLLs",
    "ApplicationBuildID",
    "ApplicationServicesVersion",
    "AsyncShutdownTimeout",
    "AvailablePageFile",
    "AvailablePhysicalMemory",
    "AvailableSwapMemory",
    "AvailableVirtualMemory",
    "BackgroundTaskMode",
    "BackgroundTaskName",
    "BlockedDllList",
    "BlocklistInitFailed",
    "Breadcrumbs",
    "BuildID",
    "Comments",
    "ContentSandboxCapabilities",
    "ContentSandboxCapable",
    "ContentSandboxEnabled",
    "ContentSandboxLevel",
    "ContentSandboxWin32kState",
    "CPUMicrocodeVersion",
    "CrashTime",
    "CrashType",
    "CycleCollector",
    "DesktopEnvironment",
    "DeviceResetReason",
    "DistributionID",
    "DOMFissionEnabled",
    "DOMIPCEnabled",
    "DumperError",
    "EMCheckCompatibility",
    "EventLoopNestingLevel",
    "FontName",
    "GeckoViewVersion",
    "GleanVersion",
    "GMPLibraryPath",
    "GMPPlugin",
    "GPUProcessLaunchCount",
    "GPUProcessStatus",
    "GpuSandboxLevel",
    "GraphicsCompileShader",
    "GraphicsCriticalError",
    "GraphicsDrawShader",
    "GraphicsFontFile",
    "GraphicsNumActiveRenderers",
    "GraphicsNumRenderers",
    "GraphicsStartupTest",
    "Hang",
    "HasDeviceTouchScreen",
    "HeadlessMode",
    "InstallTime",
    "ipc_channel_error",
    "IpcCreatePipeCloExecErrno",
    "IpcCreatePipeFcntlErrno",
    "IpcCreatePipeSocketPairErrno",
    "IPCFatalErrorMsg",
    "IPCFatalErrorProtocol",
    "IPCMessageLargeBufferShmemFailureSize",
    "IPCMessageName",
    "IPCMessageSize",
    "IPCReadErrorReason",
    "IPCShutdownState",
    "IPCSystemError",
    "IsGarbageCollecting",
    "IsWayland",
    "IsWebRenderResourcePathOverridden",
    "JavaException",
    "JavaStackTrace",
    "JSActorMessage",
    "JSActorName",
    "JSLargeAllocationFailure",
    "JSModuleLoadError",
    "JSOutOfMemory",
    "LastInteractionDuration",
    "LastStartupWasCrash",
    "LauncherProcessState",
    "LinuxMemoryPSI",
    "LinuxUnderMemoryPressure",
    "LowPhysicalMemoryEvents",
    "MacAvailableMemorySysctl",
    "MacMemoryPressure",
    "MacMemoryPressureCriticalTime",
    "MacMemoryPressureNormalTime",
    "MacMemoryPressureSysctl",
    "MacMemoryPressureWarningTime",
    "MainThreadRunnableName",
    "Marionette",
    "MemtestOutput",
    "MinidumpSha256Hash",
    "ModuleSignatureInfo",
    "MozCrashReason",
    "NimbusEnrollments",
    "Notes",
    "OOMAllocationSize",
    "PHCAllocStack",
    "PHCBaseAddress",
    "PHCFreeStack",
    "PHCKind",
    "PHCUsableSize",
    "PluginFilename",
    "PluginName",
    "PluginVersion",
    "ProcessType",
    "ProductID",
    "ProductName",
    "ProfileDirectory",
    "ProfilerChildShutdownPhase",
    "PurgeablePhysicalMemory",
    "QuotaManagerShutdownTimeout",
    "QuotaManagerStorageIsNetworkResource",
    "RDDProcessStatus",
    "ReleaseChannel",
    "RemoteAgent",
    "RemoteType",
    "SafeMode",
    "SecondsSinceLastCrash",
    "ServerURL",
    "ShutdownProgress",
    "ShutdownReason",
    "StackTraces",
    "StartupCacheValid",
    "StartupCrash",
    "StartupTime",
    "StorageConnectionNotClosed",
    "SubmittedFrom",
    "SystemMemoryUsePercentage",
    "TelemetryClientId",
    "TelemetryEnvironment",
    "TelemetryProfileGroupId",
    "TelemetryServerURL",
    "TelemetrySessionId",
    "TestBoolean",
    "TestInteger",
    "TestKey",
    "TestUnicode",
    "TextureUsage",
    "Throttleable",
    "TotalPageFile",
    "TotalPhysicalMemory",
    "TotalVirtualMemory",
    "UnknownNetAddrSocketFamily",
    "UptimeTS",
    "URL",
    "URLSegments",
    "User32BeforeBlocklist",
    "useragent_locale",
    "UserFontRulesExhausted",
    "UtilityActorsName",
    "UtilityProcessStatus",
    "Vendor",
    "Version",
    "VRProcessStatus",
    "WasmLibrarySandboxMallocFailed",
    "WindowsErrorReporting",
    "WindowsFileDialogErrorCode",
    "WindowsPackageFamilyName",
    "Winsock_LSP",
    "XPCOMSpinEventLoopStack",
];

// Annotations which should be skipped when they have specific values
struct CrashAnnotationSkipValue {
  annotation: CrashAnnotation,
  value: &'static [u8],
}

static CRASH_ANNOTATIONS_SKIP_VALUES: &[CrashAnnotationSkipValue] = &[
    CrashAnnotationSkipValue { annotation: CrashAnnotation::BlocklistInitFailed, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::EventLoopNestingLevel, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::IsGarbageCollecting, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::LowPhysicalMemoryEvents, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::OOMAllocationSize, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::TextureUsage, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::User32BeforeBlocklist, value: b"0" },
];


/// Returns the type of a crash annotation.
///
/// # Arguments
///
/// * `annotation` - a crash annotation
pub(crate) fn type_of_annotation(annotation: CrashAnnotation) -> CrashAnnotationType {
  CRASH_ANNOTATION_TYPES[annotation as usize]
}

/// Checks if the annotation should be included. Some annotations are skipped
/// if their value matches a specific one (like the value 0).
///
/// # Arguments
///
/// * `annotation` - the crash annotation to be checked
/// * `value` - the contents of the annotation as a string
pub(crate) fn should_include_annotation(annotation: CrashAnnotation, value: &[u8]) -> bool {
    if let Some(skip_value) = CRASH_ANNOTATIONS_SKIP_VALUES
        .iter()
        .find(|&a| a.annotation == annotation)
    {
        skip_value.value != value
    } else {
        true
    }
}
